/**
 * @file IPairingCallback.h
 *
 * @par SW-Component
 * State machine for pairing
 *
 * @brief Callback interface definition for pairing state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the callback interface to be used by specific pairing state machine.
 *          Callback interface must be provided by generic pairing state machine.
 */

#ifndef _I_PAIRING_CALLBACK_H_
#define _I_PAIRING_CALLBACK_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;
class App2Bts_BaseMessage;
class PairingData;

/**
 * Interface definition.
 */
class IPairingCallback
{
public:
   /**
    * Destructor.
    */
   virtual ~IPairingCallback() {}

   /**
    * Result for start pairing.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: remote device address
    * @param[in] result: result
    */
   virtual void startPairingResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result) = 0;

   /**
    * Indicate link key.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] result: result
    * @param[in] linkKeyType: link key type
    * @param[in] linkKey: link key
    * @param[in] dLinkKey: d link key
    */
   virtual void indicateLinkKey(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result, IN const BTSLinkKeyType linkKeyType, IN const BTSLinkKey& linkKey, IN const BTSDLinkKey& dLinkKey) = 0;

   /**
    * Handle end of successful incoming pairing.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    */
   virtual void handleEndOfSuccessfulIncomingPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address) = 0;

   /**
    * Indicate pin code request.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    */
   virtual void indicatePinCodeRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address) = 0;

   /**
    * Indicate secure simple pairing request.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] sspMode: mode used for SSP
    * @param[in] numericValue: numeric value / passkey to be used for SSP (valid for modes: BTS_SSP_WITH_NUMERIC_COMPARISON and BTS_SSP_WITH_PASSKEY_ENTRY_MODE_INPUT_BY_REMOTE_DEVICE)
    */
   virtual void indicateSecureSimplePairingRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSSspMode sspMode, IN const BTSNumericValue& numericValue) = 0;

   /**
    * Get App2Bts message.
    *
    * @param[in] address: device address
    *
    * @return = App2Bts message if found else NULL
    */
   virtual App2Bts_BaseMessage* getApp2BtsWorkingMessage(IN const BTSBDAddress& address) = 0;

   /**
    * Get device entry.
    *
    * @param[in] address: device address
    *
    * @return = device entry
    */
   virtual PairingData& getEntry(IN const BTSBDAddress& address) = 0;
};

} //btstackif

#endif //_I_PAIRING_CALLBACK_H_
