/**
 * @file LocalBluetoothName.cpp
 *
 * @par SW-Component
 * State machine for setting local Bluetooth name
 *
 * @brief Implementation of generic setting local Bluetooth name state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic setting local Bluetooth name state machine.
 */

#include "LocalBluetoothName.h"
#include "ILocalBluetoothNameRequest.h"
#include "IBasicControl.h"
#include "ITimerPool.h"
// #include "Timer.h"
#include "ISwitchBluetooth.h"
#include "IConfiguration.h"
#include "FwErrmemPrint.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/LocalBluetoothName.cpp.trc.h"
#endif
#endif

namespace btstackif {

LocalBluetoothName::LocalBluetoothName() :
_requestIf(0),
_controlIf(0),
_timerPoolIf(0),
_switchBluetoothIf(0),
_configurationIf(0),
_configurationMaster(0),
_globalConfigActive(false),
_configRequestOngoing(false),
_configRequestDone(false),
_requestItem(),
_timerLocalName(),
_timeoutLocalNameUpdate(3000),
_btStackName(),
_newName()
{
}

LocalBluetoothName::~LocalBluetoothName()
{
   _requestIf = 0;
   _controlIf = 0;
   _timerPoolIf = 0;
   _switchBluetoothIf = 0;
   _configurationIf = 0;
   _configurationMaster = 0;
}

void LocalBluetoothName::reset(void)
{
   StateMachine::reset();
   // keep _requestIf
   // keep _controlIf
   // keep _timerPoolIf
   // keep _switchBluetoothIf
   // keep _configurationIf
   // keep _configurationMaster
   _globalConfigActive = false;
   _configRequestOngoing = false;
   _configRequestDone = false;
   _requestItem.reset();
   // stop all timer
   stopTimer(_timerLocalName);
   _btStackName.clear();
   _newName.clear();

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);
   _requestIf->reset();
}

void LocalBluetoothName::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_configurationIf);

   // check current state/action
   if(App2BtsOC_SetLocalBtName == _requestItem.item.opCode)
   {
      // request ongoing => update status and result
      createStatusMsg(bts2AppMsgList, 0, 0, true, _configurationIf->getConfiguration().name);
      createResultMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, BTS_REQ_FAILED);
   }

   // reset control data
   reset();
}

void LocalBluetoothName::setInstance(IN ILocalBluetoothNameRequest* instance)
{
   _requestIf = instance;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setCallback(this);
}

void LocalBluetoothName::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setControlIf(_controlIf);
}

void LocalBluetoothName::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_ERRMEM_ASSERT(0 != _timerPoolIf);
}

void LocalBluetoothName::setSwitchBluetoothIf(IN ISwitchBluetooth* switchBluetooth)
{
   _switchBluetoothIf = switchBluetooth;

   FW_ERRMEM_ASSERT(0 != _switchBluetoothIf);
}

void LocalBluetoothName::setConfigurationIf(IN IConfiguration* configurationIf)
{
   _configurationIf = configurationIf;

   FW_ERRMEM_ASSERT(0 != _configurationIf);
}

IConfigurationClient* LocalBluetoothName::getConfigurationClientHandler(void)
{
   return this;
}

IStateMachine* LocalBluetoothName::getSmEntryInterface(void)
{
   return this;
}

void LocalBluetoothName::sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetLocalBtName& request, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_configurationIf);

   if(false == isValidGetRequest(request))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   createStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, _configurationIf->getConfiguration().name);
}

void LocalBluetoothName::sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetLocalBtName& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_configurationIf);

   // collect data for sending status and result
   BTSRequestResult sendResult;
   const BTSRequestResult inputResult = (BTSRequestResult)resultCode;

   if(false == isValidSetRequest(request))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      sendResult = BTS_REQ_INVALID_PARAM;
   }
   else if(request.getDeviceName() == _configurationIf->getConfiguration().name)
   {
      sendResult = BTS_REQ_SUCCESS;
   }
   else
   {
      sendResult = BTS_REQ_FAILED;
   }

   // check given result
   if((BTS_REQ_LAST > inputResult) && (BTS_REQ_SUCCESS != inputResult) && (BTS_REQ_SUCCESS != sendResult))
   {
      // use given result
      sendResult = inputResult;
   }

   // update status
   createStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), sendStatusToAll, _configurationIf->getConfiguration().name);
   // update result
   createResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), sendResult);
}

bool LocalBluetoothName::isValidGetRequest(IN const App2Bts_GetLocalBtName& request) const
{
   (void)(request);
   return true;
}

bool LocalBluetoothName::isValidSetRequest(IN const App2Bts_SetLocalBtName& request) const
{
   return isValidName(request.getDeviceName());
}

bool LocalBluetoothName::setLocalBluetoothName(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetLocalBtName& request)
{
   ETG_TRACE_USR2((" setLocalBluetoothName"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_requestIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_configurationIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_switchBluetoothIf);

   if(false == isValidSetRequest(request))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      createStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, _configurationIf->getConfiguration().name);
      createResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_INVALID_PARAM);
      return false;
   }

   // check if BT is on
   if(true == _switchBluetoothIf->getAppBtMode())
   {
      // BT is on => continue
      // check current local BT name
      if(request.getDeviceName() == _configurationIf->getConfiguration().name)
      {
         // send answer directly because requested name is already set
         createStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, _configurationIf->getConfiguration().name);
         createResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_SUCCESS);

         return false;
      }
      else
      {
         // different local BT name requested
         // store data and process request
         request.getCompareItem(_requestItem.item);
         _requestItem.user = request.getUser();
         _requestItem.handle = request.getSessionHandle();
         _configurationIf->getConfiguration().name = request.getDeviceName();
         // set new name
         _newName = request.getDeviceName();
         // start timer
         startTimer(_timerLocalName, _timeoutLocalNameUpdate);

         _requestIf->setLocalName(bts2IpcMsgList, bts2AppMsgList, _configurationIf->getConfiguration().name);

         return true;
      }
   }
   else
   {
      // BT is off
      // mark configuration as not finished => forces re-writing record during next configuration sequence
      _configRequestDone = false;
      // store new name to configuration and answer directly
      _configurationIf->getConfiguration().name = request.getDeviceName();
      createStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, _configurationIf->getConfiguration().name);
      createResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_SUCCESS);

      return false;
   }
}

const BTSDeviceName& LocalBluetoothName::getBtStackName(void) const
{
   return _btStackName;
}

void LocalBluetoothName::updateLocalName(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDeviceName& name, IN const bool success)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_configurationIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_configurationMaster);

   // check if request is ongoing
   if(true == _configRequestOngoing)
   {
      // local name is only configured during stack configuration phase
      FW_ERRMEM_ASSERT(true == _globalConfigActive);

      // mark as not ongoing
      _configRequestOngoing = false;
      // mark as finished
      _configRequestDone = true;
      // handle updated local name
      handleUpdatedLocalName(*_configurationIf, name, success);
      // update current name
      createStatusMsg(bts2AppMsgList, 0, 0, true, _configurationIf->getConfiguration().name);

#ifdef ENABLE_NEW_CONFIGURATION_HANDLING
      // inform configuration master to execute next step
      (void)_configurationMaster->doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
#endif
   }
   else if(App2BtsOC_SetLocalBtName == _requestItem.item.opCode)
   {
      // handle updated local name
      handleUpdatedLocalName(*_configurationIf, name, success);
      // update current name and result
      createStatusMsg(bts2AppMsgList, 0, 0, true, _configurationIf->getConfiguration().name);
      createResultMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, ((true == success) ? BTS_REQ_SUCCESS : BTS_REQ_FAILED));

      // user request is ongoing => action is finished
      handleActionFinished(messageItem);

      // reset control data because action is finished
      _requestItem.reset();
   }
}

void LocalBluetoothName::setBtStackName(IN const BTSDeviceName& name)
{
   // do not check for empty name because initial value stored here is also empty
   _btStackName = name;
}

void LocalBluetoothName::handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId)
{
   (void)(messageItem);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   if(true == _timerLocalName.compare(timerId))
   {
      if((true == _configRequestOngoing) || (App2BtsOC_SetLocalBtName == _requestItem.item.opCode))
      {
         const BTSDeviceName name;
         _requestIf->sendVirtualLocalNameUpdate(bts2IpcMsgList, bts2AppMsgList, name, BTS_IPC_UPDATE_TIMEOUT);
      }
   }
}

void LocalBluetoothName::setConfigurationMasterIf(IN IConfigurationMaster* master)
{
   _configurationMaster = master;

   FW_ERRMEM_ASSERT(0 != _configurationMaster);
}

void LocalBluetoothName::startGlobalConfiguration(void)
{
   _globalConfigActive = true;
   _configRequestOngoing = false;
   // do not reset _configRequestDone because this configuration shall be done only once
}

void LocalBluetoothName::stopGlobalConfiguration(void)
{
   _globalConfigActive = false;
   _configRequestOngoing = false;
}

bool LocalBluetoothName::isSingleConfigurationOngoing(void) const
{
   if(false == _globalConfigActive)
   {
      return false;
   }

   // check if call to set local BT name is ongoing
   return _configRequestOngoing;
}

bool LocalBluetoothName::setSingleConfiguration(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool beforeBtOn, IN const bool errorOccurred)
{
   (void)(messageItem);
   (void)(beforeBtOn);

   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_requestIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_configurationIf);

   if(false == _globalConfigActive)
   {
      return false;
   }

   // ETG_TRACE_USR3((" LocalBluetoothName::setSingleConfiguration(): beforeBtOn=%d _configRequestOngoing=%d errorOccurred=%d _configRequestDone=%d", beforeBtOn, _configRequestOngoing, errorOccurred, _configRequestDone));

   if(true == _configRequestOngoing)
   {
      // configuration request is ongoing => wait for result
      return true;
   }

   if(true == errorOccurred)
   {
      // error occurred => do not continue
      return false;
   }

   if(false == _configRequestDone)
   {
      // configure local BT name

      if(false == isValidName(_configurationIf->getConfiguration().name))
      {
         FW_ERRMEM_ASSERT_ALWAYS();

         // check current BT stack local name
         if(true == isValidName(_btStackName))
         {
            // use initial device name from stack and continue
            _configurationIf->getConfiguration().name = _btStackName;
         }
         else
         {
            // if this happens then there is a bug we have to fix
            FW_ERRMEM_ASSERT_ALWAYS();
         }
      }

      if(true == isValidName(_configurationIf->getConfiguration().name))
      {
         // compare configuration value and BT stack local name
         if(_configurationIf->getConfiguration().name == _btStackName)
         {
            // we will send a virtual Ipc2Bts message to follow normal sequence
            _requestIf->sendVirtualLocalNameUpdate(bts2IpcMsgList, bts2AppMsgList, _configurationIf->getConfiguration().name, BTS_IPC_SUCCESS);
         }
         else
         {
            // send request
            _requestIf->setLocalName(bts2IpcMsgList, bts2AppMsgList, _configurationIf->getConfiguration().name);
         }

         // set new name
         _newName = _configurationIf->getConfiguration().name;
         // start timer
         startTimer(_timerLocalName, _timeoutLocalNameUpdate);
         // mark as ongoing
         _configRequestOngoing = true;

         return true;
      }
      else
      {
         // try during next configuration sequence again => keep _configRequestDone
         return false;
      }
   }

   return false;
}

bool LocalBluetoothName::isValidName(IN const BTSDeviceName& name) const
{
   // checking for empty name is enough
   return (false == name.empty());
}

void LocalBluetoothName::handleUpdatedLocalName(IN IConfiguration& configurationIf, IN const BTSDeviceName& name, IN const bool success)
{
   // stop timer
   stopTimer(_timerLocalName);
   // working name
   BTSDeviceName workingName(name);

   if(true == success)
   {
      // compare updated and new local name to be set
      if(name == _newName)
      {
         // working name already set
      }
      else
      {
         // take over update value
         if(true == isValidName(name))
         {
            // working name already set
         }
         else
         {
            FW_ERRMEM_ASSERT_ALWAYS();
            // continue with previous name
            workingName = _btStackName;
         }
      }
   }
   else
   {
      // continue with previous name
      workingName = _btStackName;
   }

   // set values
   configurationIf.getConfiguration().name = workingName;
   _btStackName = workingName;
   // reset new local name to be set
   _newName.clear();
}

void LocalBluetoothName::createStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSDeviceName& name) const
{
   Bts2App_CurrentLocalBtName* msg = ptrNew_Bts2App_CurrentLocalBtName();
   if(0 != msg)
   {
      if((true == sendStatusToAll) || (0 == user))
      {
         msg->setUser(0); // send status message to all
         msg->setSessionHandle(0); // send status message to all
      }
      else
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
      }
      msg->setDeviceName(name);

      bts2AppMsgList.push_back(msg);
   }
}

void LocalBluetoothName::createResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSRequestResult result) const
{
   if(0 != user)
   {
      Bts2App_SetLocalBtNameResult* msg = ptrNew_Bts2App_SetLocalBtNameResult();
      if(NULL != msg)
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
         msg->setRequestResult(result);

         bts2AppMsgList.push_back(msg);
      }
   }
}

void LocalBluetoothName::handleActionFinished(OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   messageItem.item = _requestItem.item;
   messageItem.item.opCode = App2BtsOC_None; // to force default handling for end of request sequence
   messageItem.deleteMessage = true;
   findApp2BtsWorkingMessage(messageItem);
}

void LocalBluetoothName::findApp2BtsWorkingMessage(OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   if(0 == messageItem.message)
   {
      messageItem.message = getApp2BtsWorkingMessage();
   }
   FW_ERRMEM_ASSERT(0 != messageItem.message);
}

App2Bts_BaseMessage* LocalBluetoothName::getApp2BtsWorkingMessage(void)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_controlIf);

   return _controlIf->findApp2BtsWorkingMessageWrapper(_requestItem.item);
}

void LocalBluetoothName::startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout)
{
   ETG_TRACE_USR3((" startTimer: timeout=%u", timeout));

   FW_ERRMEM_IF_NULL_PTR_RETURN(_timerPoolIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   timer.setTimerPool(_timerPoolIf);
   timer.start(_controlIf, this, timeout);
}

void LocalBluetoothName::stopTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" stopTimer"));

   timer.stop();

   // do not release timer
}

void LocalBluetoothName::releaseTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" releaseTimer"));

   timer.release();
}

bool LocalBluetoothName::isTimerActive(IN const ExtendedTimerEntry& timer) const
{
   return timer.isActive();
}

} //btstackif
