/**
 * @file LocalBluetoothName.h
 *
 * @par SW-Component
 * State machine for setting local Bluetooth name
 *
 * @brief Implementation of generic setting local Bluetooth name state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of generic setting local Bluetooth name state machine.
 */

#ifndef _LOCAL_BLUETOOTH_NAME_H_
#define _LOCAL_BLUETOOTH_NAME_H_

#include "StateMachine.h"
#include "ILocalBluetoothName.h"
#include "ILocalBluetoothNameCallback.h"
#include "IExtendedTimeoutHandler.h"
#include "ExtendedTimerEntry.h"
#include "IConfigurationClient.h"

namespace btstackif {

/**
 * LocalBluetoothName class.
 */
class LocalBluetoothName : public StateMachine, public ILocalBluetoothName, public ILocalBluetoothNameCallback, public IExtendedTimeoutHandler, public IConfigurationClient
{
public:
   /**
    * Default constructor.
    */
   LocalBluetoothName();

   /**
    * Destructor.
    */
   virtual ~LocalBluetoothName();

   virtual void reset(void);

   virtual void forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void setInstance(IN ILocalBluetoothNameRequest* instance);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setTimerPoolIf(IN ITimerPool* timerPool);

   virtual void setSwitchBluetoothIf(IN ISwitchBluetooth* switchBluetooth);

   virtual void setConfigurationIf(IN IConfiguration* configurationIf);

   virtual IConfigurationClient* getConfigurationClientHandler(void);

   virtual IStateMachine* getSmEntryInterface(void);

   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetLocalBtName& request, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetLocalBtName& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const;

   virtual bool isValidGetRequest(IN const App2Bts_GetLocalBtName& request) const;

   virtual bool isValidSetRequest(IN const App2Bts_SetLocalBtName& request) const;

   virtual bool setLocalBluetoothName(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetLocalBtName& request);

   virtual const BTSDeviceName& getBtStackName(void) const;

   virtual void updateLocalName(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDeviceName& name, IN const bool success);

   virtual void setBtStackName(IN const BTSDeviceName& name);

   virtual void handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId);

   virtual void setConfigurationMasterIf(IN IConfigurationMaster* master);

   virtual void startGlobalConfiguration(void);

   virtual void stopGlobalConfiguration(void);

   virtual bool isSingleConfigurationOngoing(void) const;

   virtual bool setSingleConfiguration(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool beforeBtOn, IN const bool errorOccurred);

private:
   ILocalBluetoothNameRequest* _requestIf; /**< request interface */
   IBasicControl* _controlIf; /**< basic control interface */
   ITimerPool* _timerPoolIf; /**< timer pool interface */
   ISwitchBluetooth* _switchBluetoothIf; /**< switch Bluetooth interface */
   IConfiguration* _configurationIf; /**< configuration interface */
   IConfigurationMaster* _configurationMaster; /**< configuration master interface */
   bool _globalConfigActive; /**< flag for global configuration */
   bool _configRequestOngoing; /**< flag for ongoing config request */
   bool _configRequestDone; /**< flag for finished config request */
   BTSApp2BtsMessageRequestItem _requestItem; /**< request item */
   ExtendedTimerEntry _timerLocalName; /**< timer */
   const BTSTimeValue _timeoutLocalNameUpdate; /**< timeout for update */
   BTSDeviceName _btStackName; /**< BT stack local name */
   BTSDeviceName _newName; /**< new local name to be set */

   bool isValidName(IN const BTSDeviceName& name) const;

   void handleUpdatedLocalName(IN IConfiguration& configurationIf, IN const BTSDeviceName& name, IN const bool success);

   void createStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSDeviceName& name) const;

   void createResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSRequestResult result) const;

   void handleActionFinished(OUT BTSHandleIpc2BtsMessageItem& messageItem);

   void findApp2BtsWorkingMessage(OUT BTSHandleIpc2BtsMessageItem& messageItem);

   App2Bts_BaseMessage* getApp2BtsWorkingMessage(void);

   void startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout);

   void stopTimer(IN ExtendedTimerEntry& timer) const;

   void releaseTimer(IN ExtendedTimerEntry& timer) const;

   bool isTimerActive(IN const ExtendedTimerEntry& timer) const;
};

} //btstackif

#endif //_LOCAL_BLUETOOTH_NAME_H_
