/**
 * @file ILocalBluetoothNameRequest.h
 *
 * @par SW-Component
 * State machine for setting local Bluetooth name
 *
 * @brief Request interface definition for setting local Bluetooth name state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the request interface to be implemented by specific setting local Bluetooth name state machine.
 */

#ifndef _I_LOCAL_BLUETOOTH_NAME_REQUEST_H_
#define _I_LOCAL_BLUETOOTH_NAME_REQUEST_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class ILocalBluetoothNameCallback;
class IBasicControl;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class ILocalBluetoothNameRequest
{
public:
   /**
    * Destructor.
    */
   virtual ~ILocalBluetoothNameRequest() {}

   /**
    * Reset internal data.
    */
   virtual void reset(void) = 0;

   /**
    * Set callback instance (implementation).
    *
    * @param[in] callback: callback instance (implementation)
    */
   virtual void setCallback(IN ILocalBluetoothNameCallback* callback) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set local name.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] name: local name
    */
   virtual void setLocalName(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSDeviceName& name) = 0;

   /**
    * Send virtual update message for local name (loopback).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] name: local name
    * @param[in] errorCode: error code
    */
   virtual void sendVirtualLocalNameUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSDeviceName& name, IN const BTSIpcCommonErrorCode errorCode) = 0;
};

} //btstackif

#endif //_I_LOCAL_BLUETOOTH_NAME_REQUEST_H_
