/**
 * @file LocalAdapterModes.cpp
 *
 * @par SW-Component
 * State machine for local adapter modes
 *
 * @brief Implementation of generic local adapter modes state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic local adapter modes state machine.
 */

#include "LocalAdapterModes.h"
#include "ILocalAdapterModesRequest.h"
#include "IBasicControl.h"
#include "ITimerPool.h"
// #include "Timer.h"
#include "ISwitchBluetooth.h"
#include "IConfiguration.h"
#include "FwAssert.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/LocalAdapterModes.cpp.trc.h"
#endif
#endif

namespace btstackif {

LocalAdapterModes::LocalAdapterModes() :
_requestIf(0),
_controlIf(0),
_timerPoolIf(0),
_switchBluetoothIf(0),
_configurationIf(0),
_configurationMaster(0),
_requestItem(),
_newDiscoverable(BTS_MODE_DISABLED),
_newDiscoverableTimeout(0),
_newConnectable(BTS_MODE_DISABLED),
_newConnectableTimeout(0),
_requestedDiscoverable(BTS_MODE_DISABLED),
_requestedDiscoverableTimeout(0),
_requestedConnectable(BTS_MODE_DISABLED),
_requestedConnectableTimeout(0),
_currentDiscoverable(BTS_MODE_DISABLED),
_currentDiscoverableOffReason(BTS_MODE_OFF_REASON_INTERNAL),
_currentDiscoverableTimeout(0),
_currentConnectable(BTS_MODE_DISABLED),
_currentConnectableOffReason(BTS_MODE_OFF_REASON_INTERNAL),
_currentConnectableTimeout(0),
_previousDiscoverableOffReason(BTS_MODE_OFF_REASON_INTERNAL),
_previousConnectableOffReason(BTS_MODE_OFF_REASON_INTERNAL),
_internalDiscoverable(BTS_MODE_DISABLED),
_internalDiscoverableTimeout(0),
_internalConnectable(BTS_MODE_DISABLED),
_internalConnectableTimeout(0),
_timerDiscoverableMode(),
_timerConnectableMode(),
_timerProperty(),
_timeoutPropertyUpdate(2000),
_maxNmbGet(1),
_stackDiscoverable(false),
_stackPairable(false),
_stackConnectable(false),
_stackDiscoverableTimeout(0),
_stackPairableTimeout(0),
_timeoutMarker(1),
_forcedMarker(2),
_defaultPairableMode(true),
_updateDiscoverableMode(false),
_globalConfigActive(false),
_configRequestDone(false),
_settingLocalAdapterModesActive(false),
_testTriggerBlockSetDiscoverable(false)
{
   for(size_t i = 0; i < LAST; i++)
   {
      _setActive[i] = false;
      _getActive[i] = false;
      _counterGet[i] = 0;
      _forcedSet[i] = false;
   }
}

LocalAdapterModes::~LocalAdapterModes()
{
   _requestIf = 0;
   _controlIf = 0;
   _timerPoolIf = 0;
   _switchBluetoothIf = 0;
   _configurationIf = 0;
   _configurationMaster = 0;
}

void LocalAdapterModes::reset(void)
{
   StateMachine::reset();
   // keep _requestIf
   // keep _controlIf
   // keep _timerPoolIf
   // keep _switchBluetoothIf
   // keep _configurationIf
   // keep _configurationMaster
   _requestItem.reset();
   _newDiscoverable = BTS_MODE_DISABLED;
   _newDiscoverableTimeout = 0;
   _newConnectable = BTS_MODE_DISABLED;
   _newConnectableTimeout = 0;
   _requestedDiscoverable = BTS_MODE_DISABLED;
   _requestedDiscoverableTimeout = 0;
   _requestedConnectable = BTS_MODE_DISABLED;
   _requestedConnectableTimeout = 0;
   _currentDiscoverable = BTS_MODE_DISABLED;
   _currentDiscoverableOffReason = BTS_MODE_OFF_REASON_INTERNAL;
   _currentDiscoverableTimeout = 0;
   _currentConnectable = BTS_MODE_DISABLED;
   _currentConnectableOffReason = BTS_MODE_OFF_REASON_INTERNAL;
   _currentConnectableTimeout = 0;
   _previousDiscoverableOffReason = BTS_MODE_OFF_REASON_INTERNAL;
   _previousConnectableOffReason = BTS_MODE_OFF_REASON_INTERNAL;
   _internalDiscoverable = BTS_MODE_DISABLED;
   _internalDiscoverableTimeout = 0;
   _internalConnectable = BTS_MODE_DISABLED;
   _internalConnectableTimeout = 0;
   for(size_t i = 0; i < LAST; i++)
   {
      _setActive[i] = false;
      _getActive[i] = false;
      _counterGet[i] = 0;
      _forcedSet[i] = false;
   }
   _stackDiscoverable = false;
   _stackPairable = false;
   _stackConnectable = false;
   _stackDiscoverableTimeout = 0;
   _stackPairableTimeout = 0;
   // stop all timer
   stopTimer(_timerDiscoverableMode);
   stopTimer(_timerConnectableMode);
   stopTimer(_timerProperty);

   _globalConfigActive = false;
   _configRequestDone = false;
   _settingLocalAdapterModesActive = false;

   _testTriggerBlockSetDiscoverable = false;

   FW_IF_NULL_PTR_RETURN(_requestIf);
   _requestIf->reset();

   // Bluetooth is switched off during restart => reset start values
   FW_IF_NULL_PTR_RETURN(_configurationIf);
   setInitialModes(_configurationIf->getConfiguration().initialDiscoverableMode, _configurationIf->getConfiguration().initialConnectableMode);
}

void LocalAdapterModes::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   // check current state/action
   if(App2BtsOC_SetLocalAdapterModes == _requestItem.item.opCode)
   {
      // request ongoing => update status and result
      createDiscoverableStatusMsg(bts2AppMsgList, 0, 0, true, BTS_MODE_DISABLED, BTS_MODE_OFF_REASON_INTERNAL);
      createConnectableStatusMsg(bts2AppMsgList, 0, 0, true, BTS_MODE_DISABLED, BTS_MODE_OFF_REASON_INTERNAL);
      createResultMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, (((BTS_MODE_DISABLED == _internalDiscoverable) && (BTS_MODE_DISABLED == _internalConnectable)) ? BTS_REQ_SUCCESS : BTS_REQ_FAILED));
   }
   else
   {
      // no request ongoing
      if(BTS_MODE_ENABLED == _currentDiscoverable)
      {
         // discoverable mode enabled => update status
         createDiscoverableStatusMsg(bts2AppMsgList, 0, 0, true, BTS_MODE_DISABLED, BTS_MODE_OFF_REASON_INTERNAL);
      }

      if(BTS_MODE_ENABLED == _currentConnectable)
      {
         // connectable mode enabled => update status
         createConnectableStatusMsg(bts2AppMsgList, 0, 0, true, BTS_MODE_DISABLED, BTS_MODE_OFF_REASON_INTERNAL);
      }
   }

   // reset control data
   reset();
}

void LocalAdapterModes::setInstance(IN ILocalAdapterModesRequest* instance)
{
   _requestIf = instance;

   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setCallback(this);
}

void LocalAdapterModes::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_NORMAL_ASSERT(0 != _controlIf);

   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setControlIf(_controlIf);
}

void LocalAdapterModes::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_NORMAL_ASSERT(0 != _timerPoolIf);
}

void LocalAdapterModes::setSwitchBluetoothIf(IN ISwitchBluetooth* switchBluetooth)
{
   _switchBluetoothIf = switchBluetooth;

   FW_NORMAL_ASSERT(0 != _switchBluetoothIf);
}

void LocalAdapterModes::setConfigurationIf(IN IConfiguration* configuration)
{
   _configurationIf = configuration;

   FW_IF_NULL_PTR_RETURN(_configurationIf);

   // Bluetooth is switched off during startup => reset start values
   setInitialModes(_configurationIf->getConfiguration().initialDiscoverableMode, _configurationIf->getConfiguration().initialConnectableMode);
}

IConfigurationClient* LocalAdapterModes::getConfigurationClientHandler(void)
{
   return this;
}

IStateMachine* LocalAdapterModes::getSmEntryInterface(void)
{
   return this;
}

void LocalAdapterModes::sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetLocalAdapterModes& request, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   if(false == isValidGetRequest(request))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createDiscoverableStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, _currentDiscoverable, _currentDiscoverableOffReason);
   createConnectableStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, _currentConnectable, _currentConnectableOffReason);
}

void LocalAdapterModes::sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetLocalAdapterModes& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   // check if triggered by internal message
   if(true == request.getInternalMessageFlag())
   {
      return;
   }

   // collect data for sending status and result
   BTSRequestResult sendResult(BTS_REQ_SUCCESS);
   const BTSRequestResult inputResult = (BTSRequestResult)resultCode;

   if(false == isValidSetRequest(request))
   {
      FW_NORMAL_ASSERT_ALWAYS();

      sendResult = BTS_REQ_INVALID_PARAM;
   }
   else
   {
      // check current status
      if((BTS_MODE_UNCHANGED != getValidMode(request.getDiscoverableMode())) && (_currentDiscoverable != getValidMode(request.getDiscoverableMode())))
      {
         sendResult = BTS_REQ_FAILED;
      }
      if((BTS_MODE_UNCHANGED != getValidMode(request.getConnectableMode())) && (_currentConnectable != getValidMode(request.getConnectableMode())))
      {
         sendResult = BTS_REQ_FAILED;
      }
   }

   // check given result
   if((BTS_REQ_LAST > inputResult) && (BTS_REQ_SUCCESS != inputResult) && (BTS_REQ_SUCCESS != sendResult))
   {
      // use given result
      sendResult = inputResult;
   }

   // update status
   createDiscoverableStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), sendStatusToAll, _currentDiscoverable, _currentDiscoverableOffReason);
   createConnectableStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), sendStatusToAll, _currentConnectable, _currentConnectableOffReason);
   // update result
   createResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), sendResult);
}

bool LocalAdapterModes::isValidGetRequest(IN const App2Bts_GetLocalAdapterModes& request) const
{
   (void)(request);
   return true;
}

bool LocalAdapterModes::isValidSetRequest(IN const App2Bts_SetLocalAdapterModes& request) const
{
   // invalid modes will be handled as BTS_MODE_DISABLED
   (void)(request);
   return true;
}

bool LocalAdapterModes::setModes(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetLocalAdapterModes& request)
{
   ETG_TRACE_USR2((" setModes"));

   FW_IF_NULL_PTR_RETURN_FALSE(_requestIf);
   FW_IF_NULL_PTR_RETURN_FALSE(_switchBluetoothIf);

   if(false == isValidSetRequest(request))
   {
      FW_NORMAL_ASSERT_ALWAYS();

      createDiscoverableStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, _currentDiscoverable, _currentDiscoverableOffReason);
      createConnectableStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, _currentConnectable, _currentConnectableOffReason);
      createResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_INVALID_PARAM);
      return false;
   }

   // check given input data
   BTSLocalMode newDiscoverableMode(getValidMode(request.getDiscoverableMode()));
   BTSLocalMode newConnectableMode(getValidMode(request.getConnectableMode()));

   // check if BT is on
   if(true == _switchBluetoothIf->getAppBtMode())
   {
      bool modeCompare(false);
      processNewLocalModes(modeCompare, newDiscoverableMode, request.getDiscoverableTimeout(), newConnectableMode, request.getConnectableTimeout(), request.getInternalMessageFlag(), request.getUserData());

      // check compare result
      if(true == modeCompare)
      {
         // send answer directly because requested modes are already set

         // check if triggered by internal message
         if(false == request.getInternalMessageFlag())
         {
            // update status
            createDiscoverableStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, _currentDiscoverable, _currentDiscoverableOffReason);
            createConnectableStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, _currentConnectable, _currentConnectableOffReason);
            // update result
            createResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_SUCCESS);
         }

         // no further steps necessary
         return false;
      }
      else
      {
         // different mode requested

         // store data and process request
         request.getCompareItem(_requestItem.item);
         _requestItem.user = request.getUser();
         _requestItem.handle = request.getSessionHandle();

         prepareSendLocalAdapterModes();
         sendLocalAdapterModes(bts2IpcMsgList, bts2AppMsgList);

         return true;
      }
   }
   else
   {
      // check if triggered by internal message
      if(true == request.getInternalMessageFlag())
      {
         // what happened?
         FW_NORMAL_ASSERT_ALWAYS();

         // no further steps necessary during BT off
         return false;
      }
      else
      {
         // BT is off => store new modes to configuration and answer directly

         BTSDiscoverableTimeout newDiscoverableTimeout(0);
         BTSConnectableTimeout newConnectableTimeout(0);

         // check timeout values
         if(BTS_MODE_ENABLED == newDiscoverableMode)
         {
            newDiscoverableTimeout = request.getDiscoverableTimeout();
         }

         if(BTS_MODE_ENABLED == newConnectableMode)
         {
            newConnectableTimeout = request.getConnectableTimeout();
         }

         // HINT: only infinite timeouts are allowed
         if(0 < newDiscoverableTimeout)
         {
            // this makes no sense
            FW_NORMAL_ASSERT_ALWAYS();
         }

         if(0 < newConnectableTimeout)
         {
            // this makes no sense
            FW_NORMAL_ASSERT_ALWAYS();
         }

#if 1
         // new handling: same as normal processing during BT on
         bool modeCompare(false);
         processNewLocalModes(modeCompare, newDiscoverableMode, 0, newConnectableMode, 0, false, 0);

         // set current values (matches handling during setting modes during BT on)
         _currentDiscoverable = _internalDiscoverable;
         _currentConnectable = _internalConnectable;
#else
         // old handling: set directly
         // check current local modes
         if(BTS_MODE_DISABLED == newDiscoverableMode)
         {
            _currentDiscoverableOffReason = BTS_MODE_OFF_REASON_APP_REQUESTED;
         }
         else if(BTS_MODE_ENABLED == newDiscoverableMode)
         {
            _currentDiscoverableOffReason = BTS_MODE_OFF_REASON_NOT_VALID;
         }
         else
         {
            newDiscoverableMode = _currentDiscoverable;
         }

         if(BTS_MODE_DISABLED == newConnectableMode)
         {
            _currentConnectableOffReason = BTS_MODE_OFF_REASON_APP_REQUESTED;
         }
         else if(BTS_MODE_ENABLED == newConnectableMode)
         {
            _currentConnectableOffReason = BTS_MODE_OFF_REASON_NOT_VALID;
         }
         else
         {
            newConnectableMode = _currentConnectable;
         }

         // store current modes
         setCurrentLocalModes(newDiscoverableMode, 0, newConnectableMode, 0);
#endif

         // update status
         createDiscoverableStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, _currentDiscoverable, _currentDiscoverableOffReason);
         createConnectableStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, _currentConnectable, _currentConnectableOffReason);
         // update result
         createResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_SUCCESS);

         // no further steps necessary
         return false;
      }
   }
}

void LocalAdapterModes::storeNewModes(IN const App2Bts_SetLocalAdapterModes& request)
{
   ETG_TRACE_USR2((" storeNewModes"));

   if(false == isValidSetRequest(request))
   {
      FW_NORMAL_ASSERT_ALWAYS();

      return;
   }

   // check given input data
   const BTSLocalMode newDiscoverableMode(getValidMode(request.getDiscoverableMode()));
   const BTSLocalMode newConnectableMode(getValidMode(request.getConnectableMode()));

   // store values
   setNewLocalModes(newDiscoverableMode, request.getDiscoverableTimeout(), newConnectableMode, request.getConnectableTimeout());
   setOffReasons(newDiscoverableMode, newConnectableMode, request.getInternalMessageFlag(), (_timeoutMarker == request.getUserData()));
   checkForcedMarker(newDiscoverableMode, newConnectableMode, request.getInternalMessageFlag(), (_forcedMarker == request.getUserData()));
}

void LocalAdapterModes::setInitialModes(IN const BTSLocalMode discoverableMode, IN const BTSLocalMode connectableMode)
{
   // check given input data
   const BTSLocalMode newDiscoverableMode(getValidMode(discoverableMode));
   const BTSLocalMode newConnectableMode(getValidMode(connectableMode));

   // same as normal processing during BT on
   bool modeCompare(false);
   processNewLocalModes(modeCompare, newDiscoverableMode, 0, newConnectableMode, 0, false, 0);

   // set current values (matches handling during setting modes during BT on)
   _currentDiscoverable = _internalDiscoverable;
   _currentConnectable = _internalConnectable;

   // reset off reasons
   if(BTS_MODE_DISABLED == _currentDiscoverable)
   {
      _currentDiscoverableOffReason = BTS_MODE_OFF_REASON_INTERNAL;
   }
   if(BTS_MODE_DISABLED == _currentConnectable)
   {
      _currentConnectableOffReason = BTS_MODE_OFF_REASON_INTERNAL;
   }
}

bool LocalAdapterModes::isSetLocalModesSequenceOngoing(void) const
{
   return isSequenceOngoing();
}

void LocalAdapterModes::setDiscoverableTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSDiscoverableTimeout discoverableTimeout)
{
   FW_IF_NULL_PTR_RETURN(_requestIf);

   ::std::vector< Bts2App_BaseMessage* > sendBts2AppMsgList;
   _requestIf->setDiscoverableTimeout(bts2IpcMsgList, sendBts2AppMsgList, discoverableTimeout);
}

bool LocalAdapterModes::getSettingLocalAdapterModesActive(void) const
{
   return _settingLocalAdapterModesActive;
}

void LocalAdapterModes::forceLocalModesUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   FW_IF_NULL_PTR_RETURN(_configurationIf);

   updateDiscoverable(bts2IpcMsgList, bts2AppMsgList, messageItem, false, true, true);
   updatePairable(bts2IpcMsgList, bts2AppMsgList, messageItem, false, true, true);
   updateConnectable(bts2IpcMsgList, bts2AppMsgList, messageItem, false, true, true);

   // Bluetooth is switched off => reset start values
   setInitialModes(_configurationIf->getConfiguration().initialDiscoverableMode, _configurationIf->getConfiguration().initialConnectableMode);
}

void LocalAdapterModes::setTestTriggerBlockSetDiscoverable(IN const bool enable)
{
   _testTriggerBlockSetDiscoverable = enable;
}

void LocalAdapterModes::updateDiscoverable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const bool success, IN const bool force /*= false*/)
{
   updateMode(bts2IpcMsgList, bts2AppMsgList, messageItem, DISCOVERABLE, enabled, success, force);
}

void LocalAdapterModes::updatePairable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const bool success, IN const bool force /*= false*/)
{
   updateMode(bts2IpcMsgList, bts2AppMsgList, messageItem, PAIRABLE, enabled, success, force);
}

void LocalAdapterModes::updateConnectable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const bool success, IN const bool force /*= false*/)
{
   updateMode(bts2IpcMsgList, bts2AppMsgList, messageItem, CONNECTABLE, enabled, success, force);
}

void LocalAdapterModes::updateDiscoverableTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoverableTimeout timeout, IN const bool success, IN const bool force /*= false*/)
{
   // not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(timeout);
   (void)(success);
   (void)(force);
}

void LocalAdapterModes::updatePairableTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoverableTimeout timeout, IN const bool success, IN const bool force /*= false*/)
{
   // not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(timeout);
   (void)(success);
   (void)(force);
}

void LocalAdapterModes::setInitialStackDiscoverableMode(IN const bool enabled)
{
   setStackValue(DISCOVERABLE, enabled);
}

void LocalAdapterModes::setInitialStackPairableMode(IN const bool enabled)
{
   setStackValue(PAIRABLE, enabled);
}

void LocalAdapterModes::setInitialStackConnectableMode(IN const bool enabled)
{
   setStackValue(CONNECTABLE, enabled);
}

void LocalAdapterModes::handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId)
{
   (void)(messageItem);

   FW_IF_NULL_PTR_RETURN(_requestIf);
   FW_IF_NULL_PTR_RETURN(_controlIf);

   // find related timer
   if(true == _timerDiscoverableMode.compare(timerId))
   {
      ETG_TRACE_USR2((" handleExtendedTimeout(): discoverable mode"));

      // after timeout, local adapter modes will be set back to 'DISABLED'
      // use App2Bts message to perform all necessary checks e.g. doubled requests
      _controlIf->pushApp2BtsMessage(createInternalSetLocalAdapterModesRequest(BTS_MODE_DISABLED, BTS_MODE_UNCHANGED, _timeoutMarker));
   }
   else if(true == _timerConnectableMode.compare(timerId))
   {
      ETG_TRACE_USR2((" handleExtendedTimeout(): connectable mode"));

      // after timeout, local adapter modes will be set back to 'DISABLED'
      // use App2Bts message to perform all necessary checks e.g. doubled requests
      _controlIf->pushApp2BtsMessage(createInternalSetLocalAdapterModesRequest(BTS_MODE_UNCHANGED, BTS_MODE_DISABLED, _timeoutMarker));
   }
   else if(true == _timerProperty.compare(timerId))
   {
      // check current active request
      if(true == _setActive[DISCOVERABLE])
      {
         _requestIf->sendVirtualDiscoverableUpdate(bts2IpcMsgList, bts2AppMsgList, false, BTS_IPC_UPDATE_TIMEOUT);
      }
      else if(true == _setActive[PAIRABLE])
      {
         _requestIf->sendVirtualPairableUpdate(bts2IpcMsgList, bts2AppMsgList, false, BTS_IPC_UPDATE_TIMEOUT);
      }
      else if(true == _setActive[CONNECTABLE])
      {
         _requestIf->sendVirtualConnectableUpdate(bts2IpcMsgList, bts2AppMsgList, false, BTS_IPC_UPDATE_TIMEOUT);
      }
   }
}

void LocalAdapterModes::setConfigurationMasterIf(IN IConfigurationMaster* master)
{
   _configurationMaster = master;

   FW_NORMAL_ASSERT(0 != _configurationMaster);
}

void LocalAdapterModes::startGlobalConfiguration(void)
{
   _globalConfigActive = true;
   resetAllFlags();
   _configRequestDone = false; // because this configuration shall be done during each configuration sequence
   _settingLocalAdapterModesActive = false;
}

void LocalAdapterModes::stopGlobalConfiguration(void)
{
   _globalConfigActive = false;
   resetAllFlags();
   _settingLocalAdapterModesActive = false;
}

bool LocalAdapterModes::isSingleConfigurationOngoing(void) const
{
   if(false == _globalConfigActive)
   {
      return false;
   }

   // check if set local modes sequence is ongoing
   return isSequenceOngoing();
}

bool LocalAdapterModes::setSingleConfiguration(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool beforeBtOn, IN const bool errorOccurred)
{
   (void)(messageItem);
   (void)(beforeBtOn);

   FW_IF_NULL_PTR_RETURN_FALSE(_requestIf);
   FW_IF_NULL_PTR_RETURN_FALSE(_configurationIf);

   if(false == _globalConfigActive)
   {
      return false;
   }

   // ETG_TRACE_USR3((" LocalAdapterModes::setSingleConfiguration(): beforeBtOn=%d isSequenceOngoing=%d errorOccurred=%d _configRequestDone=%d", beforeBtOn, isSequenceOngoing(), errorOccurred, _configRequestDone));

   _settingLocalAdapterModesActive = true;

   if(true == isSequenceOngoing())
   {
      // configuration request is ongoing => wait for result
      return true;
   }

   if(true == errorOccurred)
   {
      // error occurred => do not continue
      return false;
   }

   if(false == _configRequestDone)
   {
      // configure local modes
      prepareSendLocalAdapterModes(true);
      sendLocalAdapterModes(bts2IpcMsgList, bts2AppMsgList);

      return true;
   }

   return false;
}

void LocalAdapterModes::processNewLocalModes(OUT bool& modeCompareResult, IN const BTSLocalMode discoverable, IN const BTSDiscoverableTimeout discoverableTimeout, IN const BTSLocalMode connectable, IN const BTSConnectableTimeout connectableTimeout, IN const bool internal, IN const unsigned int data)
{
   // first step: set new modes and off reasons
   setNewLocalModes(discoverable, discoverableTimeout, connectable, connectableTimeout);
   setOffReasons(discoverable, connectable, internal, (_timeoutMarker == data));
   checkForcedMarker(discoverable, connectable, internal, (_forcedMarker == data));

   // second step: convert to internal states
   convertNewStatesToInternalState();

   // third step: compare; timers are checked by next function call (handle timers independent of modes)
   bool timeoutCompare(false);
   compareNewInternalLocalModes(modeCompareResult, timeoutCompare);

   // fourth step: check timer (handle timers independent of modes)
   checkTimer(timeoutCompare);
}

void LocalAdapterModes::setNewLocalModes(IN const BTSLocalMode discoverable, IN const BTSDiscoverableTimeout discoverableTimeout, IN const BTSLocalMode connectable, IN const BTSConnectableTimeout connectableTimeout)
{
   // take over new value
   if(BTS_MODE_DISABLED == discoverable)
   {
      _newDiscoverable = discoverable;
      _newDiscoverableTimeout = 0;
      _requestedDiscoverable = discoverable;
      _requestedDiscoverableTimeout = 0;
   }
   else if(BTS_MODE_ENABLED == discoverable)
   {
      _newDiscoverable = discoverable;
      _newDiscoverableTimeout = discoverableTimeout;
      _requestedDiscoverable = discoverable;
      _requestedDiscoverableTimeout = discoverableTimeout;
   }
   else
   {
      // reuse old requested state
      _newDiscoverable = _requestedDiscoverable;
      _newDiscoverableTimeout = _requestedDiscoverableTimeout;
   }

   // take over new value
   if(BTS_MODE_DISABLED == connectable)
   {
      _newConnectable = connectable;
      _newConnectableTimeout = 0;
      _requestedConnectable = connectable;
      _requestedConnectableTimeout = 0;
   }
   else if(BTS_MODE_ENABLED == connectable)
   {
      _newConnectable = connectable;
      _newConnectableTimeout = connectableTimeout;
      _requestedConnectable = connectable;
      _requestedConnectableTimeout = connectableTimeout;
   }
   else
   {
      // reuse old requested state
      _newConnectable = _requestedConnectable;
      _newConnectableTimeout = _requestedConnectableTimeout;
   }
}

void LocalAdapterModes::setOffReasons(IN const BTSLocalMode discoverable, IN const BTSLocalMode connectable, IN const bool internal, IN const bool timeout)
{
   // backup off reasons first
   _previousDiscoverableOffReason = _currentDiscoverableOffReason;
   _previousConnectableOffReason = _currentConnectableOffReason;

   if(BTS_MODE_DISABLED == discoverable)
   {
      if(true == timeout)
      {
         _currentDiscoverableOffReason = BTS_MODE_OFF_REASON_TIMEOUT;
      }
      else if(true == internal)
      {
         _currentDiscoverableOffReason = BTS_MODE_OFF_REASON_INTERNAL;
      }
      else
      {
         _currentDiscoverableOffReason = BTS_MODE_OFF_REASON_APP_REQUESTED;
      }
   }
   else if(BTS_MODE_ENABLED == discoverable)
   {
      _currentDiscoverableOffReason = BTS_MODE_OFF_REASON_NOT_VALID;
   }
   else
   {
      // keep off reason
   }

   if(BTS_MODE_DISABLED == connectable)
   {
      // check also convertNewStatesToInternalState()
      if((BTS_MODE_ENABLED == discoverable) || (BTS_MODE_ENABLED == _newDiscoverable))
      {
         // connectable will be set ON too => same off reason
         _currentConnectableOffReason = BTS_MODE_OFF_REASON_NOT_VALID;
      }
      else
      {
         // consider connectable mode
         if(true == timeout)
         {
            _currentConnectableOffReason = BTS_MODE_OFF_REASON_TIMEOUT;
         }
         else if(true == internal)
         {
            _currentConnectableOffReason = BTS_MODE_OFF_REASON_INTERNAL;
         }
         else
         {
            _currentConnectableOffReason = BTS_MODE_OFF_REASON_APP_REQUESTED;
         }
      }
   }
   else if(BTS_MODE_ENABLED == connectable)
   {
      _currentConnectableOffReason = BTS_MODE_OFF_REASON_NOT_VALID;
   }
   else
   {
      // check also convertNewStatesToInternalState()
      if(BTS_MODE_ENABLED == discoverable)
      {
         // connectable will be set ON too => same off reason
         _currentConnectableOffReason = BTS_MODE_OFF_REASON_NOT_VALID;
      }
      else if(BTS_MODE_DISABLED == discoverable)
      {
         if(BTS_MODE_ENABLED == _newConnectable)
         {
            _currentConnectableOffReason = BTS_MODE_OFF_REASON_NOT_VALID;
         }
         else
         {
            _currentConnectableOffReason = BTS_MODE_OFF_REASON_INTERNAL;
         }
      }
      else
      {
         // keep off reason
      }
   }
}

void LocalAdapterModes::checkOffReasons(void)
{
   if(BTS_MODE_ENABLED == _currentDiscoverable)
   {
      // the only valid off reason is NOT_VALID
      if(BTS_MODE_OFF_REASON_NOT_VALID != _currentDiscoverableOffReason)
      {
         // something went wrong
         if(BTS_MODE_OFF_REASON_NOT_VALID == _previousDiscoverableOffReason)
         {
            // it seems that setting discoverable mode off at Bluetooth stack failed
            ETG_TRACE_ERR((" checkOffReasons(): it seems that setting discoverable mode off at Bluetooth stack failed"));

            // take over previous off reason
            _currentDiscoverableOffReason = _previousDiscoverableOffReason;
         }
         else
         {
            // unclear what happened
            ETG_TRACE_ERRMEM((" #CONN: BtStackIf: _previousDiscoverableOffReason=%d", _previousDiscoverableOffReason));

            // set off reason
            _currentDiscoverableOffReason = BTS_MODE_OFF_REASON_NOT_VALID;
         }
      }
   }
   else
   {
      // the only valid off reasons are APP_REQUESTED, TIMEOUT, INTERNAL
      if((BTS_MODE_OFF_REASON_APP_REQUESTED != _currentDiscoverableOffReason) && (BTS_MODE_OFF_REASON_TIMEOUT != _currentDiscoverableOffReason) && (BTS_MODE_OFF_REASON_INTERNAL != _currentDiscoverableOffReason))
      {
         // something went wrong
         if((BTS_MODE_OFF_REASON_APP_REQUESTED == _previousDiscoverableOffReason) || (BTS_MODE_OFF_REASON_TIMEOUT == _previousDiscoverableOffReason) || (BTS_MODE_OFF_REASON_INTERNAL == _previousDiscoverableOffReason))
         {
            // it seems that setting discoverable mode on at Bluetooth stack failed
            ETG_TRACE_ERR((" checkOffReasons(): it seems that setting discoverable mode on at Bluetooth stack failed"));

            // take over previous off reason
            _currentDiscoverableOffReason = _previousDiscoverableOffReason;
         }
         else
         {
            // unclear what happened
            ETG_TRACE_ERRMEM((" #CONN: BtStackIf: _previousDiscoverableOffReason=%d", _previousDiscoverableOffReason));

            // set off reason
            _currentDiscoverableOffReason = BTS_MODE_OFF_REASON_INTERNAL;
         }
      }
   }

   if(BTS_MODE_ENABLED == _currentConnectable)
   {
      // the only valid off reason is NOT_VALID
      if(BTS_MODE_OFF_REASON_NOT_VALID != _currentConnectableOffReason)
      {
         // something went wrong
         if(BTS_MODE_OFF_REASON_NOT_VALID == _previousConnectableOffReason)
         {
            // it seems that setting connectable mode off at Bluetooth stack failed
            ETG_TRACE_ERR((" checkOffReasons(): it seems that setting connectable mode off at Bluetooth stack failed"));

            // take over previous off reason
            _currentConnectableOffReason = _previousConnectableOffReason;
         }
         else
         {
            // unclear what happened
            ETG_TRACE_ERRMEM((" #CONN: BtStackIf: _previousConnectableOffReason=%d", _previousConnectableOffReason));

            // set off reason
            _currentConnectableOffReason = BTS_MODE_OFF_REASON_NOT_VALID;
         }
      }
   }
   else
   {
      // the only valid off reasons are APP_REQUESTED, TIMEOUT, INTERNAL
      if((BTS_MODE_OFF_REASON_APP_REQUESTED != _currentConnectableOffReason) && (BTS_MODE_OFF_REASON_TIMEOUT != _currentConnectableOffReason) && (BTS_MODE_OFF_REASON_INTERNAL != _currentConnectableOffReason))
      {
         // something went wrong
         if((BTS_MODE_OFF_REASON_APP_REQUESTED == _previousConnectableOffReason) || (BTS_MODE_OFF_REASON_TIMEOUT == _previousConnectableOffReason) || (BTS_MODE_OFF_REASON_INTERNAL == _previousConnectableOffReason))
         {
            // it seems that setting connectable mode on at Bluetooth stack failed
            ETG_TRACE_ERR((" checkOffReasons(): it seems that setting connectable mode on at Bluetooth stack failed"));

            // take over previous off reason
            _currentConnectableOffReason = _previousConnectableOffReason;
         }
         else
         {
            // unclear what happened
            ETG_TRACE_ERRMEM((" #CONN: BtStackIf: _previousConnectableOffReason=%d", _previousConnectableOffReason));

            // set off reason
            _currentConnectableOffReason = BTS_MODE_OFF_REASON_INTERNAL;
         }
      }
   }
}

void LocalAdapterModes::checkActiveModeTimer(void)
{
   if(BTS_MODE_DISABLED == _currentDiscoverable)
   {
      // stop timer if running
      if(true == isTimerActive(_timerDiscoverableMode))
      {
         stopTimer(_timerDiscoverableMode);
         ETG_TRACE_USR2((" checkTimer(): discoverable: stop timer"));
      }
   }

   if(BTS_MODE_DISABLED == _currentConnectable)
   {
      // stop timer if running
      if(true == isTimerActive(_timerConnectableMode))
      {
         stopTimer(_timerConnectableMode);
         ETG_TRACE_USR2((" checkTimer(): connectable: stop timer"));
      }
   }
}

void LocalAdapterModes::checkForcedMarker(IN const BTSLocalMode discoverable, IN const BTSLocalMode connectable, IN const bool internal, IN const bool forced)
{
   if(false == internal)
   {
      return;
   }

   if(false == forced)
   {
      return;
   }

   if(BTS_MODE_UNCHANGED != discoverable)
   {
      _forcedSet[DISCOVERABLE] = true;
   }

   if(BTS_MODE_UNCHANGED != connectable)
   {
      // old: _forcedSet[PAIRABLE] = true;
      _forcedSet[CONNECTABLE] = true;
   }
}

void LocalAdapterModes::convertNewStatesToInternalState(void)
{
   /**
    * Conversion table:
    *  -----------------------------------------------------------------------------------------------
    * |  requested state (from application)   |    internal state correction (matches Genivi)        |
    * |                                       |                                                      |
    * |---------------------------------------|------------------------------------------------------|
    * | discoverable     | connectable        |   discoverable   |   pairable   |   connectable      |
    * |------------------|--------------------|------------------|--------------|--------------------|
    * |      0           |          0         |        0         |      0       |         0          |
    * |      0           |          1         |        0         |      1       |         1          |
    * |      1           |          0         |        1         |      1       |         1          |
    * |      1           |          1         |        1         |      1       |         1          |
    *  -----------------------------------------------------------------------------------------------
    */

   // HINT: values for _newDiscoverable and _newConnectable are either BTS_MODE_ENABLED or BTS_MODE_DISABLED

   // first: set discoverable mode
   _internalDiscoverable = _newDiscoverable;
   _internalDiscoverableTimeout = _newDiscoverableTimeout;

   // second: set connectable mode
   if(BTS_MODE_ENABLED == _internalDiscoverable)
   {
      if(BTS_MODE_ENABLED == _newConnectable)
      {
         _internalConnectableTimeout = _newConnectableTimeout;
      }
      else
      {
         _internalConnectableTimeout = _internalDiscoverableTimeout;
      }
      _internalConnectable = BTS_MODE_ENABLED;
   }
   else
   {
      _internalConnectable = _newConnectable;
      _internalConnectableTimeout = _newConnectableTimeout;
   }
}

void LocalAdapterModes::compareNewInternalLocalModes(OUT bool& modeCompareResult, OUT bool& timeoutCompareResult) const
{
   bool equalModes(true);
   bool equalTimeouts(true);

   if((_internalDiscoverable != _currentDiscoverable) || (true == _forcedSet[DISCOVERABLE]))
   {
      equalModes = false;
   }
   else if((_internalConnectable != _currentConnectable) || (true == _forcedSet[CONNECTABLE]))
   {
      equalModes = false;
   }

   if((_internalDiscoverableTimeout != _currentDiscoverableTimeout) || (true == _forcedSet[DISCOVERABLE]))
   {
      equalTimeouts = false;
   }
   else if((_internalConnectableTimeout != _currentConnectableTimeout) || (true == _forcedSet[CONNECTABLE]))
   {
      equalTimeouts = false;
   }

   modeCompareResult = equalModes;
   timeoutCompareResult = equalTimeouts;

   // in case of no change of requested modes we will ignore any timeout change
   if(true == equalModes)
   {
      timeoutCompareResult = true;
   }
}

void LocalAdapterModes::checkTimer(IN const bool timeoutCompareResult)
{
   if(true == timeoutCompareResult)
   {
      // timeout values are unchanged => return
      return;
   }

   // check for starting or stopping timer
   // HINT: timeout values are set properly
   if(_internalDiscoverableTimeout != _currentDiscoverableTimeout)
   {
      if(0 == _internalDiscoverableTimeout)
      {
         stopTimer(_timerDiscoverableMode);
         ETG_TRACE_USR2((" checkTimer(): discoverable: stop timer"));
      }
      else
      {
         startTimer(_timerDiscoverableMode, 1000 * _internalDiscoverableTimeout);
         ETG_TRACE_USR2((" checkTimer(): discoverable: start timer (%u ms)", (1000 * _internalDiscoverableTimeout)));
      }
   }

   if(_internalConnectableTimeout != _currentConnectableTimeout)
   {
      if(0 == _internalConnectableTimeout)
      {
         stopTimer(_timerConnectableMode);
         ETG_TRACE_USR2((" checkTimer(): connectable: stop timer"));
      }
      else
      {
         startTimer(_timerConnectableMode, 1000 * _internalConnectableTimeout);
         ETG_TRACE_USR2((" checkTimer(): connectable: start timer (%u ms)", (1000 * _internalConnectableTimeout)));
      }
   }

   _currentDiscoverableTimeout = _internalDiscoverableTimeout;
   _currentConnectableTimeout = _internalConnectableTimeout;
}

void LocalAdapterModes::prepareSendLocalAdapterModes(IN const bool forceSending /*= false*/)
{
   if((_internalDiscoverable != _currentDiscoverable) || (true == forceSending) || (true == _forcedSet[DISCOVERABLE]))
   {
      _setActive[DISCOVERABLE] = true;
   }
   if((_internalConnectable != _currentConnectable) || (true == forceSending) || (true == _forcedSet[CONNECTABLE]))
   {
      // old: _setActive[PAIRABLE] = true;
      _setActive[CONNECTABLE] = true;
   }

   // 2017-04-25: switch pairable mode always on because (see email subject "ALPS Evolution Genivi: automatic discoverable change"):
   // - Pairable mode is an Evolution internal feature. There is no impact on Bluetooth receiver/transmitter.
   // - Pairable mode is only switched off automatically by Evolution while switching BT off (Powered=false).
   // - There is no side effect if pairable mode is enabled while discoverable and connectable mode are disabled.
   // => check current stack value
   if(_defaultPairableMode != _stackPairable)
   {
      _setActive[PAIRABLE] = true;
   }

   // reset flags and counter for GET requests
   for(size_t i = 0; i < LAST; i++)
   {
      _getActive[i] = false;
      _counterGet[i] = _maxNmbGet;
   }
}

void LocalAdapterModes::sendLocalAdapterModes(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool setResponseFlag /*= false*/)
{
   (void)(setResponseFlag);

   FW_IF_NULL_PTR_RETURN(_requestIf);

   bool msgSent(false);

   // send step by step
   {
      if(true == _setActive[DISCOVERABLE])
      {
         if((_stackDiscoverable != (BTS_MODE_ENABLED == _internalDiscoverable)) || (true == _forcedSet[DISCOVERABLE]))
         {
            bool sendSetDiscoverable(true);

            //===================================================================================================================
            // debug section start
            if(true == _testTriggerBlockSetDiscoverable)
            {
               _testTriggerBlockSetDiscoverable = false;
               sendSetDiscoverable = false;
            }
            // debug section end
            //===================================================================================================================

            if(true == sendSetDiscoverable)
            {
               _requestIf->setDiscoverable(bts2IpcMsgList, bts2AppMsgList, (BTS_MODE_ENABLED == _internalDiscoverable));
            }
         }
         else
         {
            // we will send a virtual Ipc2Bts message to follow normal sequence
            _requestIf->sendVirtualDiscoverableUpdate(bts2IpcMsgList, bts2AppMsgList, (BTS_MODE_ENABLED == _internalDiscoverable), BTS_IPC_SUCCESS);
         }

         // start timer
         startTimer(_timerProperty, _timeoutPropertyUpdate);

         msgSent = true;
      }
   }

   if(false == msgSent)
   {
      if(true == _getActive[DISCOVERABLE])
      {
         _requestIf->getDiscoverable(bts2IpcMsgList, bts2AppMsgList);

         // no timer needed for get

         msgSent = true;
      }
   }

   if(false == msgSent)
   {
      if(true == _setActive[PAIRABLE])
      {
         // old: if((_stackPairable != (BTS_MODE_ENABLED == _internalConnectable)) || (true == _forcedSet[PAIRABLE]))
         if((_stackPairable != _defaultPairableMode) || (true == _forcedSet[PAIRABLE]))
         {
            // old: _requestIf->setPairable(bts2IpcMsgList, bts2AppMsgList, (BTS_MODE_ENABLED == _internalConnectable));
            _requestIf->setPairable(bts2IpcMsgList, bts2AppMsgList, _defaultPairableMode);
         }
         else
         {
            // we will send a virtual Ipc2Bts message to follow normal sequence
            // old: _requestIf->sendVirtualPairableUpdate(bts2IpcMsgList, bts2AppMsgList, (BTS_MODE_ENABLED == _internalConnectable), BTS_IPC_SUCCESS);
            _requestIf->sendVirtualPairableUpdate(bts2IpcMsgList, bts2AppMsgList, _defaultPairableMode, BTS_IPC_SUCCESS);
         }

         // start timer
         startTimer(_timerProperty, _timeoutPropertyUpdate);

         msgSent = true;
      }
   }

   if(false == msgSent)
   {
      if(true == _getActive[PAIRABLE])
      {
         _requestIf->getPairable(bts2IpcMsgList, bts2AppMsgList);

         // no timer needed for get

         msgSent = true;
      }
   }

   if(false == msgSent)
   {
      if(true == _setActive[CONNECTABLE])
      {
         if((_stackConnectable != (BTS_MODE_ENABLED == _internalConnectable)) || (true == _forcedSet[CONNECTABLE]))
         {
            _requestIf->setConnectable(bts2IpcMsgList, bts2AppMsgList, (BTS_MODE_ENABLED == _internalConnectable));
         }
         else
         {
            // we will send a virtual Ipc2Bts message to follow normal sequence
            _requestIf->sendVirtualConnectableUpdate(bts2IpcMsgList, bts2AppMsgList, (BTS_MODE_ENABLED == _internalConnectable), BTS_IPC_SUCCESS);
         }

         // start timer
         startTimer(_timerProperty, _timeoutPropertyUpdate);

         msgSent = true;
      }
   }

   if(false == msgSent)
   {
      if(true == _getActive[CONNECTABLE])
      {
         _requestIf->getConnectable(bts2IpcMsgList, bts2AppMsgList);

         // no timer needed for get

         // msgSent = true;
      }
   }
}

void LocalAdapterModes::setCurrentLocalModes(IN const BTSLocalMode discoverable, IN const BTSDiscoverableTimeout discoverableTimeout, IN const BTSLocalMode connectable, IN const BTSConnectableTimeout connectableTimeout)
{
   _currentDiscoverable = discoverable;
   _currentDiscoverableTimeout = discoverableTimeout;
   _currentConnectable = connectable;
   _currentConnectableTimeout = connectableTimeout;
}

BTSRequestResult LocalAdapterModes::assessCurrentLocalModes(void)
{
   BTSRequestResult result = BTS_REQ_SUCCESS;

   // check current stack values
   if(_stackDiscoverable == (BTS_MODE_ENABLED == _internalDiscoverable))
   {
      _currentDiscoverable = _internalDiscoverable;
   }
   else
   {
      result = BTS_REQ_FAILED;
      if(true == _stackDiscoverable)
      {
         _currentDiscoverable = BTS_MODE_ENABLED;
      }
      else
      {
         _currentDiscoverable = BTS_MODE_DISABLED;
      }
   }

   if((_stackPairable == (BTS_MODE_ENABLED == _internalConnectable)) && (_stackConnectable == (BTS_MODE_ENABLED == _internalConnectable)))
   {
      _currentConnectable = _internalConnectable;
   }
   else
   {
      if(_stackConnectable == (BTS_MODE_ENABLED == _internalConnectable))
      {
         _currentConnectable = _internalConnectable;
      }
      else
      {
         result = BTS_REQ_FAILED;
         if(true == _stackConnectable)
         {
            _currentConnectable = BTS_MODE_ENABLED;
         }
         else
         {
            _currentConnectable = BTS_MODE_DISABLED;
         }
      }
   }

   return result;
}

void LocalAdapterModes::handleActionFinished(OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   messageItem.item = _requestItem.item;
   messageItem.item.opCode = App2BtsOC_None; // to force default handling for end of request sequence
   messageItem.deleteMessage = true;
   findApp2BtsWorkingMessage(messageItem);
}

void LocalAdapterModes::findApp2BtsWorkingMessage(OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   if(0 == messageItem.message)
   {
      messageItem.message = getApp2BtsWorkingMessage();
   }
   // there is no application message in some use cases --- FW_NORMAL_ASSERT(0 != messageItem.message);
}

App2Bts_BaseMessage* LocalAdapterModes::getApp2BtsWorkingMessage(void)
{
   FW_IF_NULL_PTR_RETURN_NULL(_controlIf);

   return _controlIf->findApp2BtsWorkingMessageWrapper(_requestItem.item);
}

void LocalAdapterModes::updateMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const Property property, IN const bool enabled, IN const bool success, IN const bool force /*= false*/)
{
   FW_IF_NULL_PTR_RETURN(_switchBluetoothIf);
   FW_IF_NULL_PTR_RETURN(_controlIf);

   if(LAST <= property)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // stop timer
   stopPropertyTimer(property);

   // check if setting sequence is ongoing or spontaneous update from stack side happened
   if(true == isSequenceOngoing(property))
   {
      // reset active flag
      resetFlag(property);

      // check result
      if(true == success)
      {
         // store new value in every case
         setStackValue(property, enabled);
      }
      else
      {
         // send a get request to request current status
         prepareGetLocalAdapterMode(property);
      }

      // check for sending next request
      if(true == isSequenceOngoing())
      {
         // find application message (helpful for debugging)
         findApp2BtsWorkingMessage(messageItem);

         sendLocalAdapterModes(bts2IpcMsgList, bts2AppMsgList);
      }
      else
      {
         // take over new vales
         const BTSRequestResult result(assessCurrentLocalModes());

         // set modes sequence is finished => check off reasons now
         checkOffReasons();

         // set modes sequence is finished => check timer
         checkActiveModeTimer();

         // update status
         bool updateDiscoverableStatus(true);

         // check if discoverable mode shall not be updated during switching BT on
         if(false == _updateDiscoverableMode)
         {
            if((true == _switchBluetoothIf->getSwitchActive()) && (BTS_BT_MODE_ON == _switchBluetoothIf->getNewBtMode()))
            {
               updateDiscoverableStatus = false;
               ETG_TRACE_USR2((" updateMode(): DO NOT UPDATE DISCOVERABLE STATUS"));
            }
         }

         if(true == updateDiscoverableStatus)
         {
            createDiscoverableStatusMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, true, _currentDiscoverable, _currentDiscoverableOffReason);
         }
         createConnectableStatusMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, true, _currentConnectable, _currentConnectableOffReason);
         // update result
         createResultMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, result);

         // action is finished
         handleActionFinished(messageItem);

         // reset control data because action is finished
         _requestItem.reset();

         // configuration is done now (if ongoing)
         _configRequestDone = true;
      }
   }
   else
   {
      // spontaneous update from stack side happened
      const bool oldMode(getStackValue(property));

      // check current state
      if(true == isSequenceOngoing())
      {
         // set local modes sequence is ongoing but related property is currently not processed => store value only; status update will happen at the end of the sequence
         // check result
         if(true == success)
         {
            // store new value in every case
            setStackValue(property, enabled);
            // continue with existing off reason
         }
         else
         {
            // continue with previous value
         }
      }
      else
      {
         // set local modes sequence is not ongoing => store value and update status
         // check result
         if(true == success)
         {
            // check if BT is on or new value is forced
            if((true == _switchBluetoothIf->getAppBtMode()) || (true == force))
            {
               // store current modes
               const BTSLocalMode oldDiscoverable(_currentDiscoverable);
               const BTSLocalMode oldConnectable(_currentConnectable);
               // store new value in every case
               setStackValue(property, enabled);
               // take over new vales
               (void)assessCurrentLocalModes();
               // check for changed value
               if(oldDiscoverable != _currentDiscoverable)
               {
                  // set off reason
                  setOffReason(property, BTS_MODE_OFF_REASON_INTERNAL);
                  // update status
                  createDiscoverableStatusMsg(bts2AppMsgList, 0, 0, true, _currentDiscoverable, _currentDiscoverableOffReason);
               }
               else if(oldConnectable != _currentConnectable)
               {
                  // set off reason
                  setOffReason(property, BTS_MODE_OFF_REASON_INTERNAL);
                  // update status
                  createConnectableStatusMsg(bts2AppMsgList, 0, 0, true, _currentConnectable, _currentConnectableOffReason);
               }
            }
            else
            {
               // ignore because BT is off; while switching to BT off all stack values are set to off
            }
         }
         else
         {
            // continue with previous value
         }
      }

      // check if stack value changed
      const bool newMode(getStackValue(property));
      if((oldMode != newMode) && (false == newMode) && (true == _switchBluetoothIf->getAppBtMode()) && (true == isModeTimerActive(property)))
      {
         // current mode was changed by stack e.g. discoverable mode is switched off by stack in case of ongoing audio streaming
         // handle only discoverable mode
         if(DISCOVERABLE == property)
         {
            // stop timer
            // timer is stopped while processing internal app request

            // send switch off request to stack (stack will remember host application decision)
            if(true == isSequenceOngoing())
            {
               // sequence is still ongoing => mark property to be switched off (forced)
               // send internal message because we want to finish current sequence
               _controlIf->sendInternalApp2BtsMessage(createInternalSetLocalAdapterModesRequest(BTS_MODE_DISABLED, BTS_MODE_UNCHANGED, _forcedMarker), true);
            }
            else
            {
               // no sequence ongoing => trigger new internal app request (similar to timeout; forced)
               // send internal message
               _controlIf->sendInternalApp2BtsMessage(createInternalSetLocalAdapterModesRequest(BTS_MODE_DISABLED, BTS_MODE_UNCHANGED, _forcedMarker), true);
            }
         }
      }
   }
}

bool LocalAdapterModes::isSequenceOngoing(void) const
{
   // check related active flags
   bool ongoing(false);

   for(size_t i = 0; i < LAST; i++)
   {
      if((true == _setActive[i]) || (true == _getActive[i]))
      {
         ongoing = true;
         break;
      }
   }

   return ongoing;
}

bool LocalAdapterModes::isSequenceOngoing(IN const Property property) const
{
   if(LAST > property)
   {
      return (_setActive[property] || _getActive[property]);
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }
}

void LocalAdapterModes::resetFlag(IN const Property property)
{
   if(LAST > property)
   {
      // sequence starts with SET; GET is added if SET failed; only 1 request is set at the same time; therefore both can be deleted
      _setActive[property] = false;
      _getActive[property] = false;
      _forcedSet[property] = false;
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void LocalAdapterModes::resetAllFlags(void)
{
   for(size_t i = 0; i < LAST; i++)
   {
      resetFlag((Property)i);
   }
}

void LocalAdapterModes::setStackValue(IN const Property property, IN const bool enabled)
{
   switch(property)
   {
      case DISCOVERABLE:
         _stackDiscoverable = enabled;
         break;
      case PAIRABLE:
         _stackPairable = enabled;
         break;
      case CONNECTABLE:
         _stackConnectable = enabled;
         break;
      case LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }
}

bool LocalAdapterModes::getStackValue(IN const Property property) const
{
   bool enabled(false);

   switch(property)
   {
      case DISCOVERABLE:
         enabled = _stackDiscoverable;
         break;
      case PAIRABLE:
         enabled = _stackPairable;
         break;
      case CONNECTABLE:
         enabled = _stackConnectable;
         break;
      case LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   return enabled;
}

bool LocalAdapterModes::isModeTimerActive(IN const Property property) const
{
   bool enabled(false);

   switch(property)
   {
      case DISCOVERABLE:
         enabled = isTimerActive(_timerDiscoverableMode);
         break;
      case PAIRABLE:
         // ignore because stack connectable mode is considered for app connectable mode
         break;
      case CONNECTABLE:
         enabled = isTimerActive(_timerConnectableMode);
         break;
      case LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   return enabled;
}

void LocalAdapterModes::setOffReason(IN const Property property, IN const BTSLocalModeOffReason reason)
{
   switch(property)
   {
      case DISCOVERABLE:
         if(true == _stackDiscoverable)
         {
            _currentDiscoverableOffReason = BTS_MODE_OFF_REASON_NOT_VALID;
         }
         else
         {
            _currentDiscoverableOffReason = reason;
         }
         break;
      case PAIRABLE:
         // ignore because connectable mode is considered for connectable off reason
         break;
      case CONNECTABLE:
         if(true == _stackConnectable)
         {
            _currentConnectableOffReason = BTS_MODE_OFF_REASON_NOT_VALID;
         }
         else
         {
            _currentConnectableOffReason = reason;
         }
         break;
      case LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }
}

void LocalAdapterModes::stopPropertyTimer(IN const Property property)
{
   switch(property)
   {
      case DISCOVERABLE:
         stopTimer(_timerProperty);
         break;
      case PAIRABLE:
         stopTimer(_timerProperty);
         break;
      case CONNECTABLE:
         stopTimer(_timerProperty);
         break;
      case LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }
}

void LocalAdapterModes::prepareGetLocalAdapterMode(IN const Property property)
{
   if(LAST > property)
   {
      // set only to active if remaining requests are available
      if(0 < _counterGet[property])
      {
         _getActive[property] = true;
         --_counterGet[property];
      }
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void LocalAdapterModes::startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout)
{
   ETG_TRACE_USR3((" startTimer: timeout=%u", timeout));

   FW_IF_NULL_PTR_RETURN(_timerPoolIf);
   FW_IF_NULL_PTR_RETURN(_controlIf);

   timer.setTimerPool(_timerPoolIf);
   timer.start(_controlIf, this, timeout);
}

void LocalAdapterModes::stopTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" stopTimer"));

   timer.stop();

   // do not release timer
}

void LocalAdapterModes::releaseTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" releaseTimer"));

   timer.release();
}

bool LocalAdapterModes::isTimerActive(IN const ExtendedTimerEntry& timer) const
{
   return timer.isActive();
}

void LocalAdapterModes::createDiscoverableStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSLocalMode status, IN const BTSLocalModeOffReason reason) const
{
   Bts2App_CurrentDiscoverableMode* msg = ptrNew_Bts2App_CurrentDiscoverableMode();
   if(0 != msg)
   {
      if((true == sendStatusToAll) || (0 == user))
      {
         msg->setUser(0); // send status message to all
         msg->setSessionHandle(0); // send status message to all
      }
      else
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
      }
      msg->setDiscoverableMode(status);
      if(BTS_MODE_ENABLED == status)
      {
         msg->setLocalModeOffReason(BTS_MODE_OFF_REASON_NOT_VALID);
      }
      else
      {
         // correct reason if necessary
         if(BTS_MODE_OFF_REASON_NOT_VALID == reason)
         {
            msg->setLocalModeOffReason(BTS_MODE_OFF_REASON_INTERNAL);
         }
         else
         {
            msg->setLocalModeOffReason(reason);
         }
      }

      bts2AppMsgList.push_back(msg);
   }
}

void LocalAdapterModes::createConnectableStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSLocalMode status, IN const BTSLocalModeOffReason reason) const
{
   Bts2App_CurrentConnectableMode* msg = ptrNew_Bts2App_CurrentConnectableMode();
   if(0 != msg)
   {
      if((true == sendStatusToAll) || (0 == user))
      {
         msg->setUser(0); // send status message to all
         msg->setSessionHandle(0); // send status message to all
      }
      else
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
      }
      msg->setConnectableMode(status);
      if(BTS_MODE_ENABLED == status)
      {
         msg->setLocalModeOffReason(BTS_MODE_OFF_REASON_NOT_VALID);
      }
      else
      {
         // correct reason if necessary
         if(BTS_MODE_OFF_REASON_NOT_VALID == reason)
         {
            msg->setLocalModeOffReason(BTS_MODE_OFF_REASON_INTERNAL);
         }
         else
         {
            msg->setLocalModeOffReason(reason);
         }
      }

      bts2AppMsgList.push_back(msg);
   }
}

void LocalAdapterModes::createResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSRequestResult result) const
{
   if(0 != user)
   {
      Bts2App_SetLocalAdapterModesResult* msg = ptrNew_Bts2App_SetLocalAdapterModesResult();
      if(0 != msg)
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
         msg->setRequestResult(result);

         bts2AppMsgList.push_back(msg);
      }
   }
}

App2Bts_SetLocalAdapterModes* LocalAdapterModes::createInternalSetLocalAdapterModesRequest(IN const BTSLocalMode discoverable, IN const BTSLocalMode connectable, IN const unsigned int data) const
{
   App2Bts_SetLocalAdapterModes* msg = ptrNew_App2Bts_SetLocalAdapterModes();
   if(0 != msg)
   {
      msg->setDiscoverableMode(discoverable);
      msg->setDiscoverableTimeout(0);
      msg->setConnectableMode(connectable);
      msg->setConnectableTimeout(0);
      msg->setInternalMessageFlag(true);
      msg->setUserData(data);
   }

   return msg;
}

} //btstackif
