/**
 * @file LocalAdapterModes.h
 *
 * @par SW-Component
 * State machine for local adapter modes
 *
 * @brief Implementation of generic local adapter modes state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of generic local adapter modes state machine.
 */

#ifndef _LOCAL_ADAPTER_MODES_H_
#define _LOCAL_ADAPTER_MODES_H_

#include "StateMachine.h"
#include "ILocalAdapterModes.h"
#include "ILocalAdapterModesCallback.h"
#include "IExtendedTimeoutHandler.h"
#include "IConfigurationClient.h"
#include "ExtendedTimerEntry.h"

namespace btstackif {

/**
 * LocalAdapterModes class.
 */
class LocalAdapterModes : public StateMachine, public ILocalAdapterModes, public ILocalAdapterModesCallback, public IExtendedTimeoutHandler, public IConfigurationClient
{
public:
   /**
    * Default constructor.
    */
   LocalAdapterModes();

   /**
    * Destructor.
    */
   virtual ~LocalAdapterModes();

   virtual void reset(void);

   virtual void forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void setInstance(IN ILocalAdapterModesRequest* instance);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setTimerPoolIf(IN ITimerPool* timerPool);

   virtual void setSwitchBluetoothIf(IN ISwitchBluetooth* switchBluetooth);

   virtual void setConfigurationIf(IN IConfiguration* configuration);

   virtual IConfigurationClient* getConfigurationClientHandler(void);

   virtual IStateMachine* getSmEntryInterface(void);

   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetLocalAdapterModes& request, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetLocalAdapterModes& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const;

   virtual bool isValidGetRequest(IN const App2Bts_GetLocalAdapterModes& request) const;

   virtual bool isValidSetRequest(IN const App2Bts_SetLocalAdapterModes& request) const;

   virtual bool setModes(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetLocalAdapterModes& request);

   virtual void storeNewModes(IN const App2Bts_SetLocalAdapterModes& request);

   virtual void setInitialModes(IN const BTSLocalMode discoverableMode, IN const BTSLocalMode connectableMode);

   virtual bool isSetLocalModesSequenceOngoing(void) const;

   virtual void setDiscoverableTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSDiscoverableTimeout discoverableTimeout);

   virtual bool getSettingLocalAdapterModesActive(void) const;

   virtual void forceLocalModesUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem);

   virtual void setTestTriggerBlockSetDiscoverable(IN const bool enable);

   virtual void updateDiscoverable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const bool success, IN const bool force = false);

   virtual void updatePairable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const bool success, IN const bool force = false);

   virtual void updateConnectable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const bool success, IN const bool force = false);

   virtual void updateDiscoverableTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoverableTimeout timeout, IN const bool success, IN const bool force = false);

   virtual void updatePairableTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoverableTimeout timeout, IN const bool success, IN const bool force = false);

   virtual void setInitialStackDiscoverableMode(IN const bool enabled);

   virtual void setInitialStackPairableMode(IN const bool enabled);

   virtual void setInitialStackConnectableMode(IN const bool enabled);

   virtual void handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId);

   virtual void setConfigurationMasterIf(IN IConfigurationMaster* master);

   virtual void startGlobalConfiguration(void);

   virtual void stopGlobalConfiguration(void);

   virtual bool isSingleConfigurationOngoing(void) const;

   virtual bool setSingleConfiguration(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool beforeBtOn, IN const bool errorOccurred);

private:
   // #Property#ignore#
   enum Property
   {
      DISCOVERABLE = 0, /**< discoverable */
      PAIRABLE, /**< pairable */
      CONNECTABLE, /**< connectable */
      // -------------------------------------------------------------------------
      LAST /**< end marker, must be the last one */
   };

   ILocalAdapterModesRequest* _requestIf; /**< request interface */
   IBasicControl* _controlIf; /**< basic control interface */
   ITimerPool* _timerPoolIf; /**< timer pool interface */
   ISwitchBluetooth* _switchBluetoothIf; /**< switch Bluetooth interface */
   IConfiguration* _configurationIf; /**< configuration interface */
   IConfigurationMaster* _configurationMaster; /**< configuration master interface */
   BTSApp2BtsMessageRequestItem _requestItem; /**< request item */

   BTSLocalMode _newDiscoverable; /**< new discoverable mode to be set => requested state (with UNCHANGED) */
   BTSDiscoverableTimeout _newDiscoverableTimeout; /**< new timeout for discoverable mode => requested value */
   BTSLocalMode _newConnectable; /**< new connectable mode to be set => requested state (with UNCHANGED) */
   BTSConnectableTimeout _newConnectableTimeout; /**< new timeout for connectable mode => requested value */

   BTSLocalMode _requestedDiscoverable; /**< new discoverable mode to be set => requested state (without UNCHANGED) */
   BTSDiscoverableTimeout _requestedDiscoverableTimeout; /**< new timeout for discoverable mode => requested value */
   BTSLocalMode _requestedConnectable; /**< new connectable mode to be set => requested state (without UNCHANGED) */
   BTSConnectableTimeout _requestedConnectableTimeout; /**< new timeout for connectable mode => requested value */

   BTSLocalMode _currentDiscoverable; /**< current discoverable mode which is currently set from application point of view => current state */
   BTSLocalModeOffReason _currentDiscoverableOffReason; /**< off reason for current discoverable mode */
   BTSDiscoverableTimeout _currentDiscoverableTimeout; /**< current timeout for discoverable mode => current value */
   BTSLocalMode _currentConnectable; /**< current connectable mode which is currently set from application point of view => current state */
   BTSLocalModeOffReason _currentConnectableOffReason; /**< off reason for current connectable mode */
   BTSConnectableTimeout _currentConnectableTimeout; /**< current timeout for connectable mode => current value */

   BTSLocalModeOffReason _previousDiscoverableOffReason; /**< off reason for previous discoverable mode */
   BTSLocalModeOffReason _previousConnectableOffReason; /**< off reason for previous connectable mode */

   BTSLocalMode _internalDiscoverable; /**< internal discoverable state => used for sending DBUS message */
   BTSDiscoverableTimeout _internalDiscoverableTimeout; /**< internal timeout for discoverable mode => used for sending DBUS message */
   BTSLocalMode _internalConnectable; /**< internal connectable state => used for sending DBUS message */
   BTSConnectableTimeout _internalConnectableTimeout; /**< internal timeout for connectable mode => used for sending DBUS message */

   ExtendedTimerEntry _timerDiscoverableMode; /**< timer */
   ExtendedTimerEntry _timerConnectableMode; /**< timer */
   ExtendedTimerEntry _timerProperty; /**< timer */
   const BTSTimeValue _timeoutPropertyUpdate; /**< timeout property update */

   bool _setActive[LAST]; /**< flags for active SET request */
   bool _getActive[LAST]; /**< flags for active GET request */
   unsigned char _counterGet[LAST]; /**< counter for GET requests */
   const unsigned char _maxNmbGet; /**< maximum number for GET requests for 1 property */
   bool _forcedSet[LAST]; /**< flags for forced SET request */

   bool _stackDiscoverable; /**< current stack value */
   bool _stackPairable; /**< current stack value */
   bool _stackConnectable; /**< current stack value */
   BTSDiscoverableTimeout _stackDiscoverableTimeout; /**< current stack value */
   BTSDiscoverableTimeout _stackPairableTimeout; /**< current stack value */

   const unsigned int _timeoutMarker; /**< timeout marker */
   const unsigned int _forcedMarker; /**< forced marker */
   const bool _defaultPairableMode; /**< default pairable mode */
   const bool _updateDiscoverableMode; /**< flag indicating that discoverable mode shall also be updated during switching BT on */

   bool _globalConfigActive; /**< flag for global configuration */
   bool _configRequestDone; /**< flag for finished config request */
   bool _settingLocalAdapterModesActive; /**< flag for started setting sequence */

   bool _testTriggerBlockSetDiscoverable; /**< flag for blocking set command for discoverable mode */

   inline BTSLocalMode getValidMode(IN const BTSLocalMode mode) const { if(BTS_MODE_LAST > mode) return mode; else return BTS_MODE_DISABLED; }

   void processNewLocalModes(OUT bool& modeCompareResult, IN const BTSLocalMode discoverable, IN const BTSDiscoverableTimeout discoverableTimeout, IN const BTSLocalMode connectable, IN const BTSConnectableTimeout connectableTimeout, IN const bool internal, IN const unsigned int data);

   void setNewLocalModes(IN const BTSLocalMode discoverable, IN const BTSDiscoverableTimeout discoverableTimeout, IN const BTSLocalMode connectable, IN const BTSConnectableTimeout connectableTimeout);

   void setOffReasons(IN const BTSLocalMode discoverable, IN const BTSLocalMode connectable, IN const bool internal, IN const bool timeout);

   void checkOffReasons(void);

   void checkActiveModeTimer(void);

   void checkForcedMarker(IN const BTSLocalMode discoverable, IN const BTSLocalMode connectable, IN const bool internal, IN const bool forced);

   void convertNewStatesToInternalState(void);

   void compareNewInternalLocalModes(OUT bool& modeCompareResult, OUT bool& timeoutCompareResult) const;

   void checkTimer(IN const bool timeoutCompareResult);

   void prepareSendLocalAdapterModes(IN const bool forceSending = false);

   void sendLocalAdapterModes(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const bool setResponseFlag = false);

   void setCurrentLocalModes(IN const BTSLocalMode discoverable, IN const BTSDiscoverableTimeout discoverableTimeout, IN const BTSLocalMode connectable, IN const BTSConnectableTimeout connectableTimeout);

   BTSRequestResult assessCurrentLocalModes(void);

   void handleActionFinished(OUT BTSHandleIpc2BtsMessageItem& messageItem);

   void findApp2BtsWorkingMessage(OUT BTSHandleIpc2BtsMessageItem& messageItem);

   App2Bts_BaseMessage* getApp2BtsWorkingMessage(void);

   void updateMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const Property property, IN const bool enabled, IN const bool success, IN const bool force = false);

   bool isSequenceOngoing(void) const;

   bool isSequenceOngoing(IN const Property property) const;

   void resetFlag(IN const Property property);

   void resetAllFlags(void);

   void setStackValue(IN const Property property, IN const bool enabled);

   bool getStackValue(IN const Property property) const;

   bool isModeTimerActive(IN const Property property) const;

   void setOffReason(IN const Property property, IN const BTSLocalModeOffReason reason);

   void stopPropertyTimer(IN const Property property);

   void prepareGetLocalAdapterMode(IN const Property property);

   void startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout);

   void stopTimer(IN ExtendedTimerEntry& timer) const;

   void releaseTimer(IN ExtendedTimerEntry& timer) const;

   bool isTimerActive(IN const ExtendedTimerEntry& timer) const;

   void createDiscoverableStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSLocalMode status, IN const BTSLocalModeOffReason reason) const;

   void createConnectableStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSLocalMode status, IN const BTSLocalModeOffReason reason) const;

   void createResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSRequestResult result) const;

   App2Bts_SetLocalAdapterModes* createInternalSetLocalAdapterModesRequest(IN const BTSLocalMode discoverable, IN const BTSLocalMode connectable, IN const unsigned int data) const;
};

} //btstackif

#endif //_LOCAL_ADAPTER_MODES_H_
