/**
 * @file ILocalAdapterModes.h
 *
 * @par SW-Component
 * State machine for local adapter modes
 *
 * @brief Interface definition for generic local adapter modes state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for local adapter modes state machine.
 */

#ifndef _I_LOCAL_ADAPTER_MODES_H_
#define _I_LOCAL_ADAPTER_MODES_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class ILocalAdapterModesRequest;
class IBasicControl;
class ITimerPool;
class ISwitchBluetooth;
class IConfiguration;
class IConfigurationClient;
class IStateMachine;
class App2Bts_GetLocalAdapterModes;
class App2Bts_SetLocalAdapterModes;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class ILocalAdapterModes
{
public:
   /**
    * Destructor.
    */
   virtual ~ILocalAdapterModes() {}

   /**
    * Set specific instance (implementation).
    *
    * @param[in] instance: instance (implementation)
    */
   virtual void setInstance(IN ILocalAdapterModesRequest* instance) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set timer pool interface.
    *
    * @param[in] timerPool: timer pool interface
    */
   virtual void setTimerPoolIf(IN ITimerPool* timerPool) = 0;

   /**
    * Set switch Bluetooth interface.
    *
    * @param[in] switchBluetooth: witch Bluetooth interface
    */
   virtual void setSwitchBluetoothIf(IN ISwitchBluetooth* switchBluetooth) = 0;

   /**
    * Set configuration interface.
    *
    * @param[in] configuration: configuration interface
    */
   virtual void setConfigurationIf(IN IConfiguration* configuration) = 0;

   /**
    * Get configuration client handler.
    *
    * @return = configuration client handler
    */
   virtual IConfigurationClient* getConfigurationClientHandler(void) = 0;

   /**
    * Get SM entry interface.
    *
    * @return = SM entry interface
    */
   virtual IStateMachine* getSmEntryInterface(void) = 0;

   /**
    * Send current status for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] statusCode: status code
    */
   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetLocalAdapterModes& request, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Send status and result for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] sendStatusToAll: flag for sending status to all or not
    * @param[in] resultCode: result code
    * @param[in] statusCode: status code
    */
   virtual void sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetLocalAdapterModes& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidGetRequest(IN const App2Bts_GetLocalAdapterModes& request) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidSetRequest(IN const App2Bts_SetLocalAdapterModes& request) const = 0;

   /**
    * Set local adapter modes.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool setModes(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetLocalAdapterModes& request) = 0;

   /**
    * Store new local adapter modes. To be used if processing of request is not possible now but new modes shall be used later.
    *
    * @param[in] request: request
    */
   virtual void storeNewModes(IN const App2Bts_SetLocalAdapterModes& request) = 0;

   /**
    * Set initial local adapter modes (configuration values).
    *
    * @param[out] discoverableMode: initial discoverable mode
    * @param[out] connectableMode: initial connectable mode
    */
   virtual void setInitialModes(IN const BTSLocalMode discoverableMode, IN const BTSLocalMode connectableMode) = 0;

   /**
    * Check if sequence is ongoing.
    *
    * @return = true: sequence is ongoing,
    * @return = false: sequence is not ongoing
    */
   virtual bool isSetLocalModesSequenceOngoing(void) const = 0;

   /**
    * Set discoverable timeout. TODO: quick hack for NCG3D-15977 TODO: add to configuration sequence TODO: remove interface
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[in] discoverableTimeout: discoverable timeout
    */
   virtual void setDiscoverableTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSDiscoverableTimeout discoverableTimeout) = 0;

   /**
    * Check if configuration sequence is ongoing.
    *
    * @return = true: sequence is ongoing,
    * @return = false: sequence is not ongoing
    */
   virtual bool getSettingLocalAdapterModesActive(void) const = 0;

   /**
    * Forces update of local modes to false.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    */
   virtual void forceLocalModesUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTestTriggerBlockSetDiscoverable(IN const bool enable) = 0;
};

} //btstackif

#endif //_I_LOCAL_ADAPTER_MODES_H_
