/**
 * @file LinkKey.cpp
 *
 * @par SW-Component
 * State machine for link key
 *
 * @brief Implementation of generic link key state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic link key state machine.
 */

#include "LinkKey.h"
#include "ILinkKeyRequest.h"
#include "IDeviceManager.h"
#include "FwAssert.h"
#include "FwStringUtils.h"
#include "FwBluetoothStringUtils.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"

namespace btstackif {

LinkKey::LinkKey() :
_requestIf(0),
_deviceManager(0),
_pendingList()
{
}

LinkKey::~LinkKey()
{
   _requestIf = 0;
   _deviceManager = 0;
}

void LinkKey::reset(void)
{
   StateMachine::reset();
   // keep _requestIf
   // keep _deviceManager
   _pendingList.clear();

   FW_IF_NULL_PTR_RETURN(_requestIf);
   _requestIf->reset();
}

void LinkKey::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   // nothing to do
   (void)(bts2AppMsgList);

   // reset control data
   reset();
}

void LinkKey::setInstance(IN ILinkKeyRequest* instance)
{
   _requestIf = instance;

   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setCallback(this);
}

void LinkKey::setDeviceManager(IN IDeviceManager* deviceManager)
{
   _deviceManager = deviceManager;

   FW_IF_NULL_PTR_RETURN(_deviceManager);

   _deviceManager->registerObserver(this);
}

IStateMachine* LinkKey::getSmEntryInterface(void)
{
   return this;
}

bool LinkKey::isValidRequest(IN const App2Bts_SetLinkKey& request) const
{
   // TODO: [low]: check link key type + link key + d link key

   return ::fw::isValidBdAddress(request.getBDAddress());
}

bool LinkKey::setLinkKey(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetLinkKey& request)
{
   FW_IF_NULL_PTR_RETURN_FALSE(_requestIf);

   if(false == isValidRequest(request))
   {
      FW_NORMAL_ASSERT_ALWAYS();

      // just wait for end of pairing or remote connect
      return false;
   }

   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);

   // process request
   _requestIf->setLinkKey(bts2IpcMsgList, bts2AppMsgList, workingAddress, request.getLinkKeyType(), request.getLinkKey(), request.getDLinkKey());

   // return false because no further processing necessary
   return false;
}

void LinkKey::indicateLinkKeyRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   (void)(bts2IpcMsgList);
   (void)(messageItem);

   FW_IF_NULL_PTR_RETURN(_deviceManager);

   // check if device is available
   if(true == _deviceManager->isDeviceAdded(address))
   {
      // we can directly forward to application
      createLinkKeyRequestedMsg(bts2AppMsgList, address);

      // check if given device is in pending list (could happen if first link key request is not answered by application); if found remove
      removeDevice(address);
   }
   else
   {
      // we have to add to pending list
      if(true == isDevicePending(address))
      {
         // entry is already stored => should never happen
         FW_NORMAL_ASSERT_ALWAYS();
      }
      else
      {
         // add entry
         _pendingList.push_back(address);
      }
   }
}

void LinkKey::deviceAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   (void)(bts2IpcMsgList);
   (void)(messageItem);

   // check if given device is in pending list
   if(true == isDevicePending(address))
   {
      // entry is available => forward request to application and remove
      createLinkKeyRequestedMsg(bts2AppMsgList, address);
      removeDevice(address);
   }
}

void LinkKey::deviceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   // check if given device is in pending list; if found remove
   removeDevice(address);
}

void LinkKey::deviceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
}

void LinkKey::deviceUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
}

void LinkKey::deviceConnectionStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSStatusTransition aclTransition, IN const bool aclConnected, IN const BTSStatusTransition anyProfileTransition, IN const bool anyProfileConnected)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
   (void)(aclTransition);
   (void)(aclConnected);
   (void)(anyProfileTransition);
   (void)(anyProfileConnected);
}

void LinkKey::deviceCreationFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
   (void)(result);
}

void LinkKey::deviceRemovalFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
   (void)(result);
}

void LinkKey::createLinkKeyRequestedMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address) const
{
   Bts2App_LinkKeyRequested* msg = ptrNew_Bts2App_LinkKeyRequested();
   if(0 != msg)
   {
      msg->setUser(NULL); // send status message to all
      msg->setSessionHandle(0); // send status message to all
      msg->setBDAddress(address);

      bts2AppMsgList.push_back(msg);
   }
}

bool LinkKey::isDevicePending(IN const BTSBDAddress& address) const
{
   for(::std::vector< BTSBDAddress >::const_iterator it = _pendingList.begin(); it != _pendingList.end(); ++it)
   {
      if(*it == address)
      {
         return true;
      }
   }

   return false;
}

void LinkKey::removeDevice(IN const BTSBDAddress& address)
{
   for(::std::vector< BTSBDAddress >::iterator it = _pendingList.begin(); it != _pendingList.end(); ++it)
   {
      if(*it == address)
      {
         _pendingList.erase(it);
         break;
      }
   }
}

} //btstackif
