/**
 * @file ILinkKey.h
 *
 * @par SW-Component
 * State machine for link key
 *
 * @brief Interface definition for generic link key state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for link key state machine.
 */

#ifndef _I_LINK_KEY_H_
#define _I_LINK_KEY_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class ILinkKeyRequest;
class IDeviceManager;
class IStateMachine;
class App2Bts_SetLinkKey;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class ILinkKey
{
public:
   /**
    * Destructor.
    */
   virtual ~ILinkKey() {}

   /**
    * Set specific instance (implementation).
    *
    * @param[in] instance: instance (implementation)
    */
   virtual void setInstance(IN ILinkKeyRequest* instance) = 0;

   /**
    * Set device manager interface.
    *
    * @param[in] deviceManager: device manager
    */
   virtual void setDeviceManager(IN IDeviceManager* deviceManager) = 0;

   /**
    * Get SM entry interface.
    *
    * @return = SM entry interface
    */
   virtual IStateMachine* getSmEntryInterface(void) = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidRequest(IN const App2Bts_SetLinkKey& request) const = 0;

   /**
    * Set link key.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool setLinkKey(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetLinkKey& request) = 0;
};

} //btstackif

#endif //_I_LINK_KEY_H_
