/**
 * @file Inquiry.h
 *
 * @par SW-Component
 * State machine for inquiry
 *
 * @brief Implementation of generic inquiry state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of generic inquiry state machine.
 */

#ifndef _INQUIRY_H_
#define _INQUIRY_H_

#include "StateMachine.h"
#include "IInquiry.h"
#include "IInquiryCallback.h"
//#include "IExtendedTimeoutHandler.h"

#include <set>

namespace btstackif {

/**
 * Inquiry class.
 */
class Inquiry : public StateMachine, public IInquiry, public IInquiryCallback/*, public IExtendedTimeoutHandler*/
{
public:
   /**
    * Default constructor.
    */
   Inquiry();

   /**
    * Destructor.
    */
   virtual ~Inquiry();

   virtual void reset(void);

   virtual void forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void setInstance(IN IInquiryRequest* instance);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setDeviceManagerIf(IN IDeviceManager* deviceManager);

   virtual void setTimerPoolIf(IN ITimerPool* timerPool);

   virtual IStateMachine* getSmEntryInterface(void);

   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetDiscoveringStatus& request, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendStatusAndStartResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartDiscovery& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendStatusAndStopResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StopDiscovery& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const;

   virtual bool isValidGetRequest(IN const App2Bts_GetDiscoveringStatus& request) const;

   virtual bool isValidStartRequest(IN const App2Bts_StartDiscovery& request) const;

   virtual bool isValidStopRequest(IN const App2Bts_StopDiscovery& request) const;

   virtual bool startDiscovery(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartDiscovery& request);

   virtual bool stopDiscovery(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StopDiscovery& request);

   virtual void setStopRequested(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void triggerStopDiscovery(IN const bool highPrio = true);

   virtual bool isDiscoveryOngoing(void) const;

   virtual bool isRequestOngoing(void) const;

   virtual void startDiscoveryResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSRequestResult result);

   virtual void stopDiscoveryResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSRequestResult result);

   virtual void updateStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoveringStatus status);

   virtual void deviceFound(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoveredDeviceInfo& device);

   virtual App2Bts_BaseMessage* getApp2BtsWorkingMessage(void);

   virtual void registerObserver(IN IInquiryObserver* observer);

   // virtual void handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId);

private:
   IInquiryRequest* _requestIf; /**< request interface */
   IBasicControl* _controlIf; /**< basic control interface */
   IDeviceManager* _deviceManagerIf; /**< device manager interface */
   ITimerPool* _timerPoolIf; /**< timer pool interface */
   BTSApp2BtsMessageRequestItem _requestItem; /**< request item */
   bool _discovering; /**< discovering status */
   bool _ongoing; /**< flag for ongoing request (start or stop) */
   bool _pending; /**< flag for pending result (start or stop) */
   bool _stopRequested; /**< flag for requested stop */
   bool _internalStop; /**< flag for internal stop */
   ::std::set< BTSBDAddress > _deviceList; /**< list with found devices */
   ::std::set< IInquiryObserver* > _observerList; /**< observer list */

   void createDiscoveringStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSDiscoveringStatus status) const;

   void createStartDiscoveryResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSRequestResult result) const;

   void createStopDiscoveryResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSRequestResult result) const;

   void createDiscoveredDeviceFoundMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSDiscoveredDeviceInfo& device) const;

   void handleActionFinished(OUT BTSHandleIpc2BtsMessageItem& messageItem);

   inline BTSDiscoveringStatus getStatus(void) const { if(true == _discovering) { return BTS_DISCOVERING_ON; } else { return BTS_DISCOVERING_OFF; } }

   void checkForStartInquiryCompleted(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem);

   void checkForStopInquiryCompleted(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem);

   void handleRequestCompleted(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool sendStatusToAll, IN const bool start);

   void informObservers(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoveringStatus status) const;

   void checkForSendingStop(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);
};

} //btstackif

#endif //_INQUIRY_H_
