/**
 * @file IInquiryCallback.h
 *
 * @par SW-Component
 * State machine for inquiry
 *
 * @brief Callback interface definition for inquiry state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the callback interface to be used by specific inquiry state machine.
 *          Callback interface must be provided by generic inquiry state machine.
 */

#ifndef _I_INQUIRY_CALLBACK_H_
#define _I_INQUIRY_CALLBACK_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;
class App2Bts_BaseMessage;

/**
 * Interface definition.
 */
class IInquiryCallback
{
public:
   /**
    * Destructor.
    */
   virtual ~IInquiryCallback() {}

   /**
    * Result for starting discovery.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result
    */
   virtual void startDiscoveryResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSRequestResult result) = 0;

   /**
    * Result for stopping discovery.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result
    */
   virtual void stopDiscoveryResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSRequestResult result) = 0;

   /**
    * Update discovering status.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] status: status
    */
   virtual void updateStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoveringStatus status) = 0;

   /**
    * Indicate found device.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] status: status
    */
   virtual void deviceFound(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoveredDeviceInfo& device) = 0;

   /**
    * Get App2Bts message.
    *
    * @return = App2Bts message if found else NULL
    */
   virtual App2Bts_BaseMessage* getApp2BtsWorkingMessage(void) = 0;
};

} //btstackif

#endif //_I_INQUIRY_CALLBACK_H_
