/**
 * @file HfpDeviceCapabilities.cpp
 *
 * @par SW-Component
 * State machine for HFP device capabilities
 *
 * @brief Implementation of generic HFP device capabilities state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic HFP device capabilities state machine.
 */

#include "HfpDeviceCapabilities.h"
#include "IHfpDeviceCapabilitiesRequest.h"
#include "IBasicControl.h"
#include "FwErrmemPrint.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/HfpDeviceCapabilities.cpp.trc.h"
#endif
#endif

namespace btstackif {

HfpDeviceCapabilities::HfpDeviceCapabilities() :
_requestIf(0),
_controlIf(0),
_managerServiceAvailable(false),
_deviceList()
{
}

HfpDeviceCapabilities::~HfpDeviceCapabilities()
{
   _requestIf = 0;
   _controlIf = 0;
}

void HfpDeviceCapabilities::reset(void)
{
   StateMachine::reset();
   // keep _requestIf
   // keep _controlIf
   _managerServiceAvailable = false;
   _deviceList.clear();

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);
   _requestIf->reset();
}

void HfpDeviceCapabilities::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   // reset control data
   reset();
}

void HfpDeviceCapabilities::setInstance(IN IHfpDeviceCapabilitiesRequest* instance)
{
   _requestIf = instance;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setCallback(this);
}

void HfpDeviceCapabilities::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_ERRMEM_ASSERT(0 != _controlIf);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setControlIf(_controlIf);
}

IStateMachine* HfpDeviceCapabilities::getSmEntryInterface(void)
{
   return this;
}

void HfpDeviceCapabilities::getModemsResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSRequestResult result)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(result);

   // all related information was received before => no further action necessary
   // no retry

   // ensure that any App2Bts message gets deleted
   messageItem.deleteMessage = true;
}

void HfpDeviceCapabilities::getModemPropertiesResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(address);
   (void)(result);

   // all related information was received before => no further action necessary
   // no retry

   // ensure that any App2Bts message gets deleted
   messageItem.deleteMessage = true;
}

void HfpDeviceCapabilities::getHandsfreePropertiesResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(address);
   (void)(result);

   // all related information was received before => no further action necessary
   // no retry

   // ensure that any App2Bts message gets deleted
   messageItem.deleteMessage = true;
}

void HfpDeviceCapabilities::updateHandsfreeFeatures(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSHandsfreeAgFeatures& features)
{
   (void)(bts2IpcMsgList);
   (void)(messageItem);

   HfpDeviceCapabilitiesData& entry = checkDeviceList(address);

   // remember whether update is necessary
   bool update(false == entry.ringtoneDataAvailable.getBit(HfpDeviceCapabilitiesData::UPDATE_DONE));

   // check for new status
   if(entry.ringtoneSupported != features.getBit(BTS_HFP_AG_FEATURE_INBAND_RING))
   {
      entry.ringtoneSupported = features.getBit(BTS_HFP_AG_FEATURE_INBAND_RING);
      update = true;
   }

   // remember receive of supported information
   entry.ringtoneDataAvailable.setBit(HfpDeviceCapabilitiesData::SUPPORTED_RECEIVED);

   // update status if necessary
   if((true == update) && (true == entry.ringtoneDataAvailable.getBit(HfpDeviceCapabilitiesData::ENABLED_RECEIVED)))
   {
      updateStatus(bts2AppMsgList, address, entry);
      entry.ringtoneDataAvailable.setBit(HfpDeviceCapabilitiesData::UPDATE_DONE);
   }
}

void HfpDeviceCapabilities::updateHandsfreeInbandRinging(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const bool inbandRingingEnabled)
{
   (void)(bts2IpcMsgList);
   (void)(messageItem);

   HfpDeviceCapabilitiesData& entry = checkDeviceList(address);

   // remember whether update is necessary
   bool update(false == entry.ringtoneDataAvailable.getBit(HfpDeviceCapabilitiesData::UPDATE_DONE));

   // check for new status
   if(entry.ringtoneEnabled != inbandRingingEnabled)
   {
      entry.ringtoneEnabled = inbandRingingEnabled;
      update = true;
   }

   // remember receive of enabled information
   entry.ringtoneDataAvailable.setBit(HfpDeviceCapabilitiesData::ENABLED_RECEIVED);

   // update status if necessary
   if((true == update) && (true == entry.ringtoneDataAvailable.getBit(HfpDeviceCapabilitiesData::SUPPORTED_RECEIVED)))
   {
      updateStatus(bts2AppMsgList, address, entry);
      entry.ringtoneDataAvailable.setBit(HfpDeviceCapabilitiesData::UPDATE_DONE);
   }
}

void HfpDeviceCapabilities::indicateManagerServiceAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool available)
{
   (void)(messageItem);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   // check for change from unavailable to available and vice versa
   if(_managerServiceAvailable == available)
   {
      return;
   }

   // store new value
   _managerServiceAvailable = available;

   // request modems if service is available
   if(true == _managerServiceAvailable)
   {
      // HINT: as part of connection management it is not necessary to request modems because no HFP connection exists during this moment
      // but we will do it because it is needed as improvement for test runner as well as for unit testing
      _requestIf->getModems(bts2IpcMsgList, bts2AppMsgList);
   }
}

void HfpDeviceCapabilities::modemAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   HfpDeviceCapabilitiesData& entry = checkDeviceList(address);

   if(false == entry.modemAdded)
   {
      ETG_TRACE_USR2((" modemAdded: address=%s", address.c_str()));

      entry.modemAdded = true;

      // reset data
      entry.modemAvailable = false;
      entry.hfInterfaceAdded = false;
      entry.hfInterfaceAvailable = false;
      entry.ringtoneSupported = false;
      entry.ringtoneEnabled = false;
      entry.ringtoneDataAvailable.setData(0);
   }
}

void HfpDeviceCapabilities::modemRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   // in case of modem removed the modem is no longer available
   modemUnavailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address);

   ::std::map< BTSBDAddress, HfpDeviceCapabilitiesData >::iterator it = _deviceList.find(address);
   if(_deviceList.end() != it)
   {
      _deviceList.erase(it);

      ETG_TRACE_USR2((" modemRemoved: _deviceList.size()=%u", _deviceList.size()));
   }
}

void HfpDeviceCapabilities::modemAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   (void)(messageItem);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   HfpDeviceCapabilitiesData& entry = checkDeviceList(address);

   if(false == entry.modemAvailable)
   {
      ETG_TRACE_USR2((" modemAvailable: address=%s", address.c_str()));

      entry.modemAvailable = true;

      // request modem properties
      _requestIf->getModemProperties(bts2IpcMsgList, bts2AppMsgList, address);
   }
}

void HfpDeviceCapabilities::modemUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   HfpDeviceCapabilitiesData& entry = checkDeviceList(address);

   if(true == entry.modemAvailable)
   {
      ETG_TRACE_USR2((" modemUnavailable: address=%s", address.c_str()));

      entry.modemAvailable = false;
   }
}

void HfpDeviceCapabilities::handsfreeInterfaceAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   HfpDeviceCapabilitiesData& entry = checkDeviceList(address);

   if(false == entry.hfInterfaceAdded)
   {
      ETG_TRACE_USR2((" handsfreeInterfaceAdded: address=%s", address.c_str()));

      entry.hfInterfaceAdded = true;

      // reset flags to force status update
      entry.ringtoneDataAvailable.setData(0);
   }
}

void HfpDeviceCapabilities::handsfreeInterfaceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   // in case of HF interface removed the HF interface is no longer available
   handsfreeInterfaceUnavailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address);

   HfpDeviceCapabilitiesData& entry = checkDeviceList(address);

   if(true == entry.hfInterfaceAdded)
   {
      ETG_TRACE_USR2((" handsfreeInterfaceRemoved: address=%s", address.c_str()));

      entry.hfInterfaceAdded = false;

      // do not reset flags
   }
}

void HfpDeviceCapabilities::handsfreeInterfaceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   (void)(messageItem);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   HfpDeviceCapabilitiesData& entry = checkDeviceList(address);

   if(false == entry.hfInterfaceAvailable)
   {
      ETG_TRACE_USR2((" handsfreeInterfaceAvailable: address=%s", address.c_str()));

      entry.hfInterfaceAvailable = true;

      // request hands-free properties
      // HINT: we need an update of hands-free properties
      _requestIf->getHandsfreeProperties(bts2IpcMsgList, bts2AppMsgList, address);
   }
}

void HfpDeviceCapabilities::handsfreeInterfaceUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   HfpDeviceCapabilitiesData& entry = checkDeviceList(address);

   if(true == entry.hfInterfaceAvailable)
   {
      ETG_TRACE_USR2((" handsfreeInterfaceUnavailable: address=%s", address.c_str()));

      entry.hfInterfaceAvailable = false;
   }
}

HfpDeviceCapabilitiesData& HfpDeviceCapabilities::checkDeviceList(IN const BTSBDAddress& address)
{
   ::std::map< BTSBDAddress, HfpDeviceCapabilitiesData >::iterator it = _deviceList.find(address);
   if(_deviceList.end() == it)
   {
      // add new entry
      ETG_TRACE_USR2((" checkDeviceList: _deviceList.size()=%u", (1 + _deviceList.size())));
      return _deviceList[address];
   }
   else
   {
      // continue with existing entry
      return it->second;
   }
}


void HfpDeviceCapabilities::updateStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const HfpDeviceCapabilitiesData& entry) const
{
   Bts2App_DeviceCapabilities* msg = ptrNew_Bts2App_DeviceCapabilities();
   if(0 != msg)
   {
      msg->setUser(0); // send status message to all
      msg->setSessionHandle(0); // send status message to all
      msg->setBDAddress(address);
      BTSDeviceCapabilities& capa = msg->getDeviceCapabilitiesMutable();
      capa.inbandRingtoneSupported = (true == entry.ringtoneSupported) && (true == entry.ringtoneEnabled);

      bts2AppMsgList.push_back(msg);
   }
}

} //btstackif
