/**
 * @file IDeviceManagerCallback.h
 *
 * @par SW-Component
 * State machine for device manager
 *
 * @brief Callback interface definition for device manager state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the callback interface to be used by specific device manager state machine.
 *          Callback interface must be provided by generic device manager state machine.
 */

#ifndef _I_DEVICE_MANAGER_CALLBACK_H_
#define _I_DEVICE_MANAGER_CALLBACK_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;
class App2Bts_BaseMessage;
class DeviceManagerData;

/**
 * Interface definition.
 */
class IDeviceManagerCallback
{
public:
   /**
    * Destructor.
    */
   virtual ~IDeviceManagerCallback() {}

   /**
    * Result for disconnect device.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] result: result
    */
   virtual void disconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result) = 0;

   /**
    * Result for create device.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] result: result
    */
   virtual void createResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result) = 0;

   /**
    * Result for remove device.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result
    */
   virtual void removeResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSRequestResult result) = 0;

   /**
    * Update ACL status of device connection.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] status: status
    * @param[in] reason: disconnect reason
    */
   virtual void updateAclStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSConnectionStatus status, IN const BTSDisconnectReason reason) = 0;

   /**
    * Update device name.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] name: name
    */
   virtual void updateName(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSDeviceName& name) = 0;

   /**
    * Update device COD.
    *
    * @param[in] address: device address
    * @param[in] cod: COD
    */
   virtual void updateCod(IN const BTSBDAddress& address, IN const BTSCod cod) = 0;

   /**
    * Update role.
    *
    * @param[in] address: device address
    * @param[in] master: master/slave flag
    */
   virtual void updateRole(IN const BTSBDAddress& address, IN const bool master) = 0;

   /**
    * Update control pause.
    *
    * @param[in] address: device address
    * @param[in] enableAvpPause: enable avp pause flag
    */
   virtual void updateEnableAvpPause(IN const BTSBDAddress& address, IN const bool enableAvpPause) = 0;

   /**
    * Update power mode.
    *
    * @param[in] address: device address
    * @param[in] mode: power mode
    */
   virtual void updatePowerMode(IN const BTSBDAddress& address, IN const BTSBtPowerMode mode) = 0;

   /**
    * Signal for device added.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    */
   virtual void deviceAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address) = 0;

   /**
    * Signal for device removed.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    */
   virtual void deviceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address) = 0;

   /**
    * Signal for device available.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    */
   virtual void deviceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address) = 0;

   /**
    * Signal for device unavailable.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    */
   virtual void deviceUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address) = 0;

   /**
    * Get App2Bts message.
    *
    * @param[in] address: device address
    *
    * @return = App2Bts message if found else NULL
    */
   virtual App2Bts_BaseMessage* getApp2BtsWorkingMessage(IN const BTSBDAddress& address) = 0;

   /**
    * Get device entry.
    *
    * @param[in] address: device address
    *
    * @return = device entry
    */
   virtual DeviceManagerData& getEntry(IN const BTSBDAddress& address) = 0;
};

} //btstackif

#endif //_I_DEVICE_MANAGER_CALLBACK_H_
