/**
 * @file IDeviceManager.h
 *
 * @par SW-Component
 * State machine for device manager
 *
 * @brief Interface definition for generic device manager state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for device manager state machine.
 */

#ifndef _I_DEVICE_MANAGER_H_
#define _I_DEVICE_MANAGER_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IDeviceManagerRequest;
class IBasicControl;
class ITimerPool;
class IStateMachine;
class App2Bts_GetDeviceConnectionStatus;
class App2Bts_DisconnectDevice;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;
class IDeviceObserver;

/**
 * Interface definition.
 */
class IDeviceManager
{
public:
   /**
    * Destructor.
    */
   virtual ~IDeviceManager() {}

   /**
    * Set specific instance (implementation).
    *
    * @param[in] instance: instance (implementation)
    */
   virtual void setInstance(IN IDeviceManagerRequest* instance) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set timer pool interface.
    *
    * @param[in] timerPool: timer pool interface
    */
   virtual void setTimerPoolIf(IN ITimerPool* timerPool) = 0;

   /**
    * Get SM entry interface.
    *
    * @return = SM entry interface
    */
   virtual IStateMachine* getSmEntryInterface(void) = 0;

   /**
    * Send current status for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] user: user
    * @param[in] handle: handle
    * @param[in] statusCode: status code
    */
   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Send current status for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] statusCode: status code
    */
   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetDeviceConnectionStatus& request, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Send status and result for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] sendStatusToAll: flag for sending status to all or not
    * @param[in] resultCode: result code
    * @param[in] statusCode: status code
    */
   virtual void sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_DisconnectDevice& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidGetRequest(IN const App2Bts_GetDeviceConnectionStatus& request) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidDisconnectRequest(IN const App2Bts_DisconnectDevice& request) const = 0;

   /**
    * Disconnect device.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool disconnectDevice(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_DisconnectDevice& request) = 0;

   /**
    * Create device. Check for added/available device/protocol shall be done before calling this function.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    */
   virtual void createDevice(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address) = 0;

   /**
    * Remove device. Check for added/available device/protocol shall be done before calling this function.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    */
   virtual void removeDevice(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address) = 0;

   /**
    * Remove all devices. Check for added/available device/protocol shall be done before calling this function.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    */
   virtual void removeAllDevices(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList) = 0;

   /**
    * Check if any device is connected.
    *
    * @param[in] includingAclLink: flag indicating if ACL link shall be considered too
    *
    * @return = true: any device connected,
    * @return = false: no device disconnected
    */
   virtual bool isAnyDeviceConnected(IN const bool includingAclLink = false) const = 0;

   /**
    * Check if device is connected.
    *
    * @param[in] address: device address
    * @param[in] includingAclLink: flag indicating if ACL link shall be considered too
    *
    * @return = true: connected,
    * @return = false: disconnected
    */
   virtual bool isDeviceConnected(IN const BTSBDAddress& address, IN const bool includingAclLink = false) const = 0;

   /**
    * Check for any ongoing disconnect device activity.
    *
    * @return = true: any disconnect ongoing,
    * @return = false: no disconnect ongoing
    */
   virtual bool isAnyDisconnectOngoing(void) const = 0;

   /**
    * Check for ongoing disconnect device activity.
    *
    * @param[in] address: device address
    *
    * @return = true: disconnect ongoing,
    * @return = false: no disconnect ongoing
    */
   virtual bool isDisconnectOngoing(IN const BTSBDAddress& address) const = 0;

   /**
    * Check if device is part of internal list.
    *
    * @param[in] address: device address
    *
    * @return = true: device is added,
    * @return = false: device is not added
    */
   virtual bool isDeviceAdded(IN const BTSBDAddress& address) const = 0;

   /**
    * Check if device is available (ready to use).
    *
    * @param[in] address: device address
    *
    * @return = true: device is available,
    * @return = false: device is not available
    */
   virtual bool isDeviceAvailable(IN const BTSBDAddress& address) const = 0;

   /**
    * Get name and COD.
    *
    * @param[out] name: name
    * @param[out] cod: COD
    * @param[in] address: device address
    */
   virtual void getDeviceData(OUT BTSDeviceName& name, OUT BTSCod& cod, IN const BTSBDAddress& address) = 0;

   /**
    * Get role (master/slave).
    *
    * @param[in] address: device address
    *
    * @return = true: master role,
    * @return = false: slave role
    */
   virtual bool getRole(IN const BTSBDAddress& address) const = 0;

   /**
    * Get EnableAvpPause.
    *
    * @param[in] address: device address
    *
    * @return = true/false
    */
   virtual bool getEnableAvpPause(IN const BTSBDAddress& address) const = 0;

   /**
    * Update any profile connected data.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: address
    * @param[in] connected: connected
    * @param[in] reason: disconnect reason
    */
   virtual void updateAnyProfileConnected(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const bool connected, IN const BTSDisconnectReason reason) = 0;

   /**
    * Indicate that a profile was disconnected by exception. Assumption is that complete device will be disconnected. Ensure that only one error memory entry is written.
    *
    * @param[in] address: device address
    */
   virtual void indicateDisconnectByException(IN const BTSBDAddress& address) = 0;

   /**
    * Register observer.
    *
    * @param[in] observer: observer
    */
   virtual void registerObserver(IN IDeviceObserver* observer) = 0;
};

} //btstackif

#endif //_I_DEVICE_MANAGER_H_
