/**
 * @file DeviceManagerData.h
 *
 * @par SW-Component
 * State machine for device manager
 *
 * @brief Implementation of device manager data.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of device manager data.
 */

#ifndef _DEVICE_MANAGER_DATA_H_
#define _DEVICE_MANAGER_DATA_H_

#include "BtStackInternalTypes.h"
#include "ExtendedTimerEntry.h"

namespace btstackif {

/**
 * Device manager data class.
 */
class DeviceManagerData
{
public:
   /**
    * Type definition for device data information bits.
    * #DeviceDataInfoBit#ignore#
    */
   enum DeviceDataInfoBit
   {
      DEVICE_ADDED = 0, /**< device entry was added */
      DEVICE_AVAILABLE, /**< device is available */
      ACL_STATUS_UPDATED, /**< ACL status was updated */
      NAME_UPDATED, /**< name was updated */
      COD_UPDATED, /**< COD was updated */
      ROLE_UPDATED, /**< role was updated */
      MODE_UPDATED, /**< power mode was updated */
      CONNECTED, /**< general connected flag */
      ACL_CONNECTED, /**< ACL connection status */
      ANY_PROFILE_CONNECTED, /**< flag for any connected profile */
      ONGOING, /**< disconnect ongoing flag */
      PENDING, /**< disconnect result pending flag */
      CREATING, /**< creating flag */
      REMOVING, /**< removing flag */
      ENABLEAVPPAUSE_UPDATED, /**< enable avp pause was updated */
      // -------------------------------------------------------------------------
      DEVICE_DATA_LAST /**< end marker, must be the last one */
   };

   /**
    * Type definition for device data information.
    */
   typedef BTSBitfield<unsigned int, DeviceDataInfoBit, DEVICE_DATA_LAST> DeviceDataInfo;

   /**
    * Default constructor.
    */
   DeviceManagerData();

   /**
    * Destructor.
    */
   virtual ~DeviceManagerData();

   /**
    * Copy constructor.
    *
    * @param[in] ref: reference of message to be copied
    */
   DeviceManagerData(const DeviceManagerData& ref);

   /**
    * Assignment operator.
    *
    * @param[in] ref: reference of message to be assigned
    */
   DeviceManagerData& operator=(const DeviceManagerData& ref);

   /**
    * Equality operator.
    *
    * @param[in] ref: reference of message to be compared
    */
   bool operator==(const DeviceManagerData& ref) const;

   /**
    * Inequality operator.
    *
    * @param[in] ref: reference of message to be compared
    */
   bool operator!=(const DeviceManagerData& ref) const;

   // member with public access
   BTSDeviceName name; /**< device name */
   BTSCod cod; /**< class of device */
   DeviceDataInfo info; /**< device data information */
   BTSDisconnectReason reason; /**< disconnect reason */
   unsigned int secondaryInfo; /**< secondary device data information */
   BTSApp2BtsMessageRequestItem requestItem; /**< request item */
   ExtendedTimerEntry timer; /**< timer */
   bool master; /**< flag indicating master/slave role of local BT node */
   BTSBtPowerMode powerMode; /**< power mode */
   bool disconnectedByException; /**< flag indicating that device was disconnected by exception */
   unsigned int retryMax; /**< maximum number of retries */
   unsigned int retryNmb; /**< current retry number */
   bool waitForDisconnectedUpdate; /**< flag indicating that timer was started due to pending disconnected update */
   bool enableAvpPause; /**< flag indicating pause BT streaming before AVP disconnection enabled/disabled */
   bool pauseBtStreaming; /**< flag indicating that pause BT streaming before AVP disconnection is requested */
};

} //btstackif

#endif //_DEVICE_MANAGER_DATA_H_
