/**
 * @file DeviceManager.h
 *
 * @par SW-Component
 * State machine for device manager
 *
 * @brief Implementation of generic device manager state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of generic device manager state machine.
 */

#ifndef _DEVICE_MANAGER_H_
#define _DEVICE_MANAGER_H_

#include "StateMachine.h"
#include "IDeviceManager.h"
#include "IDeviceManagerCallback.h"
#include "IExtendedTimeoutHandler.h"
#include "DeviceManagerData.h"

#include <set>

namespace btstackif {

/**
 * Device manager class.
 */
class DeviceManager : public StateMachine, public IDeviceManager, public IDeviceManagerCallback, public IExtendedTimeoutHandler
{
public:
   /**
    * Default constructor.
    */
   DeviceManager();

   /**
    * Destructor.
    */
   virtual ~DeviceManager();

   virtual void reset(void);

   virtual void forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void setInstance(IN IDeviceManagerRequest* instance);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setTimerPoolIf(IN ITimerPool* timerPool);

   virtual IStateMachine* getSmEntryInterface(void);

   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetDeviceConnectionStatus& request, IN const BTSCommonEnumClass statusCode) const;

   virtual void sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_DisconnectDevice& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const;

   virtual bool isValidGetRequest(IN const App2Bts_GetDeviceConnectionStatus& request) const;

   virtual bool isValidDisconnectRequest(IN const App2Bts_DisconnectDevice& request) const;

   virtual bool disconnectDevice(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_DisconnectDevice& request);

   virtual void createDevice(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   virtual void removeDevice(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   virtual void removeAllDevices(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual bool isAnyDeviceConnected(IN const bool includingAclLink = false) const;

   virtual bool isDeviceConnected(IN const BTSBDAddress& address, IN const bool includingAclLink = false) const;

   virtual bool isAnyDisconnectOngoing(void) const;

   virtual bool isDisconnectOngoing(IN const BTSBDAddress& address) const;

   virtual bool isDeviceAdded(IN const BTSBDAddress& address) const;

   virtual bool isDeviceAvailable(IN const BTSBDAddress& address) const;

   virtual void getDeviceData(OUT BTSDeviceName& name, OUT BTSCod& cod, IN const BTSBDAddress& address);

   virtual bool getRole(IN const BTSBDAddress& address) const;

   virtual bool getEnableAvpPause(IN const BTSBDAddress& address) const;

   virtual void updateAnyProfileConnected(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const bool connected, IN const BTSDisconnectReason reason);

   virtual void indicateDisconnectByException(IN const BTSBDAddress& address);

   virtual void registerObserver(IN IDeviceObserver* observer);

   virtual void disconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result);

   virtual void createResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result);

   virtual void removeResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSRequestResult result);

   virtual void updateAclStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSConnectionStatus status, IN const BTSDisconnectReason reason);

   virtual void updateName(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSDeviceName& name);

   virtual void updateCod(IN const BTSBDAddress& address, IN const BTSCod cod);

   virtual void updateRole(IN const BTSBDAddress& address, IN const bool master);

   virtual void updateEnableAvpPause(IN const BTSBDAddress& address, IN const bool enableAvpPause);

   virtual void updatePowerMode(IN const BTSBDAddress& address, IN const BTSBtPowerMode mode);

   virtual void deviceAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void deviceUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual App2Bts_BaseMessage* getApp2BtsWorkingMessage(IN const BTSBDAddress& address);

   virtual DeviceManagerData& getEntry(IN const BTSBDAddress& address);

   virtual void handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId);

private:
   IDeviceManagerRequest* _requestIf; /**< request interface */
   IBasicControl* _controlIf; /**< basic control interface */
   ITimerPool* _timerPoolIf; /**< timer pool interface */
   ::std::map< BTSBDAddress, DeviceManagerData > _deviceList; /**< map containing device information */
   ::std::set< IDeviceObserver* > _observerList; /**< observer list */
   const BTSTimeValue _timeoutUpdateDisconnected; /**< timeout value for disconnected update */
   const BTSTimeValue _retryTimeout; /**< timeout in ms before starting retry */
   const unsigned int _defaultRetryMax; /**< default maximum number of retries */

   DeviceManagerData& checkDeviceList(IN const BTSBDAddress& address);

   void checkConnectionStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN DeviceManagerData& entry, IN const bool oldConnected, IN const bool newConnected) const;

   void checkForDisconnectCompleted(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN DeviceManagerData& entry);

   void updateDeviceName(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSDeviceName& name, IN const bool available) const;

   void createDeviceConnectionStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSBDAddress& address, IN const BTSConnectionStatus status, IN const BTSDisconnectReason reason) const;

   void createDisconnectDeviceResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSBDAddress& address, IN const BTSRequestResult result) const;

   inline bool isConnected(IN const DeviceManagerData& entry) const { return ((true == entry.info.getBit(DeviceManagerData::CONNECTED)) || (true == entry.info.getBit(DeviceManagerData::ANY_PROFILE_CONNECTED))); }

   void handleActionFinished(OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSApp2BtsMessageCompareItem& item);

   App2Bts_BaseMessage* getWorkingMessage(IN const BTSApp2BtsMessageCompareItem& item);

   void startTimer(IN DeviceManagerData& entry, IN const BTSTimeValue timeout);

   void stopTimer(IN DeviceManagerData& entry) const;

   void releaseTimer(IN DeviceManagerData& entry) const;

   void sendConnectionStatusToObserver(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSStatusTransition aclTransition, IN const bool aclConnected, IN const BTSStatusTransition anyProfileTransition, IN const bool anyProfileConnected);
};

} //btstackif

#endif //_DEVICE_MANAGER_H_
