/**
 * @file DeviceIdServiceRecord.cpp
 *
 * @par SW-Component
 * State machine for device id service record
 *
 * @brief Implementation of generic device id service record state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic device id service record state machine.
 */

#include "DeviceIdServiceRecord.h"
#include "IDeviceIdServiceRecordRequest.h"
#include "IBasicControl.h"
#include "ISwitchBluetooth.h"
#include "IConfiguration.h"
#include "IConfigurationMaster.h"
#include "FwErrmemPrint.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/DeviceIdServiceRecord.cpp.trc.h"
#endif
#endif

namespace btstackif {

DeviceIdServiceRecord::DeviceIdServiceRecord() :
_requestIf(0),
_controlIf(0),
_switchBluetoothIf(0),
_configurationIf(0),
_configurationMaster(0),
_globalConfigActive(false),
_configRequestOngoing(false),
_configRequestDone(false),
_requestItem()
{
}

DeviceIdServiceRecord::~DeviceIdServiceRecord()
{
   _requestIf = 0;
   _controlIf = 0;
   _switchBluetoothIf = 0;
   _configurationIf = 0;
   _configurationMaster = 0;
}

void DeviceIdServiceRecord::reset(void)
{
   StateMachine::reset();
   // keep _requestIf
   // keep _controlIf
   // keep _switchBluetoothIf
   // keep _configurationIf
   // keep _configurationMaster
   _globalConfigActive = false;
   _configRequestOngoing = false;
   _configRequestDone = false;
   _requestItem.reset();

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);
   _requestIf->reset();
}

void DeviceIdServiceRecord::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   // reset control data
   reset();
}

void DeviceIdServiceRecord::setInstance(IN IDeviceIdServiceRecordRequest* instance)
{
   _requestIf = instance;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setCallback(this);
}

void DeviceIdServiceRecord::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_ERRMEM_ASSERT(0 != _controlIf);
}

void DeviceIdServiceRecord::setSwitchBluetoothIf(IN ISwitchBluetooth* switchBluetooth)
{
   _switchBluetoothIf = switchBluetooth;

   FW_ERRMEM_ASSERT(0 != _switchBluetoothIf);
}

void DeviceIdServiceRecord::setConfigurationIf(IN IConfiguration* configurationIf)
{
   _configurationIf = configurationIf;

   FW_ERRMEM_ASSERT(0 != _configurationIf);
}

IConfigurationClient* DeviceIdServiceRecord::getConfigurationClientHandler(void)
{
   return this;
}

IStateMachine* DeviceIdServiceRecord::getSmEntryInterface(void)
{
   return this;
}

bool DeviceIdServiceRecord::isValidSetRequest(IN const App2Bts_SetDeviceIdServiceRecord& request) const
{
   return isValidRecord(request.getDeviceIdServiceRecord());
}

bool DeviceIdServiceRecord::setDeviceIdServiceRecord(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetDeviceIdServiceRecord& request)
{
   ETG_TRACE_USR2((" setDeviceIdServiceRecord"));

   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_requestIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_configurationIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_switchBluetoothIf);

   if(false == isValidSetRequest(request))
   {
      FW_ERRMEM_ASSERT_ALWAYS();

      // no result
      return false;
   }

   // store new record
   _configurationIf->getConfiguration().deviceIdServiceRecord = request.getDeviceIdServiceRecord();

   // check if BT is on
   if(true == _switchBluetoothIf->getAppBtMode())
   {
      // BT is on => continue
      // store data and process request
      request.getCompareItem(_requestItem.item);
      _requestItem.user = request.getUser();
      _requestItem.handle = request.getSessionHandle();

      _requestIf->setDeviceIdServiceRecord(bts2IpcMsgList, bts2AppMsgList, _configurationIf->getConfiguration().deviceIdServiceRecord);

      return true;
   }
   else
   {
      // BT is off
      // mark configuration as not finished => forces re-writing record during next configuration sequence
      _configRequestDone = false;

      return false;
   }
}

void DeviceIdServiceRecord::setDeviceIdServiceRecordResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSRequestResult result)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(result);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_configurationMaster);

   // check if request is ongoing
   if(true == _configRequestOngoing)
   {
      // device id service record is only configured during stack configuration phase
      FW_ERRMEM_ASSERT(true == _globalConfigActive);

      // mark as not ongoing
      _configRequestOngoing = false;
      // mark as finished
      _configRequestDone = true;
      // ignore result because there is no retry

#ifdef ENABLE_NEW_CONFIGURATION_HANDLING
      // inform configuration master to execute next step
      (void)_configurationMaster->doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
#endif
   }
   else if(App2BtsOC_SetDeviceIdServiceRecord == _requestItem.item.opCode)
   {
      // user request is ongoing => action is finished
      handleActionFinished(messageItem);

      // reset control data because action is finished
      _requestItem.reset();
   }
}

void DeviceIdServiceRecord::setConfigurationMasterIf(IN IConfigurationMaster* master)
{
   _configurationMaster = master;

   FW_ERRMEM_ASSERT(0 != _configurationMaster);
}

void DeviceIdServiceRecord::startGlobalConfiguration(void)
{
   _globalConfigActive = true;
   _configRequestOngoing = false;
   // do not reset _configRequestDone because this configuration shall be done only once
}

void DeviceIdServiceRecord::stopGlobalConfiguration(void)
{
   _globalConfigActive = false;
   _configRequestOngoing = false;
}

bool DeviceIdServiceRecord::isSingleConfigurationOngoing(void) const
{
   if(false == _globalConfigActive)
   {
      return false;
   }

   // check if call to set device id service record is ongoing
   return _configRequestOngoing;
}

bool DeviceIdServiceRecord::setSingleConfiguration(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool beforeBtOn, IN const bool errorOccurred)
{
   (void)(messageItem);
   (void)(beforeBtOn);

   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_requestIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_configurationIf);

   if(false == _globalConfigActive)
   {
      return false;
   }

   // ETG_TRACE_USR3((" DeviceIdServiceRecord::setSingleConfiguration(): beforeBtOn=%d _configRequestOngoing=%d errorOccurred=%d _configRequestDone=%d", beforeBtOn, _configRequestOngoing, errorOccurred, _configRequestDone));

   if(true == _configRequestOngoing)
   {
      // configuration request is ongoing => wait for result
      return true;
   }

   if(true == errorOccurred)
   {
      // error occurred => do not continue
      return false;
   }

   if(false == _configRequestDone)
   {
      // configure device id service record

      if(false == isValidRecord(_configurationIf->getConfiguration().deviceIdServiceRecord))
      {
         FW_ERRMEM_ASSERT_ALWAYS();

         // try during next configuration sequence again => keep _configRequestDone

         return false;
      }
      else
      {
         _requestIf->setDeviceIdServiceRecord(bts2IpcMsgList, bts2AppMsgList, _configurationIf->getConfiguration().deviceIdServiceRecord);

         // mark as ongoing
         _configRequestOngoing = true;

         return true;
      }
   }

   return false;
}

bool DeviceIdServiceRecord::isValidRecord(IN const BTSDeviceIdServiceRecord& record) const
{
   if((0 == record.vendorID) ||
      (0 == record.productID) ||
      (0 == record.version) ||
      (0 == record.vendorIDSource))
   {
      return false;
   }
   else
   {
      return true;
   }
}

void DeviceIdServiceRecord::handleActionFinished(OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   messageItem.item = _requestItem.item;
   messageItem.item.opCode = App2BtsOC_None; // to force default handling for end of request sequence
   messageItem.deleteMessage = true;
   findApp2BtsWorkingMessage(messageItem);
}

void DeviceIdServiceRecord::findApp2BtsWorkingMessage(OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   if(0 == messageItem.message)
   {
      messageItem.message = getApp2BtsWorkingMessage();
   }
   FW_ERRMEM_ASSERT(0 != messageItem.message);
}

App2Bts_BaseMessage* DeviceIdServiceRecord::getApp2BtsWorkingMessage(void)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_NULL(_controlIf);

   return _controlIf->findApp2BtsWorkingMessageWrapper(_requestItem.item);
}

} //btstackif
