/**
 * @file IDeviceIdServiceRecord.h
 *
 * @par SW-Component
 * State machine for setting device id service record
 *
 * @brief Interface definition for generic setting device id service record state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for setting device id service record state machine.
 */

#ifndef _I_DEVICE_ID_SERVICE_RECORD_H_
#define _I_DEVICE_ID_SERVICE_RECORD_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IDeviceIdServiceRecordRequest;
class IBasicControl;
class ISwitchBluetooth;
class IConfiguration;
class IConfigurationClient;
class IStateMachine;
class App2Bts_SetDeviceIdServiceRecord;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class IDeviceIdServiceRecord
{
public:
   /**
    * Destructor.
    */
   virtual ~IDeviceIdServiceRecord() {}

   /**
    * Set specific instance (implementation).
    *
    * @param[in] instance: instance (implementation)
    */
   virtual void setInstance(IN IDeviceIdServiceRecordRequest* instance) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set switch Bluetooth interface.
    *
    * @param[in] switchBluetooth: witch Bluetooth interface
    */
   virtual void setSwitchBluetoothIf(IN ISwitchBluetooth* switchBluetooth) = 0;

   /**
    * Set configuration interface.
    *
    * @param[in] configurationIf: configuration interface
    */
   virtual void setConfigurationIf(IN IConfiguration* configurationIf) = 0;

   /**
    * Get configuration client handler.
    *
    * @return = configuration client handler
    */
   virtual IConfigurationClient* getConfigurationClientHandler(void) = 0;

   /**
    * Get SM entry interface.
    *
    * @return = SM entry interface
    */
   virtual IStateMachine* getSmEntryInterface(void) = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidSetRequest(IN const App2Bts_SetDeviceIdServiceRecord& request) const = 0;

   /**
    * Set device id service record.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool setDeviceIdServiceRecord(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetDeviceIdServiceRecord& request) = 0;
};

} //btstackif

#endif //_I_DEVICE_ID_SERVICE_RECORD_H_
