/**
 * @file ConnectionPriority.cpp
 *
 * @par SW-Component
 * State machine for connection priority
 *
 * @brief Implementation of generic connection priority state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic connection priority state machine.
 */

#include "ConnectionPriority.h"
#include "IBasicControl.h"
#include "IDeviceManager.h"
#include "IProtocolManager.h"
#include "IInquiry.h"
#include "IPairing.h"
#include "IServiceSearch.h"
#include "App2Bts_MessageWrapper.h"
#include "FwErrmemPrint.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ConnectionPriority.cpp.trc.h"
#endif
#endif

namespace btstackif {

ConnectionPriority::ConnectionPriority() :
_controlIf(0),
_deviceManagerIf(0),
_protocolManagerIf(0),
_inquiryIf(0),
_pairingIf(0),
_serviceSearchIf(0),
_priorityMode(PRIO_MODE_DEFAULT)
{
}

ConnectionPriority::~ConnectionPriority()
{
   _controlIf = 0;
   _deviceManagerIf = 0;
   _protocolManagerIf = 0;
   _inquiryIf = 0;
   _pairingIf = 0;
   _serviceSearchIf = 0;
}

void ConnectionPriority::reset(void)
{
   StateMachine::reset();
   // keep _controlIf
   // keep _deviceManagerIf
   // keep _protocolManagerIf
   // keep _inquiryIf
   // keep _pairingIf
   // keep _serviceSearchIf
   // keep _priorityMode
}

void ConnectionPriority::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   // reset control data
   reset();
}

void ConnectionPriority::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_ERRMEM_ASSERT(0 != _controlIf);
}

void ConnectionPriority::setDeviceManagerIf(IN IDeviceManager* deviceManagerIf)
{
   _deviceManagerIf = deviceManagerIf;

   FW_ERRMEM_ASSERT(0 != _deviceManagerIf);
}

void ConnectionPriority::setProtocolManagerIf(IN IProtocolManager* protocolManagerIf)
{
   _protocolManagerIf = protocolManagerIf;

   FW_ERRMEM_ASSERT(0 != _protocolManagerIf);
}

void ConnectionPriority::setInquiryIf(IN IInquiry* inquiryIf)
{
   _inquiryIf = inquiryIf;

   FW_ERRMEM_ASSERT(0 != _inquiryIf);
}

void ConnectionPriority::setPairingIf(IN IPairing* pairingIf)
{
   _pairingIf = pairingIf;

   FW_ERRMEM_ASSERT(0 != _pairingIf);
}

void ConnectionPriority::setServiceSearchIf(IN IServiceSearch* serviceSearchIf)
{
   _serviceSearchIf = serviceSearchIf;

   FW_ERRMEM_ASSERT(0 != _serviceSearchIf);
}

IStateMachine* ConnectionPriority::getSmEntryInterface(void)
{
   return this;
}

BTSApp2BtsMessageHandlingType ConnectionPriority::disconnectDevice(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_DisconnectDevice& request)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_ERROR(_deviceManagerIf, BTS_APP2BTS_DELETE);
   FW_ERRMEM_IF_NULL_PTR_RETURN_ERROR(_protocolManagerIf, BTS_APP2BTS_DELETE);
   FW_ERRMEM_IF_NULL_PTR_RETURN_ERROR(_serviceSearchIf, BTS_APP2BTS_DELETE);
   FW_ERRMEM_IF_NULL_PTR_RETURN_ERROR(_pairingIf, BTS_APP2BTS_DELETE);

   if(false == _deviceManagerIf->isValidDisconnectRequest(request))
   {
      (void)_deviceManagerIf->disconnectDevice(bts2IpcMsgList, bts2AppMsgList, request);
      // no further processing necessary
      return BTS_APP2BTS_DELETE;
   }

   bool pushRequestToWaitingQueue(false);

   // check whether disconnect for specific address is ongoing
   if(true == _deviceManagerIf->isDisconnectOngoing(request.getBDAddress()))
   {
      pushRequestToWaitingQueue = true;
   }
   // next if clause disabled because it causes wrong behavior
#if 0
   // check for request for specific address in waiting queue (if there would be a request in working queue the isDisconnectOngoing flag would be true)
   else if(true == isDisconnectRequestInWaitingQueue(request.getBDAddress()))
   {
      pushRequestToWaitingQueue = true;
   }
#endif
   else
   {
      // check further priorities
      if(true == checkDisconnectDevicePriority(request))
      {
         // check whether specific address is disconnected
         if(false == _deviceManagerIf->isDeviceConnected(request.getBDAddress(), true))
         {
            // send answer directly because device is already disconnected
            _deviceManagerIf->sendStatusAndResult(bts2AppMsgList, request, false, (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);
            // no further processing necessary
            return BTS_APP2BTS_DELETE;
         }
         else
         {
            return checkRequestResult(_deviceManagerIf->disconnectDevice(bts2IpcMsgList, bts2AppMsgList, request));
         }
      }
      else
      {
         pushRequestToWaitingQueue = true;
      }
   }

   if(true == pushRequestToWaitingQueue)
   {
      _protocolManagerIf->setRequestedConnectionState(bts2IpcMsgList, bts2AppMsgList, request.getBDAddress(), false);
      _serviceSearchIf->cancelSearch(bts2IpcMsgList, bts2AppMsgList, request.getBDAddress());
      _pairingIf->cancelPairing(bts2IpcMsgList, bts2AppMsgList, request.getBDAddress());
      ETG_TRACE_USR3((" disconnectDevice(): App2Bts 0x%04X pushed to waiting queue", request.getTraceOpCode()));
      return BTS_APP2BTS_PUSH_TO_WAITING;
   }

   return BTS_APP2BTS_DELETE;
}

BTSApp2BtsMessageHandlingType ConnectionPriority::connectProtocol(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_ConnectProtocol& request)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_ERROR(_protocolManagerIf, BTS_APP2BTS_DELETE);

   if((false == _protocolManagerIf->isProtocolEnabled(request.getProtocolId())) ||
      (false == _protocolManagerIf->isValidConnectRequest(request)))
   {
      (void)_protocolManagerIf->connectProtocol(bts2IpcMsgList, bts2AppMsgList, request);
      // no further processing necessary
      return BTS_APP2BTS_DELETE;
   }

   bool storeRequestedStateAndPushRequestToWaitingQueue(false);

   // check whether connect or disconnect (local or remote) for specific address + protocol + UUID + MAS instance is ongoing
   if(true == _protocolManagerIf->isProtocolConnectDisconnectOngoing(request.getBDAddress(), request.getProtocolId(), request.getUuid(), request.getMasInstanceName()))
   {
      storeRequestedStateAndPushRequestToWaitingQueue = true;
   }
   // check for request for specific address + protocol + UUID + MAS instance in waiting queue (if there would be a request in working queue the isProtocolConnectDisconnectOngoing flag would be true)
   else if(true == isConnectDisconnectRequestInWaitingQueue(request.getBDAddress(), request.getProtocolId(), request.getUuid(), request.getMasInstanceName()))
   {
      storeRequestedStateAndPushRequestToWaitingQueue = true;
   }
   // check whether device disconnect to same address is in waiting queue
   else if(true == isDisconnectRequestInWaitingQueue(request.getBDAddress()))
   {
      storeRequestedStateAndPushRequestToWaitingQueue = true;
   }
   // check whether specific address + protocol + UUID + MAS instance is connected
   else if(true == _protocolManagerIf->isProtocolConnected(request.getBDAddress(), request.getProtocolId(), request.getUuid(), request.getMasInstanceName()))
   {
      // send answer directly because protocol is already connected
      _protocolManagerIf->sendStatusAndResult(bts2AppMsgList, request, false, (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);
      // no further processing necessary
      return BTS_APP2BTS_DELETE;
   }
   else
   {
      // check further priorities
      if(true == checkConnectProtocolPriority(request))
      {
         return checkRequestResult(_protocolManagerIf->connectProtocol(bts2IpcMsgList, bts2AppMsgList, request));
      }
      else
      {
         storeRequestedStateAndPushRequestToWaitingQueue = true;
      }
   }

   if(true == storeRequestedStateAndPushRequestToWaitingQueue)
   {
      _protocolManagerIf->setRequestedConnectionState(bts2IpcMsgList, bts2AppMsgList, request.getBDAddress(), request.getProtocolId(), request.getUuid(), request.getMasInstanceName(), true, request.getUser(), request.getSessionHandle());
      ETG_TRACE_USR3((" connectProtocol(): App2Bts 0x%04X pushed to waiting queue", request.getTraceOpCode()));
      return BTS_APP2BTS_PUSH_TO_WAITING;
   }

   return BTS_APP2BTS_DELETE;
}

BTSApp2BtsMessageHandlingType ConnectionPriority::disconnectProtocol(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_DisconnectProtocol& request)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_ERROR(_protocolManagerIf, BTS_APP2BTS_DELETE);

   if(false == _protocolManagerIf->isValidDisconnectRequest(request))
   {
      (void)_protocolManagerIf->disconnectProtocol(bts2IpcMsgList, bts2AppMsgList, request);
      // no further processing necessary
      return BTS_APP2BTS_DELETE;
   }

   bool storeRequestedStateAndPushRequestToWaitingQueue(false);

   // check whether connect or disconnect (local or remote) for specific address + protocol + UUID + MAS instance is ongoing
   if(true == _protocolManagerIf->isProtocolConnectDisconnectOngoing(request.getBDAddress(), request.getProtocolId(), request.getUuid(), request.getMasInstanceName()))
   {
      storeRequestedStateAndPushRequestToWaitingQueue = true;
   }
   // check for request for specific address + protocol + UUID + MAS instance in waiting queue (if there would be a request in working queue the isProtocolConnectDisconnectOngoing flag would be true)
   else if(true == isConnectDisconnectRequestInWaitingQueue(request.getBDAddress(), request.getProtocolId(), request.getUuid(), request.getMasInstanceName()))
   {
      storeRequestedStateAndPushRequestToWaitingQueue = true;
   }
   // check whether specific address + protocol + UUID + MAS instance is disconnected
   else if(false == _protocolManagerIf->isProtocolConnected(request.getBDAddress(), request.getProtocolId(), request.getUuid(), request.getMasInstanceName()))
   {
      // send answer directly because protocol is already disconnected
      _protocolManagerIf->sendStatusAndResult(bts2AppMsgList, request, false, (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);
      // no further processing necessary
      return BTS_APP2BTS_DELETE;
   }
   else
   {
      // check further priorities
      if(true == checkDisconnectProtocolPriority(request))
      {
         return checkRequestResult(_protocolManagerIf->disconnectProtocol(bts2IpcMsgList, bts2AppMsgList, request));
      }
      else
      {
         storeRequestedStateAndPushRequestToWaitingQueue = true;
      }
   }

   if(true == storeRequestedStateAndPushRequestToWaitingQueue)
   {
      _protocolManagerIf->setRequestedConnectionState(bts2IpcMsgList, bts2AppMsgList, request.getBDAddress(), request.getProtocolId(), request.getUuid(), request.getMasInstanceName(), false, request.getUser(), request.getSessionHandle());
      ETG_TRACE_USR3((" disconnectProtocol(): App2Bts 0x%04X pushed to waiting queue", request.getTraceOpCode()));
      return BTS_APP2BTS_PUSH_TO_WAITING;
   }

   return BTS_APP2BTS_DELETE;
}

BTSApp2BtsMessageHandlingType ConnectionPriority::startDiscovery(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartDiscovery& request)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_ERROR(_inquiryIf, BTS_APP2BTS_DELETE);

   if(false == _inquiryIf->isValidStartRequest(request))
   {
      (void)_inquiryIf->startDiscovery(bts2IpcMsgList, bts2AppMsgList, request);
      // no further processing necessary
      return BTS_APP2BTS_DELETE;
   }

   bool pushRequestToWaitingQueue(false);

   // check whether start or stop discovery request is ongoing
   if(true == _inquiryIf->isRequestOngoing())
   {
      pushRequestToWaitingQueue = true;
   }
   // check whether request is in waiting queue (if there would be a request in working queue the isRequestOngoing flag would be true)
   else if(true == isDiscoveryRequestInWaitingQueue())
   {
      pushRequestToWaitingQueue = true;
   }
   // check whether inquiry is ongoing
   else if(true == _inquiryIf->isDiscoveryOngoing())
   {
      // send answer directly because inquiry is already ongoing
      _inquiryIf->sendStatusAndStartResult(bts2AppMsgList, request, false, (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);
      // no further processing necessary
      return BTS_APP2BTS_DELETE;
   }
   else
   {
      // check further priorities
      if(true == checkStartDiscoveryPriority(request))
      {
         return checkRequestResult(_inquiryIf->startDiscovery(bts2IpcMsgList, bts2AppMsgList, request));
      }
      else
      {
         pushRequestToWaitingQueue = true;
      }
   }

   if(true == pushRequestToWaitingQueue)
   {
      ETG_TRACE_USR3((" startDiscovery(): App2Bts 0x%04X pushed to waiting queue", request.getTraceOpCode()));
      return BTS_APP2BTS_PUSH_TO_WAITING;
   }

   return BTS_APP2BTS_DELETE;
}

BTSApp2BtsMessageHandlingType ConnectionPriority::stopDiscovery(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StopDiscovery& request)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_ERROR(_inquiryIf, BTS_APP2BTS_DELETE);

   if(false == _inquiryIf->isValidStopRequest(request))
   {
      (void)_inquiryIf->stopDiscovery(bts2IpcMsgList, bts2AppMsgList, request);
      // no further processing necessary
      return BTS_APP2BTS_DELETE;
   }

   bool pushRequestToWaitingQueue(false);

   // check whether start or stop discovery request is ongoing
   if(true == _inquiryIf->isRequestOngoing())
   {
      pushRequestToWaitingQueue = true;
   }
   // check whether request is in waiting queue (if there would be a request in working queue the isRequestOngoing flag would be true)
   else if(true == isDiscoveryRequestInWaitingQueue())
   {
      pushRequestToWaitingQueue = true;
   }
   // check whether inquiry is stopped
   else if(false == _inquiryIf->isDiscoveryOngoing())
   {
      // send answer directly because inquiry is already stopped
      _inquiryIf->sendStatusAndStopResult(bts2AppMsgList, request, false, (BTSCommonEnumClass)BTS_REQ_SUCCESS, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);
      // no further processing necessary
      return BTS_APP2BTS_DELETE;
   }
   else
   {
      // check further priorities
      if(true == checkStopDiscoveryPriority(request))
      {
         return checkRequestResult(_inquiryIf->stopDiscovery(bts2IpcMsgList, bts2AppMsgList, request));
      }
      else
      {
         pushRequestToWaitingQueue = true;
      }
   }

   if(true == pushRequestToWaitingQueue)
   {
      _inquiryIf->setStopRequested(bts2IpcMsgList, bts2AppMsgList);
      ETG_TRACE_USR3((" stopDiscovery(): App2Bts 0x%04X pushed to waiting queue", request.getTraceOpCode()));
      return BTS_APP2BTS_PUSH_TO_WAITING;
   }

   return BTS_APP2BTS_DELETE;
}

BTSApp2BtsMessageHandlingType ConnectionPriority::startPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartPairing& request)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_ERROR(_pairingIf, BTS_APP2BTS_DELETE);
   FW_ERRMEM_IF_NULL_PTR_RETURN_ERROR(_inquiryIf, BTS_APP2BTS_DELETE);

   if(false == _pairingIf->isValidStartRequest(request))
   {
      (void)_pairingIf->startPairing(bts2IpcMsgList, bts2AppMsgList, request);
      // no further processing necessary
      return BTS_APP2BTS_DELETE;
   }

   bool pushRequestToWaitingQueue(false);

   // check whether pairing is ongoing
   if(true == _pairingIf->isPairingOngoing())
   {
      // pairing is already ongoing but we allow only 1 pairing at the same time => handle directly
      return checkRequestResult(_pairingIf->startPairing(bts2IpcMsgList, bts2AppMsgList, request));
   }
   // check whether request for specific address is in waiting queue (if there would be a request in working queue the isPairingOngoing flag would be true)
   else if(true == isPairingRequestInWaitingQueue(request.getBDAddress()))
   {
      pushRequestToWaitingQueue = true;
   }
   // check whether inquiry is somehow ongoing => we have to stop inquiry first
   else if((true == _inquiryIf->isDiscoveryOngoing()) || (true == _inquiryIf->isRequestOngoing()))
   {
      _inquiryIf->setStopRequested(bts2IpcMsgList, bts2AppMsgList);
      // push request to waiting queue
      pushRequestToWaitingQueue = true;
   }
   else
   {
      // check further priorities
      if(true == checkStartPairingPriority(request))
      {
         return checkRequestResult(_pairingIf->startPairing(bts2IpcMsgList, bts2AppMsgList, request));
      }
      else
      {
         pushRequestToWaitingQueue = true;
      }
   }

   if(true == pushRequestToWaitingQueue)
   {
      ETG_TRACE_USR3((" startPairing(): App2Bts 0x%04X pushed to waiting queue", request.getTraceOpCode()));
      return BTS_APP2BTS_PUSH_TO_WAITING;
   }

   return BTS_APP2BTS_DELETE;
}

BTSApp2BtsMessageHandlingType ConnectionPriority::cancelPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_CancelPairing& request)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_ERROR(_pairingIf, BTS_APP2BTS_DELETE);

   if(false == _pairingIf->isValidCancelRequest(request))
   {
      (void)_pairingIf->cancelPairing(bts2IpcMsgList, bts2AppMsgList, request);
      // no further processing necessary
      return BTS_APP2BTS_DELETE;
   }

   bool pushRequestToWaitingQueue(false);

   // check whether request for specific address is in waiting queue
   if(true == isPairingRequestInWaitingQueue(request.getBDAddress()))
   {
      pushRequestToWaitingQueue = true;
   }
   else
   {
      // store requested state only
      // request can be deleted now because no pending request
      // done below
   }

   (void)_pairingIf->cancelPairing(bts2IpcMsgList, bts2AppMsgList, request);

   if(true == pushRequestToWaitingQueue)
   {
      ETG_TRACE_USR3((" cancelPairing(): App2Bts 0x%04X pushed to waiting queue", request.getTraceOpCode()));
      return BTS_APP2BTS_PUSH_TO_WAITING;
   }

   return BTS_APP2BTS_DELETE;
}

BTSApp2BtsMessageHandlingType ConnectionPriority::startSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartRemoteServiceSearch& request)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_ERROR(_serviceSearchIf, BTS_APP2BTS_DELETE);

   if(false == _serviceSearchIf->isValidStartRequest(request))
   {
      (void)_serviceSearchIf->startSearch(bts2IpcMsgList, bts2AppMsgList, request);
      // no further processing necessary
      return BTS_APP2BTS_DELETE;
   }

   bool pushRequestToWaitingQueue(false);

   // check whether service search for specific address is ongoing (we allow only 1 service search to the same device at the same time)
   if(true == _serviceSearchIf->isSearchOngoing(request.getBDAddress()))
   {
      pushRequestToWaitingQueue = true;
   }
   // check for request for specific address + search type in waiting queue (if there would be a request in working queue the isSearchOngoing flag would be true)
   else if(true == isServiceSearchRequestInWaitingQueue(request.getBDAddress(), request.getSearchType()))
   {
      pushRequestToWaitingQueue = true;
   }
   else
   {
      // check further priorities
      if(true == checkStartRemoteServiceSearchPriority(request))
      {
         return checkRequestResult(_serviceSearchIf->startSearch(bts2IpcMsgList, bts2AppMsgList, request));
      }
      else
      {
         pushRequestToWaitingQueue = true;
      }
   }

   if(true == pushRequestToWaitingQueue)
   {
      ETG_TRACE_USR3((" startSearch(): App2Bts 0x%04X pushed to waiting queue", request.getTraceOpCode()));
      return BTS_APP2BTS_PUSH_TO_WAITING;
   }

   return BTS_APP2BTS_DELETE;
}

BTSApp2BtsMessageHandlingType ConnectionPriority::cancelSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_CancelRemoteServiceSearch& request)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_ERROR(_serviceSearchIf, BTS_APP2BTS_DELETE);

   if(false == _serviceSearchIf->isValidCancelRequest(request))
   {
      (void)_serviceSearchIf->cancelSearch(bts2IpcMsgList, bts2AppMsgList, request);
      // no further processing necessary
      return BTS_APP2BTS_DELETE;
   }

   bool pushRequestToWaitingQueue(false);

   // check for request for specific address + search type in waiting queue
   if(true == isServiceSearchRequestInWaitingQueue(request.getBDAddress(), request.getSearchType()))
   {
      pushRequestToWaitingQueue = true;
   }
   else
   {
      // store requested state only
      // request can be deleted now because no pending request
      // done below
   }

   (void)_serviceSearchIf->cancelSearch(bts2IpcMsgList, bts2AppMsgList, request);

   if(true == pushRequestToWaitingQueue)
   {
      ETG_TRACE_USR3((" cancelSearch(): App2Bts 0x%04X pushed to waiting queue", request.getTraceOpCode()));
      return BTS_APP2BTS_PUSH_TO_WAITING;
   }

   return BTS_APP2BTS_DELETE;
}

void ConnectionPriority::checkEndOfSequence(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool finished, IN const bool firstConnectFailed /*= false*/, IN const BTSBDAddress& address /*= BTSBDAddress()*/)
{
   (void)(bts2IpcMsgList);

   ETG_TRACE_USR2((" checkEndOfSequence(): finished=%d deleteMessage=%d item.opCode=%d item.deviceAddress=%20s firstConnectFailed=%d address=%s", finished, messageItem.deleteMessage, messageItem.item.opCode, messageItem.item.deviceAddress.c_str(), firstConnectFailed, address.c_str()));

   // following ACL actions were checked:
   // * start inquiry (discovery) => search for 'IInquiryGenivi& '
   // * stop inquiry (discovery) => search for 'IInquiryGenivi& '
   // * start service search => search for 'IServiceSearchGenivi& '
   // * cancel service search => search for 'IServiceSearchGenivi& '
   // * start pairing => search for 'IPairingGenivi& '
   // * cancel pairing => search for 'IPairingGenivi& '

   if(false == finished)
   {
      // either sequence ongoing or no sequence was ongoing before processing Ipc2Bts message (e.g. spontaneous status update)
      // somehow no end of any sequence

      // *BUT:*
      // it might be that a part of a full sequence is finished and an App2Bts message shall be deleted
      // in this case the messageItem is set properly
      // but we have to avoid checking the waiting queue (note: waiting queue shall only be checked after the sequence is finished)
      messageItem.item.opCode = App2BtsOC_Ignore;
   }
   else
   {
      // sequence is finished
      // this function is also called as part of configuration sequence => avoid execution for wrong opcode
      if(false == isConnectionRelatedOpCode(messageItem.item.opCode))
      {
         // ignore
      }
      else
      {
         // check if first protocol connect failed
         if(true == firstConnectFailed)
         {
            handleAclConnectFailed(bts2AppMsgList, address);
         }

         // check if we can continue with next ACL based request
         if(true == continueWithNextAclBasedAction())
         {
            // check next one
            // compare item already set
         }
         else
         {
            // do not continue
            messageItem.item.opCode = App2BtsOC_Ignore;
         }
      }
   }
}

void ConnectionPriority::getSimilarOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, IN const App2Bts_BaseMessage& message) const
{
   switch(_priorityMode)
   {
      case PRIO_MODE_PARALLEL_PIM_MSG:
         // implement if needed
         break;
      case PRIO_MODE_DEFAULT:
      case PRIO_MODE_LAST:
      default:
         getSimilarOpCodesPrioModeDefault(itemList, message);
         break;
   }
}

void ConnectionPriority::getMatchingOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& highPrioItemList, IN const App2Bts_BaseMessage& message) const
{
   switch(_priorityMode)
   {
      case PRIO_MODE_PARALLEL_PIM_MSG:
         // implement if needed
         break;
      case PRIO_MODE_DEFAULT:
      case PRIO_MODE_LAST:
      default:
         getMatchingOpCodesPrioModeDefault(itemList, highPrioItemList, message);
         break;
   }
}

bool ConnectionPriority::checkDisconnectDevicePriority(IN const App2Bts_DisconnectDevice& request) const
{
   (void)(request);

   // following was checked before:
   // * invalid request
   // * disconnect for given device address ongoing
   // * disconnect request for given device address in waiting queue
   // * given device disconnected

   // further conditions have to be checked now depending on current priority mode

   bool allowed(false);

   switch(_priorityMode)
   {
      case PRIO_MODE_PARALLEL_PIM_MSG:
         // implement if needed
         break;
      case PRIO_MODE_DEFAULT:
      case PRIO_MODE_LAST:
      default:
         allowed = checkPrioModeDefault(App2BtsOC_DisconnectDevice);
         break;
   }

   return allowed;
}

bool ConnectionPriority::checkConnectProtocolPriority(IN const App2Bts_ConnectProtocol& request) const
{
   (void)(request);

   // following was checked before:
   // * protocol disabled on local side
   // * invalid request
   // * protocol connect or disconnect to given protocol information ongoing
   // * protocol connect or disconnect request for given protocol information in waiting queue
   // * given protocol connected

   // further conditions have to be checked now depending on current priority mode

   bool allowed(false);

   switch(_priorityMode)
   {
      case PRIO_MODE_PARALLEL_PIM_MSG:
         // implement if needed
         break;
      case PRIO_MODE_DEFAULT:
      case PRIO_MODE_LAST:
      default:
         allowed = checkPrioModeDefault(App2BtsOC_ConnectProtocol);
         break;
   }

   return allowed;
}

bool ConnectionPriority::checkDisconnectProtocolPriority(IN const App2Bts_DisconnectProtocol& request) const
{
   (void)(request);

   // following was checked before:
   // * invalid request
   // * protocol connect or disconnect to given protocol information ongoing
   // * protocol connect or disconnect request for given protocol information in waiting queue
   // * given protocol disconnected

   // further conditions have to be checked now depending on current priority mode

   bool allowed(false);

   switch(_priorityMode)
   {
      case PRIO_MODE_PARALLEL_PIM_MSG:
         // implement if needed
         break;
      case PRIO_MODE_DEFAULT:
      case PRIO_MODE_LAST:
      default:
         allowed = checkPrioModeDefault(App2BtsOC_DisconnectProtocol);
         break;
   }

   return allowed;
}

bool ConnectionPriority::checkStartDiscoveryPriority(IN const App2Bts_StartDiscovery& request) const
{
   (void)(request);

   // following was checked before:
   // * invalid request
   // * start/stop inquiry request ongoing
   // * start/stop inquiry request in waiting queue
   // * inquiry already ongoing

   // further conditions have to be checked now depending on current priority mode

   bool allowed(false);

   switch(_priorityMode)
   {
      case PRIO_MODE_PARALLEL_PIM_MSG:
         // implement if needed
         break;
      case PRIO_MODE_DEFAULT:
      case PRIO_MODE_LAST:
      default:
         allowed = checkPrioModeDefault(App2BtsOC_StartDiscovery);
         break;
   }

   return allowed;
}

bool ConnectionPriority::checkStopDiscoveryPriority(IN const App2Bts_StopDiscovery& request) const
{
   (void)(request);

   // following was checked before:
   // * invalid request
   // * start/stop inquiry request ongoing
   // * start/stop inquiry request in waiting queue
   // * inquiry already stopped

   // further conditions have to be checked now depending on current priority mode

   bool allowed(false);

   switch(_priorityMode)
   {
      case PRIO_MODE_PARALLEL_PIM_MSG:
         // implement if needed
         break;
      case PRIO_MODE_DEFAULT:
      case PRIO_MODE_LAST:
      default:
         allowed = checkPrioModeDefault(App2BtsOC_StopDiscovery);
         break;
   }

   return allowed;
}

bool ConnectionPriority::checkStartPairingPriority(IN const App2Bts_StartPairing& request) const
{
   (void)(request);

   // following was checked before:
   // * invalid request
   // * pairing ongoing
   // * start/cancel pairing request in waiting queue
   // * inquiry ongoing or start/stop inquiry ongoing

   // further conditions have to be checked now depending on current priority mode

   bool allowed(false);

   switch(_priorityMode)
   {
      case PRIO_MODE_PARALLEL_PIM_MSG:
         // implement if needed
         break;
      case PRIO_MODE_DEFAULT:
      case PRIO_MODE_LAST:
      default:
         allowed = checkPrioModeDefault(App2BtsOC_StartPairing);
         break;
   }

   return allowed;
}

bool ConnectionPriority::checkStartRemoteServiceSearchPriority(IN const App2Bts_StartRemoteServiceSearch& request) const
{
   (void)(request);

   // following was checked before:
   // * invalid request
   // * service search for given device address ongoing
   // * service search request for given device address in waiting queue

   // further conditions have to be checked now depending on current priority mode

   bool allowed(false);

   switch(_priorityMode)
   {
      case PRIO_MODE_PARALLEL_PIM_MSG:
         // implement if needed
         break;
      case PRIO_MODE_DEFAULT:
      case PRIO_MODE_LAST:
      default:
         allowed = checkPrioModeDefault(App2BtsOC_StartRemoteServiceSearch);
         break;
   }

   return allowed;
}

bool ConnectionPriority::checkPrioModeDefault(IN const BTSApp2BtsOpcode opCode) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_deviceManagerIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_protocolManagerIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_inquiryIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_pairingIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_serviceSearchIf);

   // get all needed information
   const bool anyProtocolConnectDisconnectOngoing(_protocolManagerIf->isAnyProtocolConnectDisconnectOngoing());
   const bool anyDeviceDisconnectOngoing(_deviceManagerIf->isAnyDisconnectOngoing());
   const bool anyInquiryActionOngoing(_inquiryIf->isDiscoveryOngoing() || _inquiryIf->isRequestOngoing());
   const bool anyPairingActionOngoing(_pairingIf->isPairingOngoing());
   const bool anyServiceSearchOngoing(_serviceSearchIf->isAnySearchOngoing());
   const bool anyAclBasedRequestWorking(isAnyAclBasedRequestInWorkingQueue());
   bool allowed(false);

   ETG_TRACE_USR2((" checkPrioModeDefault(): anyProtocolConnectDisconnectOngoing=%d anyDeviceDisconnectOngoing=%d anyInquiryActionOngoing=%d anyPairingActionOngoing=%d anyServiceSearchOngoing=%d anyAclBasedRequestWorking=%d", anyProtocolConnectDisconnectOngoing, anyDeviceDisconnectOngoing, anyInquiryActionOngoing, anyPairingActionOngoing, anyServiceSearchOngoing, anyAclBasedRequestWorking));

   // check priority depending on given request
   switch(opCode)
   {
      case App2BtsOC_DisconnectDevice:
      case App2BtsOC_ConnectProtocol:
      case App2BtsOC_DisconnectProtocol:
      case App2BtsOC_StartRemoteServiceSearch:
         // if no other request is ongoing then proceed
         if((false == anyProtocolConnectDisconnectOngoing) &&
            (false == anyDeviceDisconnectOngoing) &&
            (false == anyInquiryActionOngoing) &&
            (false == anyPairingActionOngoing) &&
            (false == anyServiceSearchOngoing) &&
            (false == anyAclBasedRequestWorking))
         {
            allowed = true;
         }
         break;
      case App2BtsOC_StartDiscovery:
      case App2BtsOC_StopDiscovery:
         // if no other request is ongoing then proceed
         if((false == anyProtocolConnectDisconnectOngoing) &&
            (false == anyDeviceDisconnectOngoing) &&
            (false == anyPairingActionOngoing) &&
            (false == anyServiceSearchOngoing) &&
            (false == anyAclBasedRequestWorking))
         {
            allowed = true;
         }
         break;
      case App2BtsOC_StartPairing:
         // if no other request is ongoing then proceed
         if((false == anyProtocolConnectDisconnectOngoing) &&
            (false == anyDeviceDisconnectOngoing) &&
            (false == anyServiceSearchOngoing) &&
            (false == anyAclBasedRequestWorking))
         {
            allowed = true;
         }
         break;
      default:
         FW_ERRMEM_ASSERT_ALWAYS();
         break;
   }

   return allowed;
}

void ConnectionPriority::handleAclConnectFailed(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_protocolManagerIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_serviceSearchIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_pairingIf);

   // if the connect of first protocol fails or service search fails (while device is not connected) we assume that the ACL connection failed (device is out of range or switched off)
   // - find all connect/disconnect/service search requests for the given device address
   // - answer directly with BTS_REQ_CONNECT_ACL_FAILED
   BTSApp2BtsMessageMasterCompareItem matchItem;
   ::std::vector< BTSApp2BtsMessageMasterCompareItem > matchingItemList;
   ::std::vector< BTSApp2BtsMessageMasterCompareItem > highPrioItemList;

   matchingItemList.reserve(6);

   matchItem.deviceAddress = address;
   matchItem.compareDeviceAddress = true;

   matchItem.opCode = App2BtsOC_ConnectProtocol;
   matchingItemList.push_back(matchItem);

   matchItem.opCode = App2BtsOC_DisconnectProtocol;
   matchingItemList.push_back(matchItem);

   matchItem.opCode = App2BtsOC_StartRemoteServiceSearch;
   matchingItemList.push_back(matchItem);

   matchItem.opCode = App2BtsOC_CancelRemoteServiceSearch;
   matchingItemList.push_back(matchItem);

   matchItem.opCode = App2BtsOC_StartPairing;
   matchingItemList.push_back(matchItem);

   matchItem.opCode = App2BtsOC_CancelPairing;
   matchingItemList.push_back(matchItem);

   ::std::vector< App2Bts_BaseMessage* > msgList;
   msgList.reserve(_controlIf->getApp2BtsWaitingQueueSize());

   // find matching entries in waiting queue
   (void)_controlIf->getMatchingWaitingQueueEntries(msgList, matchingItemList, highPrioItemList);

   // answer all requests (maybe no further requests are in waiting queue)
   App2Bts_BaseMessage* ptrMessage;

   for(size_t i = 0; i < msgList.size(); i++)
   {
      ptrMessage = msgList[i];

      if(0 == ptrMessage)
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         continue;
      }

      //if(true == _handleDoubledRequests)
      {
         const BTSApp2BtsOpcode opcode(ptrMessage->getOpCode());

         ETG_TRACE_USR3((" handleAclConnectFailed(): App2Bts 0x%04X to be handled", ptrMessage->getTraceOpCode()));

         // handle doubled messages depending on opcode
         // - create direct answer message (Bts2App) in sub-handler function
         // - handle any error in sub-handler function because there is the best place to handle
         // - do not delete message; this is done at the end of this function
         switch(opcode)
         {
            case App2BtsOC_ConnectProtocol:
            {
               App2Bts_ConnectProtocol* ptrMsg = static_cast< App2Bts_ConnectProtocol* >(ptrMessage);
               _protocolManagerIf->setDisconnectReason(ptrMsg->getBDAddress(), ptrMsg->getProtocolId(), ptrMsg->getUuid(), ptrMsg->getMasInstanceName(), BTS_DISCONNECT_REASON_OUT_OF_RANGE);
               _protocolManagerIf->sendStatusAndResult(bts2AppMsgList, *ptrMsg, false, BTS_REQ_CONNECT_ACL_FAILED, BTS_DISCONNECT_REASON_OUT_OF_RANGE);
               break;
            }
            case App2BtsOC_DisconnectProtocol:
            {
               App2Bts_DisconnectProtocol* ptrMsg = static_cast< App2Bts_DisconnectProtocol* >(ptrMessage);
               _protocolManagerIf->sendStatusAndResult(bts2AppMsgList, *ptrMsg, false, BTS_REQ_SUCCESS, BTS_DISCONNECT_REASON_NOT_VALID);
               break;
            }
            case App2BtsOC_StartRemoteServiceSearch:
            {
               App2Bts_StartRemoteServiceSearch* ptrMsg = static_cast< App2Bts_StartRemoteServiceSearch* >(ptrMessage);
               _serviceSearchIf->sendStatusAndResult(bts2AppMsgList, *ptrMsg, false, BTS_REQ_FAILED, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);
               break;
            }
            case App2BtsOC_CancelRemoteServiceSearch:
            {
               // nothing to do
               break;
            }
            case App2BtsOC_StartPairing:
            {
               App2Bts_StartPairing* ptrMsg = static_cast< App2Bts_StartPairing* >(ptrMessage);
               _pairingIf->sendStatusAndResult(bts2AppMsgList, *ptrMsg, false, BTS_REQ_PAIRING_UNKNOWN, BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE);
               break;
            }
            case App2BtsOC_CancelPairing:
            {
               // nothing to do
               break;
            }
            // all other
            default:
            {
               FW_ERRMEM_ASSERT_ALWAYS();
               break;
            }
         }
      }
   }

   // remove entries from waiting queue
   _controlIf->removeApp2BtsWaitingMessages(msgList);

   // delete all entries
   for(size_t i = 0; i < msgList.size(); i++)
   {
      if(0 != msgList[i])
      {
         delete msgList[i];
      }
   }

   // keep protocol, device and service search entries
}

bool ConnectionPriority::continueWithNextAclBasedAction(void) const
{
   bool allowed(false);

   switch(_priorityMode)
   {
      case PRIO_MODE_PARALLEL_PIM_MSG:
         // implement if needed
         break;
      case PRIO_MODE_DEFAULT:
      case PRIO_MODE_LAST:
      default:
         allowed = continueWithNextAclBasedActionPrioModeDefault();
         break;
   }

   return allowed;
}

bool ConnectionPriority::continueWithNextAclBasedActionPrioModeDefault(void) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_deviceManagerIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_protocolManagerIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_inquiryIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_pairingIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_serviceSearchIf);

   // if any remote connect is ongoing we shall not check waiting queue
   // if any local connect/disconnect is ongoing we shall not check waiting queue
   // if any service search is ongoing we shall not check waiting queue
   // if any pairing is ongoing we shall not check waiting queue
   // keep in mind that during this moment the current request could be still in working queue

   // check for any ongoing connect/disconnect (local or remote)
   if(true == _protocolManagerIf->isAnyProtocolConnectDisconnectOngoing())
   {
      return false;
   }
   // check for any ongoing device disconnect
   else if(true == _deviceManagerIf->isAnyDisconnectOngoing())
   {
      return false;
   }
   // check for any service search ongoing
   else if(true == _serviceSearchIf->isAnySearchOngoing())
   {
      return false;
   }
   // check for inquiry ongoing
   else if((true == _inquiryIf->isDiscoveryOngoing()) || (true == _inquiryIf->isRequestOngoing()))
   {
      return false;
   }
   // check for ongoing pairing
   else if(true == _pairingIf->isPairingOngoing())
   {
      return false;
   }

   return true;
}

bool ConnectionPriority::isDisconnectRequestInWaitingQueue(IN const BTSBDAddress& address) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_controlIf);

   BTSApp2BtsMessageMasterCompareItem item;
   ::std::vector< BTSApp2BtsMessageMasterCompareItem > itemList;

   item.deviceAddress = address;

   item.compareDeviceAddress = true;

   item.opCode = App2BtsOC_DisconnectDevice;
   itemList.push_back(item);

   // check if opcode is in waiting queue
   return _controlIf->isSimilarOpCodeInWaitingQueue(itemList);
}

bool ConnectionPriority::isConnectDisconnectRequestInWaitingQueue(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceName& masInstance) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_controlIf);

   BTSApp2BtsMessageMasterCompareItem item;
   ::std::vector< BTSApp2BtsMessageMasterCompareItem > itemList;

   itemList.reserve(2);

   item.deviceAddress = address;
   item.protocolId = protocol;
   item.sppUuid = uuid;
   item.masInstance = masInstance;

   item.compareDeviceAddress = true;
   item.compareProtocolId = true;
   item.compareSppUuid = true;
   item.compareMasInstance = true;

   item.opCode = App2BtsOC_ConnectProtocol;
   itemList.push_back(item);

   item.opCode = App2BtsOC_DisconnectProtocol;
   itemList.push_back(item);

   // check if opcode is in waiting queue
   return _controlIf->isSimilarOpCodeInWaitingQueue(itemList);
}

bool ConnectionPriority::isDiscoveryRequestInWaitingQueue(void) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_controlIf);

   BTSApp2BtsMessageMasterCompareItem item;
   ::std::vector< BTSApp2BtsMessageMasterCompareItem > itemList;

   itemList.reserve(2);

   item.opCode = App2BtsOC_StartDiscovery;
   itemList.push_back(item);

   item.opCode = App2BtsOC_StopDiscovery;
   itemList.push_back(item);

   // check if opcode is in waiting queue
   return _controlIf->isSimilarOpCodeInWaitingQueue(itemList);
}

bool ConnectionPriority::isPairingRequestInWaitingQueue(IN const BTSBDAddress& address) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_controlIf);

   BTSApp2BtsMessageMasterCompareItem item;
   ::std::vector< BTSApp2BtsMessageMasterCompareItem > itemList;

   itemList.reserve(2);

   item.deviceAddress = address;

   item.compareDeviceAddress = true;

   item.opCode = App2BtsOC_StartPairing;
   itemList.push_back(item);

   item.opCode = App2BtsOC_CancelPairing;
   itemList.push_back(item);

   // check if opcode is in waiting queue
   return _controlIf->isSimilarOpCodeInWaitingQueue(itemList);
}

bool ConnectionPriority::isServiceSearchRequestInWaitingQueue(IN const BTSBDAddress& address, IN const BTSSearchType type) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_controlIf);

   BTSApp2BtsMessageMasterCompareItem item;
   ::std::vector< BTSApp2BtsMessageMasterCompareItem > itemList;

   itemList.reserve(2);

   item.deviceAddress = address;
   item.searchType = type;

   item.compareDeviceAddress = true;
   item.compareSearchType = true;

   item.opCode = App2BtsOC_StartRemoteServiceSearch;
   itemList.push_back(item);

   item.opCode = App2BtsOC_CancelRemoteServiceSearch;
   itemList.push_back(item);

   // check if opcode is in waiting queue
   return _controlIf->isSimilarOpCodeInWaitingQueue(itemList);
}

bool ConnectionPriority::isAnyAclBasedRequestInWorkingQueue(void) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_controlIf);

   // TODO: [low]: recheck for App2BtsOC_DisconnectDevice

   BTSApp2BtsMessageMasterCompareItem item;
   ::std::vector<BTSApp2BtsMessageMasterCompareItem> itemList;

   itemList.reserve(9);

   item.opCode = App2BtsOC_DisconnectDevice;
   itemList.push_back(item);

   item.opCode = App2BtsOC_ConnectProtocol;
   itemList.push_back(item);

   item.opCode = App2BtsOC_DisconnectProtocol;
   itemList.push_back(item);

   item.opCode = App2BtsOC_StartRemoteServiceSearch;
   itemList.push_back(item);

   item.opCode = App2BtsOC_CancelRemoteServiceSearch; // is never in working queue
   itemList.push_back(item);

   item.opCode = App2BtsOC_StartDiscovery;
   itemList.push_back(item);

   item.opCode = App2BtsOC_StopDiscovery;
   itemList.push_back(item);

   item.opCode = App2BtsOC_StartPairing;
   itemList.push_back(item);

   item.opCode = App2BtsOC_CancelPairing; // is never in working queue
   itemList.push_back(item);

   // check if opcode is in working queue
   return _controlIf->isSimilarOpCodeInWorkingQueue(itemList);
}

BTSApp2BtsMessageHandlingType ConnectionPriority::checkRequestResult(IN const bool processFlag) const
{
   if(true == processFlag)
   {
      // request processing was started => push to working queue
      return BTS_APP2BTS_PUSH_TO_WORKING;
   }
   else
   {
      // no further processing necessary => delete message
      return BTS_APP2BTS_DELETE;
   }
}

bool ConnectionPriority::isConnectionRelatedOpCode(IN const BTSApp2BtsOpcode opCode) const
{
   if((App2BtsOC_DisconnectDevice == opCode) ||
      (App2BtsOC_ConnectProtocol == opCode) ||
      (App2BtsOC_DisconnectProtocol == opCode) ||
      (App2BtsOC_StartRemoteServiceSearch == opCode) ||
      (App2BtsOC_CancelRemoteServiceSearch == opCode) ||
      (App2BtsOC_StartDiscovery == opCode) ||
      (App2BtsOC_StopDiscovery == opCode) ||
      (App2BtsOC_StartPairing == opCode) ||
      (App2BtsOC_CancelPairing == opCode))
   {
      return true;
   }

   return false;
}

void ConnectionPriority::getSimilarOpCodesPrioModeDefault(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, IN const App2Bts_BaseMessage& message) const
{
   const BTSApp2BtsOpcode opCode(message.getOpCode());
   BTSApp2BtsMessageMasterCompareItem item;
   size_t reserveSize(0);

   switch(opCode)
   {
      case App2BtsOC_StartDiscovery: /*DONE*/
         reserveSize += 9;
         itemList.reserve(reserveSize);
         item.opCode = opCode;
         itemList.push_back(item);
         item.opCode = App2BtsOC_ConnectProtocol;
         itemList.push_back(item);
         item.opCode = App2BtsOC_DisconnectProtocol;
         itemList.push_back(item);
         item.opCode = App2BtsOC_DisconnectDevice;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartRemoteServiceSearch;
         itemList.push_back(item);
         item.opCode = App2BtsOC_CancelRemoteServiceSearch; // is never in working queue
         itemList.push_back(item);
         item.opCode = App2BtsOC_StopDiscovery;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartPairing;
         itemList.push_back(item);
         item.opCode = App2BtsOC_CancelPairing;
         itemList.push_back(item);
         break;
      case App2BtsOC_StopDiscovery: /*DONE*/
         reserveSize += 9;
         itemList.reserve(reserveSize);
         item.opCode = opCode;
         itemList.push_back(item);
         item.opCode = App2BtsOC_ConnectProtocol;
         itemList.push_back(item);
         item.opCode = App2BtsOC_DisconnectProtocol;
         itemList.push_back(item);
         item.opCode = App2BtsOC_DisconnectDevice;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartRemoteServiceSearch;
         itemList.push_back(item);
         item.opCode = App2BtsOC_CancelRemoteServiceSearch; // is never in working queue
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartDiscovery;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartPairing;
         itemList.push_back(item);
         item.opCode = App2BtsOC_CancelPairing;
         itemList.push_back(item);
         break;
      case App2BtsOC_StartPairing: /*DONE*/
         reserveSize += 9;
         itemList.reserve(reserveSize);
         item.opCode = opCode;
         itemList.push_back(item);
         item.opCode = App2BtsOC_ConnectProtocol;
         itemList.push_back(item);
         item.opCode = App2BtsOC_DisconnectProtocol;
         itemList.push_back(item);
         item.opCode = App2BtsOC_DisconnectDevice;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartRemoteServiceSearch;
         itemList.push_back(item);
         item.opCode = App2BtsOC_CancelRemoteServiceSearch; // is never in working queue
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartDiscovery;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StopDiscovery;
         itemList.push_back(item);
         item.opCode = App2BtsOC_CancelPairing;
         itemList.push_back(item);
         break;
      case App2BtsOC_CancelPairing: /*DONE*/
         reserveSize += 9;
         itemList.reserve(reserveSize);
         item.opCode = opCode;
         itemList.push_back(item);
         item.opCode = App2BtsOC_ConnectProtocol;
         itemList.push_back(item);
         item.opCode = App2BtsOC_DisconnectProtocol;
         itemList.push_back(item);
         item.opCode = App2BtsOC_DisconnectDevice;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartRemoteServiceSearch;
         itemList.push_back(item);
         item.opCode = App2BtsOC_CancelRemoteServiceSearch; // is never in working queue
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartDiscovery;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StopDiscovery;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartPairing;
         itemList.push_back(item);
         break;
      case App2BtsOC_ConnectProtocol: /*DONE*/
         reserveSize += 9;
         itemList.reserve(reserveSize);
         item.opCode = opCode;
         itemList.push_back(item);
         item.opCode = App2BtsOC_DisconnectProtocol;
         itemList.push_back(item);
         item.opCode = App2BtsOC_DisconnectDevice;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartRemoteServiceSearch;
         itemList.push_back(item);
         item.opCode = App2BtsOC_CancelRemoteServiceSearch; // is never in working queue
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartDiscovery;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StopDiscovery;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartPairing;
         itemList.push_back(item);
         item.opCode = App2BtsOC_CancelPairing;
         itemList.push_back(item);
         break;
      case App2BtsOC_DisconnectProtocol: /*DONE*/
         reserveSize += 9;
         itemList.reserve(reserveSize);
         item.opCode = opCode;
         itemList.push_back(item);
         item.opCode = App2BtsOC_ConnectProtocol;
         itemList.push_back(item);
         item.opCode = App2BtsOC_DisconnectDevice;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartRemoteServiceSearch;
         itemList.push_back(item);
         item.opCode = App2BtsOC_CancelRemoteServiceSearch; // is never in working queue
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartDiscovery;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StopDiscovery;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartPairing;
         itemList.push_back(item);
         item.opCode = App2BtsOC_CancelPairing;
         itemList.push_back(item);
         break;
      case App2BtsOC_DisconnectDevice: /*DONE*/
         reserveSize += 9;
         itemList.reserve(reserveSize);
         item.opCode = opCode;
         itemList.push_back(item);
         item.opCode = App2BtsOC_ConnectProtocol;
         itemList.push_back(item);
         item.opCode = App2BtsOC_DisconnectProtocol;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartRemoteServiceSearch;
         itemList.push_back(item);
         item.opCode = App2BtsOC_CancelRemoteServiceSearch; // is never in working queue
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartDiscovery;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StopDiscovery;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartPairing;
         itemList.push_back(item);
         item.opCode = App2BtsOC_CancelPairing;
         itemList.push_back(item);
         break;
      case App2BtsOC_StartRemoteServiceSearch: /*DONE*/
         reserveSize += 9;
         itemList.reserve(reserveSize);
         item.opCode = opCode;
         itemList.push_back(item);
         item.opCode = App2BtsOC_ConnectProtocol;
         itemList.push_back(item);
         item.opCode = App2BtsOC_DisconnectProtocol;
         itemList.push_back(item);
         item.opCode = App2BtsOC_DisconnectDevice;
         itemList.push_back(item);
         item.opCode = App2BtsOC_CancelRemoteServiceSearch; // is never in working queue
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartDiscovery;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StopDiscovery;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartPairing;
         itemList.push_back(item);
         item.opCode = App2BtsOC_CancelPairing;
         itemList.push_back(item);
         break;
      case App2BtsOC_CancelRemoteServiceSearch: /*DONE*/
         reserveSize += 9;
         itemList.reserve(reserveSize);
         item.opCode = opCode;
         itemList.push_back(item);
         item.opCode = App2BtsOC_ConnectProtocol;
         itemList.push_back(item);
         item.opCode = App2BtsOC_DisconnectProtocol;
         itemList.push_back(item);
         item.opCode = App2BtsOC_DisconnectDevice;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartRemoteServiceSearch;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartDiscovery;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StopDiscovery;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartPairing;
         itemList.push_back(item);
         item.opCode = App2BtsOC_CancelPairing;
         itemList.push_back(item);
         break;
      default:
         FW_ERRMEM_ASSERT_ALWAYS();
         break;
   }
}

void ConnectionPriority::getMatchingOpCodesPrioModeDefault(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& highPrioItemList, IN const App2Bts_BaseMessage& message) const
{
   const BTSApp2BtsOpcode opCode(message.getOpCode());
   BTSApp2BtsMessageMasterCompareItem item;
   size_t reserveSize(0);

   message.getCompareItem(item);

   switch(opCode)
   {
      case App2BtsOC_StartDiscovery: /*DONE*/
         reserveSize = 2;
         itemList.reserve(reserveSize);
         item.opCode = opCode;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StopDiscovery;
         itemList.push_back(item);
         break;
      case App2BtsOC_StopDiscovery: /*DONE*/
         reserveSize = 2;
         itemList.reserve(reserveSize);
         item.opCode = opCode;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartDiscovery;
         itemList.push_back(item);
         break;
      case App2BtsOC_StartPairing: /*DONE*/
         reserveSize = 2;
         itemList.reserve(reserveSize);
         item.compareDeviceAddress = true; // only 1 pairing at the same time, doubled requests will be answered with failed, client has to take care
         item.opCode = opCode;
         itemList.push_back(item);
         item.opCode = App2BtsOC_CancelPairing;
         itemList.push_back(item);
         break;
      case App2BtsOC_CancelPairing: /*DONE*/
         reserveSize = 2;
         itemList.reserve(reserveSize);
         item.compareDeviceAddress = true; // only 1 pairing at the same time, doubled requests will be answered with failed, client has to take care
         item.opCode = opCode;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartPairing;
         itemList.push_back(item);
         break;
      case App2BtsOC_ConnectProtocol: /*DONE*/
         reserveSize = 2;
         itemList.reserve(reserveSize);
         item.compareDeviceAddress = true;
         item.compareProtocolId = true;
         item.compareSppUuid = true;
         item.compareMasInstance = true;
         item.opCode = opCode;
         itemList.push_back(item);
         item.opCode = App2BtsOC_DisconnectProtocol;
         itemList.push_back(item);
         // high prio items: DisconnectDevice
         highPrioItemList.reserve(1);
         item.compareDeviceAddress = true;
         item.compareProtocolId = false;
         item.compareSppUuid = false;
         item.compareMasInstance = false;
         item.opCode = App2BtsOC_DisconnectDevice;
         highPrioItemList.push_back(item);
         break;
      case App2BtsOC_DisconnectProtocol: /*DONE*/
         reserveSize = 2;
         itemList.reserve(reserveSize);
         item.compareDeviceAddress = true;
         item.compareProtocolId = true;
         item.compareSppUuid = true;
         item.compareMasInstance = true;
         item.opCode = opCode;
         itemList.push_back(item);
         item.opCode = App2BtsOC_ConnectProtocol;
         itemList.push_back(item);
         // high prio items: DisconnectDevice
         highPrioItemList.reserve(1);
         item.compareDeviceAddress = true;
         item.compareProtocolId = false;
         item.compareSppUuid = false;
         item.compareMasInstance = false;
         item.opCode = App2BtsOC_DisconnectDevice;
         highPrioItemList.push_back(item);
         break;
      case App2BtsOC_DisconnectDevice: /*DONE*/
         reserveSize = 1;
         itemList.reserve(reserveSize);
         item.compareDeviceAddress = true;
         item.opCode = opCode;
         itemList.push_back(item);
         // high prio items: DisconnectDevice
         highPrioItemList.reserve(1);
         item.compareDeviceAddress = true;
         item.opCode = App2BtsOC_DisconnectDevice;
         highPrioItemList.push_back(item);
         break;
      case App2BtsOC_StartRemoteServiceSearch: /*DONE*/
         reserveSize = 2;
         itemList.reserve(reserveSize);
         item.compareDeviceAddress = true;
         item.compareSearchType = true;
         item.opCode = opCode;
         itemList.push_back(item);
         item.opCode = App2BtsOC_CancelRemoteServiceSearch;
         itemList.push_back(item);
         break;
      case App2BtsOC_CancelRemoteServiceSearch: /*DONE*/
         reserveSize = 2;
         itemList.reserve(reserveSize);
         item.compareDeviceAddress = true;
         item.compareSearchType = true;
         item.opCode = opCode;
         itemList.push_back(item);
         item.opCode = App2BtsOC_StartRemoteServiceSearch;
         itemList.push_back(item);
         break;
      default:
         FW_ERRMEM_ASSERT_ALWAYS();
         break;
   }
}

} //btstackif
