/**
 * @file Configuration.cpp
 *
 * @par SW-Component
 * State machine for configuration
 *
 * @brief Implementation of generic configuration state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic configuration state machine.
 */

#include "Configuration.h"
#include "IConfigurationClient.h"
#include "FwBluetoothStringUtils.h"
#include "FwStringUtils.h"
#include "FwAssert.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/Configuration.cpp.trc.h"
#endif
#endif

namespace btstackif {

Configuration::Configuration() :
_config(),
_clientListBeforeBtOn(),
_clientListAfterBtOn(),
_iteratorListBeforeBtOn(),
_iteratorListAfterBtOn(),
_ongoing(false),
_errorOccurred(false)
{
   // reserve some entries
   _clientListBeforeBtOn.reserve(10);
   _clientListAfterBtOn.reserve(10);
}

Configuration::~Configuration()
{
}

void Configuration::reset(void)
{
   StateMachine::reset();
   // keep _config
   // keep _clientListBeforeBtOn
   // keep _clientListAfterBtOn
   _iteratorListBeforeBtOn = _clientListBeforeBtOn.end();
   _iteratorListAfterBtOn = _clientListAfterBtOn.end();
   _ongoing = false;
   _errorOccurred = false;
}

void Configuration::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   // reset control data
   reset();
}

void Configuration::setConfiguration(IN const BTSLocalStackConfiguration& configuration)
{
   _config = configuration;

   convertCPWSettings();
#ifdef VARIANT_S_FTR_ENABLE_ANDROID_AUTO_WIRELESS
   addAAWSettings();
#endif
   checkSppSettings();
   correctConfiguration();
}

const BTSLocalStackConfiguration& Configuration::getConfiguration(void) const
{
   return _config;
}

BTSLocalStackConfiguration& Configuration::getConfiguration(void)
{
   return _config;
}

IConfigurationMaster* Configuration::getConfigurationMasterHandler(void)
{
   return this;
}

IStateMachine* Configuration::getSmEntryInterface(void)
{
   return this;
}

void Configuration::setOngoing(IN const bool enable)
{
   _ongoing = enable;
}

bool Configuration::getOngoing(void) const
{
   return _ongoing;
}

void Configuration::startSequence(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   start();

   setOngoing(true);

   // reset error flag
   _errorOccurred = false;

   // start with first configuration step
   BTSHandleIpc2BtsMessageItem messageItem;
   (void)executeNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
}

void Configuration::continueSequence(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   bool next(true);

   while(true == next)
   {
      if(true == executeNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem))
      {
         next = false; // message was added to list
      }
      else /*if(false == getOngoing())*/
      {
         next = false; // end of sequence reached
      }
   }
}

void Configuration::setError(IN const bool configurationError)
{
   _errorOccurred = configurationError;
}

void Configuration::registerConfigurationClient(IN IConfigurationClient* client, IN const bool beforeBtOn)
{
   FW_IF_NULL_PTR_RETURN(client);

   // please ensure outside of this function that each client is only registered once

   if(true == beforeBtOn)
   {
      _clientListBeforeBtOn.push_back(client);
   }
   else
   {
      _clientListAfterBtOn.push_back(client);
   }

   // set configuration master reference
   client->setConfigurationMasterIf(this);
}

void Configuration::start(void /*OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList*/)
{
   ETG_TRACE_USR2((" start()"));

   // inform all registered clients
   for(::std::vector< IConfigurationClient* >::const_iterator it = _clientListBeforeBtOn.begin(); it != _clientListBeforeBtOn.end(); ++it)
   {
      if(0 != *it)
      {
         (*it)->startGlobalConfiguration();
      }
   }

   for(::std::vector< IConfigurationClient* >::const_iterator it = _clientListAfterBtOn.begin(); it != _clientListAfterBtOn.end(); ++it)
   {
      if(0 != *it)
      {
         (*it)->startGlobalConfiguration();
      }
   }

   // reset iterators
   _iteratorListBeforeBtOn = _clientListBeforeBtOn.begin();
   _iteratorListAfterBtOn = _clientListAfterBtOn.begin();
}

void Configuration::stop(void /*OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList*/)
{
   ETG_TRACE_USR2((" stop()"));

   // inform all registered clients
   for(::std::vector< IConfigurationClient* >::const_iterator it = _clientListBeforeBtOn.begin(); it != _clientListBeforeBtOn.end(); ++it)
   {
      if(0 != *it)
      {
         (*it)->stopGlobalConfiguration();
      }
   }

   for(::std::vector< IConfigurationClient* >::const_iterator it = _clientListAfterBtOn.begin(); it != _clientListAfterBtOn.end(); ++it)
   {
      if(0 != *it)
      {
         (*it)->stopGlobalConfiguration();
      }
   }
}

bool Configuration::doNextConfigurationStep(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   ETG_TRACE_USR2((" doNextConfigurationStep()"));

   bool configActive(false);

   // check list before BT on
   while(_iteratorListBeforeBtOn != _clientListBeforeBtOn.end())
   {
      if(0 != *_iteratorListBeforeBtOn)
      {
         configActive = (*_iteratorListBeforeBtOn)->setSingleConfiguration(bts2IpcMsgList, bts2AppMsgList, messageItem, true, _errorOccurred);
      }

      if(true == configActive)
      {
         return true;
      }
      else
      {
         ++_iteratorListBeforeBtOn;
      }
   }

   // check list after BT on
   while(_iteratorListAfterBtOn != _clientListAfterBtOn.end())
   {
      if(0 != *_iteratorListAfterBtOn)
      {
         configActive = (*_iteratorListAfterBtOn)->setSingleConfiguration(bts2IpcMsgList, bts2AppMsgList, messageItem, false, _errorOccurred);
      }

      if(true == configActive)
      {
         return true;
      }
      else
      {
         ++_iteratorListAfterBtOn;
      }
   }

   // all is done

   return false;
}

bool Configuration::executeNextConfigurationStep(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   ETG_TRACE_USR2((" executeNextConfigurationStep()"));

   if(true == doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem))
   {
      return true;
   }
   else
   {
      // configuration sequence finished
      setOngoing(false);
      stop();
   }

   return false;
}

void Configuration::convertCPWSettings(void)
{
   // convert configuration data from old to new format
   // carPlayWirelessEnabled => sppServiceInfoList

   if(true == _config.carPlayWirelessEnabled)
   {
      bool found(false);
      ::std::string workingUuid;
      for(::std::vector< BTSSppServiceInfo >::const_iterator it = _config.sppServiceInfoList.begin(); it != _config.sppServiceInfoList.end(); ++it)
      {
         if(false == it->localUuid.empty())
         {
            workingUuid = it->localUuid;
            ::fw::convertString2LowerCase(workingUuid);

            if(::fw::getIncomingCPWUuid() == workingUuid)
            {
               found = true;
               break;
            }
         }
      }

      if(false == found)
      {
         BTSSppServiceInfo serviceInfo;
         serviceInfo.serviceName = "CarPlay Wireless";
         serviceInfo.localUuid = ::fw::getIncomingCPWUuid();
         // serviceInfo.remoteUuid

         _config.sppServiceInfoList.push_back(serviceInfo);
      }
   }
}

void Configuration::checkSppSettings(void)
{
   // verify given SPP service information

   BTSSppServiceInfoList tmpServiceInfoList;
   tmpServiceInfoList.reserve(_config.sppServiceInfoList.size());
   BTSSppServiceInfo serviceInfo;

   for(::std::vector< BTSSppServiceInfo >::const_iterator it = _config.sppServiceInfoList.begin(); it != _config.sppServiceInfoList.end(); ++it)
   {
      const bool validLocalUuid(::fw::isValid128BitUuid(it->localUuid));
      const bool validRemoteUuid(::fw::isValid128BitUuid(it->remoteUuid));

      if((true == validLocalUuid) || (true == validRemoteUuid))
      {
         // at least one UUID is valid => take over

         serviceInfo.serviceName = it->serviceName;

         if(true == validLocalUuid)
         {
            serviceInfo.localUuid = it->localUuid;
            ::fw::convertString2LowerCase(serviceInfo.localUuid);
         }

         if(true == validRemoteUuid)
         {
            serviceInfo.remoteUuid = it->remoteUuid;
            ::fw::convertString2LowerCase(serviceInfo.remoteUuid);
         }

         tmpServiceInfoList.push_back(serviceInfo);
      }

      serviceInfo.localUuid.clear();
      serviceInfo.remoteUuid.clear();
   }

   _config.sppServiceInfoList.swap(tmpServiceInfoList);
}

void Configuration::correctConfiguration(void)
{
   // enable always DID
   _config.supportedServices.setBit(BTS_SUPP_SRV_DID);
}

void Configuration::addAAWSettings(void)
{
   BTSSppServiceInfo serviceInfo;
   serviceInfo.serviceName = "AndroidAuto Wireless";
   serviceInfo.localUuid = ::fw::getIncomingAAWUuid();
   // serviceInfo.remoteUuid

   _config.sppServiceInfoList.push_back(serviceInfo);
}

} //btstackif
