/**
 * @file IConfigurationMaster.h
 *
 * @par SW-Component
 * State machine for configuration
 *
 * @brief Interface definition for configuration master.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the interface for configuration master.
 */

#ifndef _I_CONFIGURATION_MASTER_H_
#define _I_CONFIGURATION_MASTER_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IConfigurationClient;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class IConfigurationMaster
{
public:
   /**
    * Destructor.
    */
   virtual ~IConfigurationMaster() {}

   /**
    * Register configuration client.
    *
    * @param[in] client: configuration client
    * @param[in] beforeBtOn: flag for configuration before or after switching BT on
    */
   virtual void registerConfigurationClient(IN IConfigurationClient* client, IN const bool beforeBtOn) = 0;

   /**
    * Start configuration.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    */
   virtual void start(void /*OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList*/) = 0;

   /**
    * Stop configuration.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    */
   virtual void stop(void /*OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList*/) = 0;

   /**
    * Do next configuration step.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    *
    * @return = true: configuration ongoing,
    * @return = false: configuration not ongoing (finished, stopped, aborted, no configuration item available)
    */
   virtual bool doNextConfigurationStep(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem) = 0;
};

} //btstackif

#endif //_I_CONFIGURATION_MASTER_H_
