/**
 * @file IConfigurationClient.h
 *
 * @par SW-Component
 * State machine for configuration
 *
 * @brief Interface definition for configuration client.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the interface for configuration client.
 */

#ifndef _I_CONFIGURATION_CLIENT_H_
#define _I_CONFIGURATION_CLIENT_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IConfigurationMaster;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class IConfigurationClient
{
public:
   /**
    * Destructor.
    */
   virtual ~IConfigurationClient() {}

   /**
    * Set configuration master interface.
    *
    * @param[in] master: configuration master interface
    */
   virtual void setConfigurationMasterIf(IN IConfigurationMaster* master) = 0;

   /**
    * Start global configuration.
    */
   virtual void startGlobalConfiguration(void) = 0;

   /**
    * Stop global configuration.
    */
   virtual void stopGlobalConfiguration(void) = 0;

   /**
    * Check ongoing single configuration.
    *
    * @return = true: single configuration ongoing,
    * @return = false: single configuration not ongoing
    */
   virtual bool isSingleConfigurationOngoing(void) const = 0;

   /**
    * Set single configuration item.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] beforeBtOn: flag for configuration before or after switching BT on
    * @param[in] errorOccurred: flag indicating that an error occurred; client shall decide whether to continue with configuration step or not
    *
    * @return = true: single configuration ongoing,
    * @return = false: single configuration not ongoing (finished, stopped, aborted, no configuration item available)
    */
   virtual bool setSingleConfiguration(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool beforeBtOn, IN const bool errorOccurred) = 0;
};

} //btstackif

#endif //_I_CONFIGURATION_CLIENT_H_
