/**
 * @file IBasicConfiguration.h
 *
 * @par SW-Component
 * State machine for basic configuration
 *
 * @brief Interface definition for generic basic configuration state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for basic configuration state machine.
 */

#ifndef _I_BASIC_CONFIGURATION_H_
#define _I_BASIC_CONFIGURATION_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IBasicConfigurationRequest;
class IBasicControl;
class ITimerPool;
class IConfiguration;
class IConfigurationClient;
class IStateMachine;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;
class IBasicResultCallback;

/**
 * Interface definition.
 */
class IBasicConfiguration
{
public:
   /**
    * Destructor.
    */
   virtual ~IBasicConfiguration() {}

   /**
    * Set specific instance (implementation).
    *
    * @param[in] instance: instance (implementation)
    * @param[in] configurationParameter: configuration parameter id
    * @param[in] beforeBtOn: flag for configuration before or after switching BT on
    * @param[in] configureDuringStartup: flag indicating that parameter shall be configured during startup
    */
   virtual void setInstance(IN IBasicConfigurationRequest* instance, IN const BTSConfigurationParameter configurationParameter, IN const bool beforeBtOn, IN const bool configureDuringStartup = true) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set timer pool interface.
    *
    * @param[in] timerPool: timer pool interface
    */
   virtual void setTimerPoolIf(IN ITimerPool* timerPool) = 0;

   /**
    * Set configuration interface.
    *
    * @param[in] configuration: configuration interface
    */
   virtual void setConfigurationIf(IN IConfiguration* configuration) = 0;

   /**
    * Get configuration client handler.
    *
    * @return = configuration client handler
    */
   virtual IConfigurationClient* getConfigurationClientHandler(void) = 0;

   /**
    * Get SM entry interface.
    *
    * @return = SM entry interface
    */
   virtual IStateMachine* getSmEntryInterface(void) = 0;

   /**
    * Check whether runtime configuration can be started.
    *
    * @return = true: runtime configuration can be started,
    * @return = false: runtime configuration is not possible now
    */
   virtual bool isStartRuntimeConfigurationPossible(void) const = 0;

   /**
    * Set configuration parameter during runtime (integer).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] configurationParameter: configuration parameter id
    * @param[in] callback: callback
    * @param[in] data: data
    */
   virtual void setRuntimeConfigurationParameter(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN IBasicResultCallback* callback, IN const uint64_t data) = 0;

   /**
    * Set configuration parameter during runtime (string).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] configurationParameter: configuration parameter id
    * @param[in] callback: callback
    * @param[in] data: data
    */
   virtual void setRuntimeConfigurationParameter(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN IBasicResultCallback* callback, IN const ::std::string& data) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTestTriggerIgnoreUpdate(IN const bool enable) = 0;
};

} //btstackif

#endif //_I_BASIC_CONFIGURATION_H_
