/**
 * @file BasicConfiguration.h
 *
 * @par SW-Component
 * State machine for basic configuration
 *
 * @brief Implementation of generic basic configuration state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of generic basic configuration state machine.
 */

#ifndef _BASIC_CONFIGURATION_H_
#define _BASIC_CONFIGURATION_H_

#include "StateMachine.h"
#include "IBasicConfiguration.h"
#include "IBasicConfigurationCallback.h"
#include "IExtendedTimeoutHandler.h"
#include "IConfigurationClient.h"
#include "ExtendedTimerEntry.h"
#include "BasicConfigurationData.h"

namespace btstackif {

// class forward declarations
class IBasicConfigurationItem;

/**
 * BasicConfiguration class.
 */
class BasicConfiguration : public StateMachine, public IBasicConfiguration, public IBasicConfigurationCallback, public IExtendedTimeoutHandler, public IConfigurationClient
{
public:
   /**
    * Default constructor.
    */
   BasicConfiguration();

   /**
    * Destructor.
    */
   virtual ~BasicConfiguration();

   virtual void reset(void);

   virtual void forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void setInstance(IN IBasicConfigurationRequest* instance, IN const BTSConfigurationParameter configurationParameter, IN const bool beforeBtOn, IN const bool configureDuringStartup = true);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setTimerPoolIf(IN ITimerPool* timerPool);

   virtual void setConfigurationIf(IN IConfiguration* configuration);

   virtual IConfigurationClient* getConfigurationClientHandler(void);

   virtual IStateMachine* getSmEntryInterface(void);

   virtual bool isStartRuntimeConfigurationPossible(void) const;

   virtual void setRuntimeConfigurationParameter(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN IBasicResultCallback* callback, IN const uint64_t data);

   virtual void setRuntimeConfigurationParameter(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN IBasicResultCallback* callback, IN const ::std::string& data);

   virtual void setTestTriggerIgnoreUpdate(IN const bool enable);

   virtual void updateConfigurationParameter(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSConfigurationParameter configurationParameter, IN const uint64_t data, IN const bool success);

   virtual void updateConfigurationParameter(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSConfigurationParameter configurationParameter, IN const ::std::string& data, IN const bool success);

   virtual void handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId);

   virtual void setConfigurationMasterIf(IN IConfigurationMaster* master);

   virtual void startGlobalConfiguration(void);

   virtual void stopGlobalConfiguration(void);

   virtual bool isSingleConfigurationOngoing(void) const;

   virtual bool setSingleConfiguration(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool beforeBtOn, IN const bool errorOccurred);

private:
   ::std::map< BTSConfigurationParameter, BasicConfigurationData > _requestIfList; /**< map containing request interfaces and control data */
   IBasicControl* _controlIf; /**< basic control interface */
   ITimerPool* _timerPoolIf; /**< timer pool interface */
   IConfiguration* _configurationIf; /**< configuration interface */
   IConfigurationMaster* _configurationMaster; /**< configuration master interface */
   bool _globalConfigActive; /**< flag for global configuration */
   ExtendedTimerEntry _timerConfiguration; /**< timer for configuration */
   const BTSTimeValue _timeoutConfiguration; /**< timeout for configuration */
   ::std::map< BTSConfigurationParameter, IBasicConfigurationItem* > _configItemList; /**< map containing configuration items */
   bool _testTriggerIgnoreUpdate; /**< flag for ignoring configuration parameter update */

   void startSingleConfigurationSequence(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter);

   void checkAndStartSingleConfigurationSequence(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN IBasicResultCallback* callback);

   void handleConfigurationParameterUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSConfigurationParameter configurationParameter, IN const bool success);

   void resetConfigRequestControlData(void);

   void resetConfigRequestOngoingData(void);

   bool isAnyConfigRequestOngoing(void) const;

   IBasicConfigurationItem* createConfigItem(IN const BTSConfigurationParameter configurationParameter);

   void clearConfigItemList(void);

   void startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout);

   void stopTimer(IN ExtendedTimerEntry& timer) const;

   void releaseTimer(IN ExtendedTimerEntry& timer) const;

   bool isTimerActive(IN const ExtendedTimerEntry& timer) const;

   void informInitiator(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSRequestResult result, IN IBasicResultCallback* callback) const;
};

} //btstackif

#endif //_BASIC_CONFIGURATION_H_
