/**
 * @file BluetoothStackErrorHandling.cpp
 *
 * @par SW-Component
 * State machine for Bluetooth stack error handling
 *
 * @brief Implementation of generic Bluetooth stack error handling state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic Bluetooth stack error handling state machine.
 */

#include "BluetoothStackErrorHandling.h"
#include "IBluetoothStackErrorHandlingRequest.h"
#include "IBasicControl.h"
#include "FwErrmemPrint.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "FwAssert.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BluetoothStackErrorHandling.cpp.trc.h"
#endif
#endif

namespace btstackif {

BluetoothStackErrorHandling::BluetoothStackErrorHandling() :
_requestIf(0),
_controlIf(0),
_errMemHeader(" #CONN: BtStackIf: ")
{
}

BluetoothStackErrorHandling::~BluetoothStackErrorHandling()
{
   _requestIf = 0;
   _controlIf = 0;
}

void BluetoothStackErrorHandling::reset(void)
{
   StateMachine::reset();
   // keep _requestIf
   // keep _controlIf

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);
   _requestIf->reset();
}

void BluetoothStackErrorHandling::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   // reset control data
   reset();
}

void BluetoothStackErrorHandling::setInstance(IN IBluetoothStackErrorHandlingRequest* instance)
{
   _requestIf = instance;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setCallback(this);
}

void BluetoothStackErrorHandling::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_ERRMEM_ASSERT(0 != _controlIf);
}

IStateMachine* BluetoothStackErrorHandling::getSmEntryInterface(void)
{
   return this;
}

void BluetoothStackErrorHandling::handleCriticalError(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const ::std::string& stackType, IN const ::std::string& errorType, IN const ::std::string& errorMessage, IN const bool doReset)
{
   handleError(bts2IpcMsgList, bts2AppMsgList, messageItem, stackType, errorType, errorMessage, doReset, true);
}

void BluetoothStackErrorHandling::handleGeneralError(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const ::std::string& stackType, IN const ::std::string& errorType, IN const ::std::string& errorMessage)
{
   handleError(bts2IpcMsgList, bts2AppMsgList, messageItem, stackType, errorType, errorMessage, false, true);
}

void BluetoothStackErrorHandling::handleDebugTrace(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const ::std::string& stackType, IN const ::std::string& errorType, IN const ::std::string& errorMessage, IN const bool writeToErrorMemory)
{
   handleError(bts2IpcMsgList, bts2AppMsgList, messageItem, stackType, errorType, errorMessage, false, writeToErrorMemory);
}

void BluetoothStackErrorHandling::handleOtherError(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const ::std::string& stackType, IN const ::std::string& errorType, IN const ::std::string& errorMessage)
{
   handleError(bts2IpcMsgList, bts2AppMsgList, messageItem, stackType, errorType, errorMessage, false, true);
}

void BluetoothStackErrorHandling::handleError(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const ::std::string& stackType, IN const ::std::string& errorType, IN const ::std::string& errorMessage, IN const bool doReset, IN const bool writeToErrorMemory)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   // NOTE: step collect helpful information e.g. name of connected mobile phone

   // data length: "stackType: errorType: errorMessage"
   const size_t dataLength(stackType.size() + 2 + errorType.size() + 2 + errorMessage.size());
   const size_t maxTraceLength(220);
   char buffer[1 + maxTraceLength];
   size_t length;

   // write entry to error memory
   if(maxTraceLength >= (_errMemHeader.size() + dataLength))
   {
      // only 1 trace line needed
      (void)snprintf(buffer, sizeof(buffer), "%s: %s: %s", stackType.c_str(), errorType.c_str(), errorMessage.c_str());

      // no more data to be printed
      length = 0;
   }
   else
   {
      // more than 1 trace line needed
      // assuming that printing stack type and error type does not exceed the limit
      // add "[0]: "
      (void)snprintf(buffer, sizeof(buffer), "[0]: %s: %s", stackType.c_str(), errorType.c_str());

      // print error message
      length = errorMessage.size();
   }

   if(true == writeToErrorMemory)
   {
      ETG_TRACE_ERRMEM((" #CONN: BtStackIf: %s", buffer));
   }
   else
   {
      ETG_TRACE_FATAL((" #CONN: BtStackIf: %s", buffer));
   }

   const size_t printLength((maxTraceLength - _errMemHeader.size()) - 5);
   size_t count(1);
   size_t pos(0);

   // print error message
   while((length > 0) && (pos < errorMessage.size()) && (count < 10))
   {
      (void)snprintf(buffer, sizeof(buffer), "[%u]: %s", (unsigned int)count, errorMessage.substr(pos, printLength).c_str());

      if(length >= printLength)
      {
         length -= printLength;
      }
      else
      {
         length = 0;
      }
      pos += printLength;
      ++count;

      if(true == writeToErrorMemory)
      {
         ETG_TRACE_ERRMEM((" #CONN: BtStackIf: %s", buffer));
      }
      else
      {
         ETG_TRACE_FATAL((" #CONN: BtStackIf: %s", buffer));
      }
   }

   // check for triggering system reset
   if(true == doReset)
   {
      FW_FATAL_ASSERT_ALWAYS();
   }
}

} //btstackif
