/**
 * @file IBluetoothStackErrorHandlingCallback.h
 *
 * @par SW-Component
 * State machine for Bluetooth stack error handling
 *
 * @brief Callback interface definition for Bluetooth stack error handling state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the callback interface to be used by specific Bluetooth stack error handling state machine.
 *          Callback interface must be provided by generic Bluetooth stack error handling state machine.
 */

#ifndef _I_BLUETOOTH_STACK_ERROR_HANDLING_CALLBACK_H_
#define _I_BLUETOOTH_STACK_ERROR_HANDLING_CALLBACK_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class IBluetoothStackErrorHandlingCallback
{
public:
   /**
    * Destructor.
    */
   virtual ~IBluetoothStackErrorHandlingCallback() {}

   /**
    * Handle critical error.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] stackType: stack type
    * @param[in] errorType: error type
    * @param[in] errorMessage: error message
    * @param[in] doReset: reset flag
    */
   virtual void handleCriticalError(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const ::std::string& stackType, IN const ::std::string& errorType, IN const ::std::string& errorMessage, IN const bool doReset) = 0;

   /**
    * Handle general error.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] stackType: stack type
    * @param[in] errorType: error type
    * @param[in] errorMessage: error message
    */
   virtual void handleGeneralError(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const ::std::string& stackType, IN const ::std::string& errorType, IN const ::std::string& errorMessage) = 0;

   /**
    * Handle debug trace.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] stackType: stack type
    * @param[in] errorType: error type
    * @param[in] errorMessage: error message
    * @param[in] writeToErrorMemory: error memory flag
    */
   virtual void handleDebugTrace(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const ::std::string& stackType, IN const ::std::string& errorType, IN const ::std::string& errorMessage, IN const bool writeToErrorMemory) = 0;

   /**
    * Handle other error.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] stackType: stack type
    * @param[in] errorType: error type
    * @param[in] errorMessage: error message
    */
   virtual void handleOtherError(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const ::std::string& stackType, IN const ::std::string& errorType, IN const ::std::string& errorMessage) = 0;
};

} //btstackif

#endif //_I_BLUETOOTH_STACK_ERROR_HANDLING_CALLBACK_H_
