/**
 * @file BasicControl.cpp
 *
 * @par SW-Component
 * Main
 *
 * @brief Basic Control.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of basic control functionality.
 */

#include "BasicControl.h"
#include "App2Bts_BaseMessage.h"
#include "Ipc2Bts_BaseMessage.h"
#include "Bts2Ipc_BaseMessage.h"
#include "Bts2App_BaseMessage.h"
#include "IMainControl.h"
#include "IControlHandler.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BasicControl.cpp.trc.h"
#endif
#endif

//#define BTS_ENABLE_EXTENDED_DEBUG
#undef BTS_ENABLE_EXTENDED_DEBUG

namespace btstackif {

const bool BasicControl::_handleDoubledRequests = true;
const bool BasicControl::_reorderWaitingRequests = true;

BasicControl::BasicControl() :
_functionBlock(BTS_FB_LAST),
_extensions(),
_component(BTS_FB_LAST),
_stackInterface(BTS_IF_LAST)
{
   _bts2IpcMessageWasSent = false;
   _checkWaitingQueue = false;
   _ptrApp2BtsMessage = NULL;
   _mainControl = NULL;
}

BasicControl::BasicControl(IN const BTSFunctionBlock functionBlock) :
_functionBlock(functionBlock),
_extensions(),
_component(BTS_FB_LAST),
_stackInterface(BTS_IF_LAST)
{
   _bts2IpcMessageWasSent = false;
   _checkWaitingQueue = false;
   _ptrApp2BtsMessage = NULL;
   _mainControl = NULL;
}

BasicControl::BasicControl(const BasicControl& ref) :
_functionBlock(BTS_FB_LAST),
_extensions(),
_component(BTS_FB_LAST),
_stackInterface(BTS_IF_LAST)
{
   // ignore given parameter
   (void)(ref);

   // DO NOT USE!!!
   FW_NORMAL_ASSERT_ALWAYS();

   _bts2IpcMessageWasSent = false;
   _checkWaitingQueue = false;
   _ptrApp2BtsMessage = NULL;
   _mainControl = NULL;
}

BasicControl& BasicControl::operator=(const BasicControl& ref)
{
   // DO NOT USE!!!
   FW_NORMAL_ASSERT_ALWAYS();

   if(this == &ref)
   {
      return *this;
   }

   // ignore given parameter

   return *this;
}

BasicControl::~BasicControl()
{
   _app2BtsWaitingQueue.empty();
   _app2BtsWorkingQueue.empty();
   _ptrApp2BtsMessage = NULL;
   _mainControl = NULL;
}

void BasicControl::setMainControl(IN IMainControl* mainControl)
{
   if(_mainControl != mainControl)
   {
      _mainControl = mainControl;
   }
}

void BasicControl::setComponent(IN const BTSFunctionBlock component)
{
   _component = component;
}

void BasicControl::setStackInterface(IN const BTSInterfaceType stackInterface)
{
   _stackInterface = stackInterface;
}

void BasicControl::setQueueWarningSize(IN const unsigned int maxSize)
{
   _app2BtsWaitingQueue.setWarningSize(maxSize);
   _app2BtsWorkingQueue.setWarningSize(maxSize);
}

void BasicControl::pushApp2BtsMessage(IN App2Bts_BaseMessage* ptrMessage)
{
   if(NULL != ptrMessage)
   {
      delete ptrMessage;
   }
}

void BasicControl::pushIpc2BtsMessage(IN Ipc2Bts_BaseMessage* ptrMessage)
{
   if(NULL != ptrMessage)
   {
      delete ptrMessage;
   }
}

void BasicControl::setStackConfiguration(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
         OUT ::std::vector<BTSDbusInterfaceItem>& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration)
{
   (void)(component);
   (void)(stackInterface);
   (void)(subComponent);
   (void)(userMode);
   (void)(dbusInterfaces);
   (void)(configuration);
}

void BasicControl::triggerInitializedCallback(void)
{
}

void BasicControl::createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent)
{
   (void)(interface);
   (void)(availabilityEvent);
}

void BasicControl::createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent, IN const BTSBusName& busName, IN const BTSObjectPath& objPath, IN const BTSCommonEnumClass busType)
{
   (void)(interface);
   (void)(availabilityEvent);
   (void)(busName);
   (void)(objPath);
   (void)(busType);
}

void BasicControl::sendInternalApp2BtsMessage(IN App2Bts_BaseMessage* ptrMessage, IN const bool highPrio /*= false*/)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if(0 != _mainControl)
   {
      // add component and stack interface, sub component is part of message opcode
      ptrMessage->setComponent(_component);
      ptrMessage->setStackInterface(_stackInterface);
      _mainControl->pushApp2BtsMessage(ptrMessage, highPrio);
   }
   else
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
   }
}

void BasicControl::sendInternalBts2AppMessage(IN Bts2App_BaseMessage* ptrMessage)
{
   pushBts2AppMessage(ptrMessage);
}

void BasicControl::sendInternalIpc2BtsMessage(IN Ipc2Bts_BaseMessage* ptrMessage, IN const bool highPrio /*= false*/)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if(0 != _mainControl)
   {
      // add component and stack interface, sub component is part of message opcode
      ptrMessage->setComponent(_component);
      ptrMessage->setStackInterface(_stackInterface);
      _mainControl->pushIpc2BtsMessage(ptrMessage, highPrio);
   }
   else
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
   }
}

void BasicControl::sendInternalBts2IpcMessage(IN Bts2Ipc_BaseMessage* ptrMessage)
{
   pushBts2IpcMessage(ptrMessage);
}

void BasicControl::sendBts2IpcMessageList(IN ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSApp2BtsMessageCompareItem& item)
{
   if(0 < bts2IpcMsgList.size())
   {
      for(size_t i = 0; i < bts2IpcMsgList.size(); i++)
      {
         if(NULL != bts2IpcMsgList[i])
         {
            // message was created successfully
            bts2IpcMsgList[i]->setBtsSourceFunctionBlock(_functionBlock);
            bts2IpcMsgList[i]->setApp2BtsCompareItem(item);

            // forward to IPC handler
            pushBts2IpcMessage(bts2IpcMsgList[i]);
         }
      }

      bts2IpcMsgList.clear();
   }
}

void BasicControl::sendBts2AppMessageList(IN ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList)
{
   if(0 < bts2AppMsgList.size())
   {
      for(size_t i = 0; i < bts2AppMsgList.size(); i++)
      {
         if(NULL != bts2AppMsgList[i])
         {
            // message was created successfully

            // forward to main control
            pushBts2AppMessage(bts2AppMsgList[i]);
         }
      }

      bts2AppMsgList.clear();
   }
}

void BasicControl::handleTimeout(IN IExtendedTimeoutHandler* handler, IN const BTSTimerId timerId)
{
   (void)(handler);
   (void)(timerId);
}

void BasicControl::addTTYMonitorHandler(IN ITTYDeviceMonitorHandler* handler, IN const ::std::string& filter)
{
   FW_IF_NULL_PTR_RETURN(_mainControl);

   _mainControl->addTTYMonitorHandler(handler, filter);
}

void BasicControl::printQueueStatistics(void)
{
   size_t i;

   // _app2BtsWaitingQueue
   ETG_TRACE_USR1((" ***BasicControl[%d]::_app2BtsWaitingQueue.getSize()=%u", _functionBlock, _app2BtsWaitingQueue.getSize()));
   i = 0;
   for(MessageQueue<App2Bts_BaseMessage>::It it = _app2BtsWaitingQueue.getBegin(); it != _app2BtsWaitingQueue.getEnd(); ++it)
   {
      ETG_TRACE_USR1((" ***BasicControl[%d]::_app2BtsWaitingQueue[%u]: App2Bts 0x%04X", _functionBlock, i, (*it)->getTraceOpCode()));
      i++;
   }

   // _app2BtsWorkingQueue
   ETG_TRACE_USR1((" ***BasicControl[%d]::_app2BtsWorkingQueue.getSize()=%u", _functionBlock, _app2BtsWorkingQueue.getSize()));
   i = 0;
   for(MessageQueue<App2Bts_BaseMessage>::It it = _app2BtsWorkingQueue.getBegin(); it != _app2BtsWorkingQueue.getEnd(); ++it)
   {
      ETG_TRACE_USR1((" ***BasicControl[%d]::_app2BtsWorkingQueue[%u]: App2Bts 0x%04X", _functionBlock, i, (*it)->getTraceOpCode()));
      i++;
   }
}

void BasicControl::setSubControlTestCommand(IN const char* testCommand, IN const unsigned int testData)
{
   (void)(testCommand);
   (void)(testData);
}

void BasicControl::setSubControlTestCommand(IN const char* testCommand, IN const unsigned char* testData)
{
   (void)(testCommand);
   (void)(testData);
}

void BasicControl::checkWaitingQueue(IN const App2Bts_BaseMessage* ptrMessage)
{
   if(NULL == ptrMessage)
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   ETG_TRACE_USR3((" checkWaitingQueue(): App2Bts 0x%04X to be checked", ptrMessage->getTraceOpCode()));

#ifdef BTS_ENABLE_EXTENDED_DEBUG
   BTSApp2BtsMessageCompareItem compareItem;
   ptrMessage->getCompareItem(compareItem);
   ETG_TRACE_USR3((" checkWaitingQueue(): opCode=0x%04X", (BTSTraceOpCode)compareItem.opCode));
   ETG_TRACE_USR3((" checkWaitingQueue(): deviceAddress=%s", compareItem.deviceAddress.c_str()));
   ETG_TRACE_USR3((" checkWaitingQueue(): protocolId=%d", compareItem.protocolId));
   ETG_TRACE_USR3((" checkWaitingQueue(): sppUuid=%s", compareItem.sppUuid.c_str()));
   ETG_TRACE_USR3((" checkWaitingQueue(): searchType=%d", compareItem.searchType));
#endif

   // get similar opcodes
   ::std::vector<BTSApp2BtsMessageMasterCompareItem> itemList;
   getSimilarOpCodes(itemList, ptrMessage);

   for(size_t i = 0; i < itemList.size(); i++)
   {
      ETG_TRACE_USR3((" checkWaitingQueue(): similar opcodes: itemList[%u]: App2Bts 0x%04X", i, (BTSTraceOpCode)itemList[i].opCode));
   }

   bool bts2IpcMessageSent = false;
   size_t counter = 0;
   bool errorMsgPrinted = false;
   const size_t nmbWaitingQueueEntries = _app2BtsWaitingQueue.getSize();
   ::std::vector< BTSTraceOpCode > traceOpCodeListStart;

   traceOpCodeListStart.reserve(nmbWaitingQueueEntries);
   for(MessageQueue< App2Bts_BaseMessage >::It it = _app2BtsWaitingQueue.getBegin(); it != _app2BtsWaitingQueue.getEnd(); ++it)
   {
      if(0 != (*it))
      {
         traceOpCodeListStart.push_back((*it)->getTraceOpCode());
      }
   }

   // repeat check until Bts2Ipc message was sent
   while((false == bts2IpcMessageSent) /*&& (counter <= itemList.size())*/) // if matching item list also depends on address then we will "loop" several times through itemList
   {
      bts2IpcMessageSent = true;

      // check if opcode is in waiting queue
      unsigned int index = 0;
      if(true == isOpCodeInWaitingQueue(index, itemList))
      {
         ETG_TRACE_USR3((" checkWaitingQueue(): similar opcode found: index=%u", index));

         const size_t nmbWaitingQueueEntriesStart(_app2BtsWaitingQueue.getSize());
         const size_t nmbWorkingQueueEntriesStart(_app2BtsWorkingQueue.getSize());
         App2Bts_BaseMessage* ptrMessage = NULL;
         unsigned int i = 0;
         MessageQueue<App2Bts_BaseMessage>::It msgIt;

         // get related message
         for(msgIt = _app2BtsWaitingQueue.getBegin(); msgIt != _app2BtsWaitingQueue.getEnd(); ++msgIt)
         {
            if(index == i)
            {
               ptrMessage = (*msgIt);
               break;
            }
            i++;
         }

         // check waiting queue if there are messages with same opcode
         if(NULL != ptrMessage)
         {
            ::std::vector<App2Bts_BaseMessage*> msgList;
            msgList.reserve(_app2BtsWaitingQueue.getSize());

            App2Bts_BaseMessage* ptrNextMessage = NULL;

            ::std::vector<BTSApp2BtsMessageMasterCompareItem> matchingItemList;
            ::std::vector<BTSApp2BtsMessageMasterCompareItem> highPrioItemList;
            getMatchingOpCodes(matchingItemList, highPrioItemList, ptrMessage);
#ifdef BTS_ENABLE_EXTENDED_DEBUG
            for(size_t i = 0; i < matchingItemList.size(); i++)
            {
               ETG_TRACE_USR3((" checkWaitingQueue(): matching opcodes: matchingItemList[%u]: App2Bts 0x%04X (compareOpCode=%d compareDeviceAddress=%d compareProtocolId=%d compareSppUuid=%d compareSearchType=%d)", i, (BTSTraceOpCode)matchingItemList[i].opCode, matchingItemList[i].compareOpCode, matchingItemList[i].compareDeviceAddress, matchingItemList[i].compareProtocolId, matchingItemList[i].compareSppUuid, matchingItemList[i].compareSearchType));
            }
#endif
            bool highPrioItemFound = getMatchingWaitingQueueEntries(msgList, matchingItemList, highPrioItemList);
#ifdef BTS_ENABLE_EXTENDED_DEBUG
            ETG_TRACE_USR3((" checkWaitingQueue(): msgList.size()=%u", msgList.size()));
#endif

            // check if another message was found
            if(0 == msgList.size())
            {
               // at least the message used for searching must be in the list
               // should never happen else you have programmed something wrong
               // #error_indication
               FW_NORMAL_ASSERT_ALWAYS();
            }
            else if(1 == msgList.size())
            {
               // only message used for searching is in the list
               // use this one as message to be processed
               ptrNextMessage = msgList[0];

               ETG_TRACE_USR3((" checkWaitingQueue(): 1 App2Bts messages found"));
            }
            else
            {
               // more messages are in the list
               // use latest as message to be processed
               ptrNextMessage = msgList[msgList.size() - 1];

               ETG_TRACE_USR3((" checkWaitingQueue(): %u App2Bts messages found", msgList.size()));

               // handle other messages
               ::std::vector<App2Bts_BaseMessage*> doubledMsgList(msgList.begin(), (msgList.end() - 1));
               handleDoubledApp2BtsMessages(doubledMsgList);

               // messages have to be deleted
               for(size_t i = 0; i < doubledMsgList.size(); i++)
               {
                  if(NULL != doubledMsgList[i])
                  {
                     delete doubledMsgList[i];
                  }
               }
            }

            removeApp2BtsWaitingMessages(msgList);

            if(NULL != ptrNextMessage)
            {
               // process next message
               _bts2IpcMessageWasSent = false;

               BTSTraceOpCode traceOpCode = ptrNextMessage->getTraceOpCode();
#ifdef FEATURE_DISABLED
               bool waitingRequestfound = false;
#endif

               if(true == highPrioItemFound)
               {
                  // all messages have to be handled as doubled
                  ::std::vector<App2Bts_BaseMessage*> doubledMsgList(1, ptrNextMessage);
                  handleDoubledApp2BtsMessages(doubledMsgList);
                  // delete message to be processed
                  delete ptrNextMessage;
               }
               else
               {
                  pushApp2BtsMessage(ptrNextMessage); // this works only if single thread handling
                  // message to be processed was deleted within pushApp2BtsMessage()
#ifdef FEATURE_DISABLED
                  waitingRequestfound = isMatchingFrontApp2BtsWaitingQueueMessage(ptrNextMessage); // only pointer is checked; not the content
#endif
               }

               bts2IpcMessageSent = _bts2IpcMessageWasSent;

               if(false == bts2IpcMessageSent)
               {
                  // no message was sent to stack, can happen in case of timer was started, check size of waiting and working queue
                  if((nmbWaitingQueueEntriesStart > _app2BtsWaitingQueue.getSize()) && (nmbWorkingQueueEntriesStart < _app2BtsWorkingQueue.getSize()))
                  {
                     ETG_TRACE_USR3((" checkWaitingQueue(): no message sent to stack: App2Bts 0x%04X", traceOpCode));
                     bts2IpcMessageSent = true;
                  }
               }

#ifdef FEATURE_DISABLED
               // either message was sent to stack or request was split into 2 separate App2Bts messages (1->high prio App2Bts request, 2->App2Bts waiting message (on top))
               // in this case no message was sent to stack
               if(true == bts2IpcMessageSent)
               {
                  if(true == waitingRequestfound)
                  {
                     // should never happen
                     FW_NORMAL_ASSERT_ALWAYS();
                  }
               }
               else
               {
                  if(true == waitingRequestfound)
                  {
                     bts2IpcMessageSent = true;
                     ETG_TRACE_USR3((" checkWaitingQueue(): match for front waiting message: App2Bts 0x%04X", traceOpCode));
                  }
               }
#endif
            }

            msgList.clear();
         }
         else
         {
            // should never happen else you have programmed something wrong
            // #error_indication
            FW_NORMAL_ASSERT_ALWAYS();
         }
      }
      else
      {
         ETG_TRACE_USR3((" checkWaitingQueue(): similar opcode not found"));
      }

      counter++;

      if((false == bts2IpcMessageSent) && (nmbWaitingQueueEntries < counter) && (false == errorMsgPrinted))
      {
         // "we went through complete list of waiting queue entries" => therefore we should not reach this line here
         // something went wrong => print all helpful information

         ::std::vector< BTSTraceOpCode > traceOpCodeListEnd;
         traceOpCodeListEnd.reserve(_app2BtsWaitingQueue.getSize());
         for(MessageQueue< App2Bts_BaseMessage >::It it = _app2BtsWaitingQueue.getBegin(); it != _app2BtsWaitingQueue.getEnd(); ++it)
         {
            if(0 != (*it))
            {
               traceOpCodeListEnd.push_back((*it)->getTraceOpCode());
            }
         }

         ::std::vector< BTSTraceOpCode > traceOpCodeListWorking;
         _app2BtsWorkingQueue.lockAccess();
         traceOpCodeListWorking.reserve(_app2BtsWorkingQueue.getSize());
         for(MessageQueue< App2Bts_BaseMessage >::It it = _app2BtsWorkingQueue.getBegin(); it != _app2BtsWorkingQueue.getEnd(); ++it)
         {
            if(0 != (*it))
            {
               traceOpCodeListWorking.push_back((*it)->getTraceOpCode());
            }
         }
         _app2BtsWorkingQueue.unlockAccess();

         ETG_TRACE_ERRMEM((" #CONN: BtStackIf: checkWaitingQueue(): App2Bts 0x%04X to be checked, traceOpCodeListStart.size()=%u traceOpCodeListEnd.size()=%u traceOpCodeListWorking.size()=%u", ptrMessage->getTraceOpCode(), (unsigned int)traceOpCodeListStart.size(), (unsigned int)traceOpCodeListEnd.size(), (unsigned int)traceOpCodeListWorking.size()));
         for(size_t i = 0; i < traceOpCodeListStart.size(); i++)
         {
            ETG_TRACE_ERRMEM((" #CONN: BtStackIf: checkWaitingQueue(): traceOpCodeListStart[%u]=0x%04X", (unsigned int)i, traceOpCodeListStart[i]));
         }
         for(size_t i = 0; i < traceOpCodeListEnd.size(); i++)
         {
            ETG_TRACE_ERRMEM((" #CONN: BtStackIf: checkWaitingQueue(): traceOpCodeListEnd[%u]=0x%04X", (unsigned int)i, traceOpCodeListEnd[i]));
         }
         for(size_t i = 0; i < traceOpCodeListWorking.size(); i++)
         {
            ETG_TRACE_ERRMEM((" #CONN: BtStackIf: checkWaitingQueue(): traceOpCodeListWorking[%u]=0x%04X", (unsigned int)i, traceOpCodeListWorking[i]));
         }

         // print error messages only once
         errorMsgPrinted = true;
      }

      // check for breaking the loop
      if((false == bts2IpcMessageSent) && (0 < nmbWaitingQueueEntries) && ((nmbWaitingQueueEntries << 1) == counter))
      {
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      }

#if 0
      if((false == bts2IpcMessageSent) && (counter > itemList.size()))
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
      }
#endif
   }
}

void BasicControl::removeApp2BtsWaitingMessages(IN const ::std::vector< App2Bts_BaseMessage* >& app2BtsMsgList)
{
   for(size_t i = 0; i < app2BtsMsgList.size(); i++)
   {
      App2Bts_BaseMessage* ptrMessage = app2BtsMsgList[i];

      if(NULL != ptrMessage)
      {
         MessageQueue<App2Bts_BaseMessage>::It it = _app2BtsWaitingQueue.getBegin();

         while(it != _app2BtsWaitingQueue.getEnd())
         {
            if(*it == ptrMessage)
            {
               break;
            }

            ++it;
         }

         if(it != _app2BtsWaitingQueue.getEnd())
         {
            _app2BtsWaitingQueue.doErase(it);
         }
      }
   }
}

bool BasicControl::getMatchingWaitingQueueEntries(OUT ::std::vector< App2Bts_BaseMessage* >& msgList, IN const ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, IN const ::std::vector< BTSApp2BtsMessageMasterCompareItem >& highPrioItemList)
{
   bool highPrioItemFound = false;

   for(MessageQueue<App2Bts_BaseMessage>::It it = _app2BtsWaitingQueue.getBegin(); it != _app2BtsWaitingQueue.getEnd(); ++it)
   {
      BTSApp2BtsMessageCompareItem item;
      (*it)->getCompareItem(item);

#ifdef BTS_ENABLE_EXTENDED_DEBUG
      ETG_TRACE_USR3((" getMatchingWaitingQueueEntries(): opCode=0x%04X", (BTSTraceOpCode)item.opCode));
      ETG_TRACE_USR3((" getMatchingWaitingQueueEntries(): deviceAddress=%s", item.deviceAddress.c_str()));
      ETG_TRACE_USR3((" getMatchingWaitingQueueEntries(): protocolId=%d", item.protocolId));
      ETG_TRACE_USR3((" getMatchingWaitingQueueEntries(): sppUuid=%s", item.sppUuid.c_str()));
      ETG_TRACE_USR3((" getMatchingWaitingQueueEntries(): searchType=%d", item.searchType));
#endif

#if 1
      // check for high priority item
      bool found = false;
      for(size_t i = 0; i < highPrioItemList.size(); i++)
      {
         if(true == highPrioItemList[i].isMatch(item))
         {
            // do not check further entries
            found = true;
            highPrioItemFound = true;
            break;
         }
      }
#endif

      for(size_t i = 0; i < itemList.size(); i++)
      {
         if(true == itemList[i].isMatch(item))
         {
            msgList.push_back((*it));
#ifdef BTS_ENABLE_EXTENDED_DEBUG
            ETG_TRACE_USR3((" getMatchingWaitingQueueEntries(): match for opCode=0x%04X", (BTSTraceOpCode)item.opCode));
#endif
            // check for high priority item: if same as normal item then message has to be handled via normal mechanism
            if(true == highPrioItemFound)
            {
               highPrioItemFound = false;
            }

            // end loop now to avoid pushing same message several times
            break;
         }
#ifdef BTS_ENABLE_EXTENDED_DEBUG
         else
         {
            ETG_TRACE_USR3((" getMatchingWaitingQueueEntries(): no match for opCode=0x%04X", (BTSTraceOpCode)item.opCode));
         }
#endif
      }

#if 1
      if(true == found)
      {
         // do not check further entries
         ETG_TRACE_USR3((" getMatchingWaitingQueueEntries(): break => do not check further entries (highPrioItemFound=%d)", highPrioItemFound));
         break;
      }
#endif
   }

#ifdef BTS_ENABLE_EXTENDED_DEBUG
   ETG_TRACE_USR3((" getMatchingWaitingQueueEntries(): msgList.size()=%u", msgList.size()));
#endif

   return highPrioItemFound;
}

App2Bts_BaseMessage* BasicControl::findApp2BtsWorkingMessageWrapper(IN const BTSApp2BtsMessageCompareItem& item)
{
   BTSApp2BtsMessageMasterCompareItem masterCompareItem;
   // compare all
   masterCompareItem.compareOpCode = true;
   masterCompareItem.compareDeviceAddress = true;
   masterCompareItem.compareProtocolId = true;
   masterCompareItem.compareSppUuid = true;
   masterCompareItem.compareMasInstance = true;
   masterCompareItem.compareSearchType = true;

   masterCompareItem.opCode = item.opCode;
   masterCompareItem.deviceAddress = item.deviceAddress;
   masterCompareItem.protocolId = item.protocolId;
   masterCompareItem.sppUuid = item.sppUuid;
   masterCompareItem.masInstance = item.masInstance;
   masterCompareItem.searchType = item.searchType;

   App2Bts_BaseMessage* ptrMessage = findApp2BtsWorkingMessage(masterCompareItem);

   if(NULL != ptrMessage)
   {
      ETG_TRACE_USR3((" findApp2BtsWorkingMessage(): App2Bts 0x%04X found in working queue", ptrMessage->getTraceOpCode()));
   }
   else
   {
      ETG_TRACE_USR3((" findApp2BtsWorkingMessage(): App2Bts message not found in working queue"));
   }

   return ptrMessage;
}

void BasicControl::pushApp2BtsMsgToWorkingQueue(IN App2Bts_BaseMessage* message, bool withLock /*= true*/)
{
   FW_IF_NULL_PTR_RETURN(message);

   _app2BtsWorkingQueue.push(message, withLock);
}

void BasicControl::pushOnTopApp2BtsMsgToWorkingQueue(IN App2Bts_BaseMessage* message, bool withLock /*= true*/)
{
   FW_IF_NULL_PTR_RETURN(message);

   _app2BtsWorkingQueue.pushOnTop(message, withLock);
}

void BasicControl::pushApp2BtsMsgToWaitingQueue(IN App2Bts_BaseMessage* message, bool withLock /*= true*/)
{
   FW_IF_NULL_PTR_RETURN(message);

   _app2BtsWaitingQueue.push(message, withLock);

   if(true == _reorderWaitingRequests)
   {
      if(true == withLock)
      {
         _app2BtsWaitingQueue.lockAccess();
      }

      // given chance to reorder waiting requests
      reorderApp2BtsWaitingQueue(_app2BtsWaitingQueue);

      if(true == withLock)
      {
         _app2BtsWaitingQueue.unlockAccess();
      }
   }
}

void BasicControl::pushOnTopApp2BtsMsgToWaitingQueue(IN App2Bts_BaseMessage* message, bool withLock /*= true*/)
{
   FW_IF_NULL_PTR_RETURN(message);

   _app2BtsWaitingQueue.pushOnTop(message, withLock);
}

unsigned int BasicControl::getApp2BtsWorkingQueueSize(void) const
{
   return _app2BtsWorkingQueue.getSize();
}

unsigned int BasicControl::getApp2BtsWaitingQueueSize(void) const
{
   return _app2BtsWaitingQueue.getSize();
}

bool BasicControl::isSimilarOpCodeInWaitingQueue(IN const ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList) const
{
   for(MessageQueue<App2Bts_BaseMessage>::ConstIt it = _app2BtsWaitingQueue.getBegin(); it != _app2BtsWaitingQueue.getEnd(); ++it)
   {
      BTSApp2BtsMessageCompareItem item;
      (*it)->getCompareItem(item);

      for(size_t i = 0; i < itemList.size(); i++)
      {
         if(true == itemList[i].isMatch(item))
         {
            return true;
         }
      }
   }

   return false;
}

bool BasicControl::isSimilarOpCodeInWorkingQueue(IN const ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList) const
{
   for(MessageQueue<App2Bts_BaseMessage>::ConstIt it = _app2BtsWorkingQueue.getBegin(); it != _app2BtsWorkingQueue.getEnd(); ++it)
   {
      BTSApp2BtsMessageCompareItem item;
      (*it)->getCompareItem(item);

      for(size_t i = 0; i < itemList.size(); i++)
      {
         if(true == itemList[i].isMatch(item))
         {
            return true;
         }
      }
   }

   return false;
}

void BasicControl::sendDirectAnswerForApp2BtsMessages(IN const ::std::vector<App2Bts_BaseMessage*>& msgList, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(msgList);
   (void)(resultCode);
   (void)(statusCode);
   // if needed implement in child class
}

void BasicControl::sendDirectAnswerForApp2BtsMessagesWrapper(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN const ::std::vector<App2Bts_BaseMessage*>& msgList, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   (void)(bts2AppMsgList);
   (void)(msgList);
   (void)(resultCode);
   (void)(statusCode);
   // if needed implement in child class
}

void BasicControl::getWorkingQueueEntries(OUT ::std::vector<App2Bts_BaseMessage*>& msgList, IN bool withLock /*= true*/)
{
   if(true == withLock)
   {
      _app2BtsWorkingQueue.lockAccess();
   }

   msgList.reserve(_app2BtsWorkingQueue.getSize());

   for(MessageQueue<App2Bts_BaseMessage>::It it = _app2BtsWorkingQueue.getBegin(); it != _app2BtsWorkingQueue.getEnd(); ++it)
   {
      msgList.push_back((*it));
   }

   if(true == withLock)
   {
      _app2BtsWorkingQueue.unlockAccess();
   }
}

void BasicControl::getWaitingQueueEntries(OUT ::std::vector<App2Bts_BaseMessage*>& msgList, IN bool withLock /*= true*/)
{
   if(true == withLock)
   {
      _app2BtsWaitingQueue.lockAccess();
   }

   msgList.reserve(_app2BtsWaitingQueue.getSize());

   for(MessageQueue<App2Bts_BaseMessage>::It it = _app2BtsWaitingQueue.getBegin(); it != _app2BtsWaitingQueue.getEnd(); ++it)
   {
      msgList.push_back((*it));
   }

   if(true == withLock)
   {
      _app2BtsWaitingQueue.unlockAccess();
   }
}

void BasicControl::emptyWorkingQueue(IN bool withLock /*= true*/)
{
   _app2BtsWorkingQueue.empty(withLock);
}

void BasicControl::emptyWaitingQueue(IN bool withLock /*= true*/)
{
   _app2BtsWaitingQueue.empty(withLock);
}

void BasicControl::triggerResetOfDbusIfHandler(void)
{
   FW_IF_NULL_PTR_RETURN(_mainControl);

   _mainControl->triggerResetOfDbusIfHandler(_component, _stackInterface, _functionBlock);
}

void BasicControl::reorderApp2BtsWaitingQueue(INOUT MessageQueue< App2Bts_BaseMessage >& waitingQueue)
{
   (void)(waitingQueue);
   // if needed implement in child class
}

bool BasicControl::doApp2BtsMsgPrecheck(OUT bool& rejectRequest, OUT BTSCommonEnumClass& resultCode, OUT BTSCommonEnumClass& statusCode, OUT bool& skipOpCodeCheck, IN App2Bts_BaseMessage* ptrMessage)
{
   rejectRequest = false;
   resultCode = BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE;
   statusCode = BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE;
   skipOpCodeCheck = false;
   (void)(ptrMessage);
   return true;
}

App2Bts_BaseMessage* BasicControl::findApp2BtsWaitingMessage(IN const BTSApp2BtsOpcode opcode)
{
   // old: for(::std::list<App2Bts_BaseMessage*>::iterator it = _app2BtsWaitingQueue.getBegin(); it != _app2BtsWaitingQueue.getEnd(); ++it)
   for(MessageQueue<App2Bts_BaseMessage>::It it = _app2BtsWaitingQueue.getBegin(); it != _app2BtsWaitingQueue.getEnd(); ++it)
   {
      if((*it)->getOpCode() == opcode)
      {
         return (*it);
      }
   }

   return NULL;
}

void BasicControl::removeAndDeleteApp2BtsWaitingMessage(IN App2Bts_BaseMessage* ptrMessage)
{
   if(NULL != ptrMessage)
   {
      MessageQueue<App2Bts_BaseMessage>::It it = _app2BtsWaitingQueue.getBegin();

      while(it != _app2BtsWaitingQueue.getEnd())
      {
         if(*it == ptrMessage)
         {
            break;
         }

         ++it;
      }

      if(it != _app2BtsWaitingQueue.getEnd())
      {
         _app2BtsWaitingQueue.doErase(it);
      }

      delete ptrMessage;
   }
}

App2Bts_BaseMessage* BasicControl::findApp2BtsWorkingMessage(IN const BTSApp2BtsMessageMasterCompareItem& masterItem)
{
   for(MessageQueue<App2Bts_BaseMessage>::It it = _app2BtsWorkingQueue.getBegin(); it != _app2BtsWorkingQueue.getEnd(); ++it)
   {
      BTSApp2BtsMessageCompareItem item;
      (*it)->getCompareItem(item);

      if(true == masterItem.isMatch(item))
      {
         return (*it);
      }
   }

   return NULL;
}

void BasicControl::removeAndDeleteApp2BtsWorkingMessage(IN App2Bts_BaseMessage* ptrMessage)
{
   if(NULL != ptrMessage)
   {
      MessageQueue<App2Bts_BaseMessage>::It it = _app2BtsWorkingQueue.getBegin();

      while(it != _app2BtsWorkingQueue.getEnd())
      {
         if(*it == ptrMessage)
         {
            break;
         }

         ++it;
      }

      if(it != _app2BtsWorkingQueue.getEnd())
      {
         _app2BtsWorkingQueue.doErase(it);
      }

      delete ptrMessage;
   }
}

void BasicControl::removeApp2BtsWaitingMessage(IN const App2Bts_BaseMessage* ptrMessage)
{
   if(NULL != ptrMessage)
   {
      MessageQueue<App2Bts_BaseMessage>::It it = _app2BtsWaitingQueue.getBegin();

      while(it != _app2BtsWaitingQueue.getEnd())
      {
         if(*it == ptrMessage)
         {
            break;
         }

         ++it;
      }

      if(it != _app2BtsWaitingQueue.getEnd())
      {
         _app2BtsWaitingQueue.doErase(it);
      }
   }
}

void BasicControl::removeApp2BtsWorkingMessage(IN const App2Bts_BaseMessage* ptrMessage)
{
   if(NULL != ptrMessage)
   {
      MessageQueue<App2Bts_BaseMessage>::It it = _app2BtsWorkingQueue.getBegin();

      while(it != _app2BtsWorkingQueue.getEnd())
      {
         if(*it == ptrMessage)
         {
            break;
         }

         ++it;
      }

      if(it != _app2BtsWorkingQueue.getEnd())
      {
         _app2BtsWorkingQueue.doErase(it);
      }
   }
}

void BasicControl::pushBts2IpcMessage(IN Bts2Ipc_BaseMessage* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if(0 != _mainControl)
   {
      // add component and stack interface, sub component is part of message opcode
      ptrMessage->setComponent(_component);
      ptrMessage->setStackInterface(_stackInterface);
      _mainControl->pushBts2IpcMessage(ptrMessage);
   }
   else
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
   }
}

void BasicControl::pushBts2AppMessage(IN Bts2App_BaseMessage* ptrMessage)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if(0 != _mainControl)
   {
      // add component and stack interface, sub component is part of message opcode
      ptrMessage->setComponent(_component);
      ptrMessage->setStackInterface(_stackInterface);
      _mainControl->pushBts2AppMessage(ptrMessage);
   }
   else
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
   }
}

bool BasicControl::isOpCodeInWaitingQueue(OUT unsigned int& msgIndex, IN const ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList)
{
   unsigned int counter = 0;

   for(MessageQueue<App2Bts_BaseMessage>::It it = _app2BtsWaitingQueue.getBegin(); it != _app2BtsWaitingQueue.getEnd(); ++it)
   {
      BTSApp2BtsMessageCompareItem item;
      (*it)->getCompareItem(item);

      for(size_t i = 0; i < itemList.size(); i++)
      {
         if(true == itemList[i].isMatch(item))
         {
            msgIndex = counter;
            return true;
         }
      }

      counter++;
   }

   return false;
}

bool BasicControl::isMatchingFrontApp2BtsWaitingQueueMessage(IN const App2Bts_BaseMessage* ptrMessage) const
{
   if(NULL == ptrMessage)
   {
      return false;
   }

   if(0 == _app2BtsWaitingQueue.getSize())
   {
      return false;
   }

   if(_app2BtsWaitingQueue.getBegin() == _app2BtsWaitingQueue.getEnd())
   {
      return false;
   }

   if(*_app2BtsWaitingQueue.getBegin() == ptrMessage)
   {
      return true;
   }

   return false;
}

void BasicControl::addControlHandler(IN IControlHandler* handler)
{
   FW_IF_NULL_PTR_RETURN(handler);

   _extensions.push_back(handler);
}

bool BasicControl::processApp2BtsMessage(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_BaseMessage* message)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->handleApp2BtsMessage(bts2IpcMsgList, bts2AppMsgList, deleteApp2BtsMessage, message))
         {
            return true;
         }
      }
   }

   return false;
}

bool BasicControl::executeApp2BtsMsgPrecheck(OUT bool& continueProcessing, OUT bool& rejectRequest, OUT BTSCommonEnumClass& resultCode, OUT BTSCommonEnumClass& statusCode, OUT bool& skipOpCodeCheck, IN App2Bts_BaseMessage* message)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->doApp2BtsMsgPrecheck(continueProcessing, rejectRequest, resultCode, statusCode, skipOpCodeCheck, message))
         {
            return true;
         }
      }
   }

   return false;
}

bool BasicControl::processIpc2BtsMessage(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BaseMessage* message)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->handleIpc2BtsMessage(bts2IpcMsgList, bts2AppMsgList, messageItem, message))
         {
            return true;
         }
      }
   }

   return false;
}

void BasicControl::executeStackConfiguration(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
         OUT ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         _extensions[i]->setStackConfiguration(component, stackInterface, subComponent, userMode, dbusInterfaces, configuration);
      }
   }
}

bool BasicControl::processTriggerInitializedCallback(void)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->triggerInitializedCallback())
         {
            return true;
         }
      }
   }

   return false;
}

bool BasicControl::executeSubControlTestCommand(IN const char* testCommand, IN const unsigned int testData)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->setSubControlTestCommand(testCommand, testData))
         {
            return true;
         }
      }
   }

   return false;
}

bool BasicControl::executeSubControlTestCommand(IN const char* testCommand, IN const unsigned char* testData)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->setSubControlTestCommand(testCommand, testData))
         {
            return true;
         }
      }
   }

   return false;
}

bool BasicControl::requestSimilarOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, IN const App2Bts_BaseMessage* message) const
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->getSimilarOpCodes(itemList, message))
         {
            return true;
         }
      }
   }

   return false;
}

bool BasicControl::requestMatchingOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& highPrioItemList, IN const App2Bts_BaseMessage* message) const
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->getMatchingOpCodes(itemList, highPrioItemList, message))
         {
            return true;
         }
      }
   }

   return false;
}

bool BasicControl::processDoubledApp2BtsMessages(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_BaseMessage* message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->handleDoubledApp2BtsMessages(bts2AppMsgList, message, resultCode, statusCode))
         {
            return true;
         }
      }
   }

   return false;
}

} //btstackif
