/**
 * @file IBasicControl.h
 *
 * @par SW-Component
 * Interface
 *
 * @brief Basic control interface.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface class for basic control functionality.
 */

#ifndef _I_BASIC_CONTROL_H_
#define _I_BASIC_CONTROL_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IMainControl;
class App2Bts_BaseMessage;
class Bts2App_BaseMessage;
class Ipc2Bts_BaseMessage;
class Bts2Ipc_BaseMessage;
class IExtendedTimeoutHandler;
class ITTYDeviceMonitorHandler;

/**
 *
 */
class IBasicControl
{
public:
   virtual ~IBasicControl() {}

   virtual void setMainControl(IN IMainControl* mainControl) = 0;

   virtual void setComponent(IN const BTSFunctionBlock component) = 0;

   virtual void setStackInterface(IN const BTSInterfaceType stackInterface) = 0;

   virtual void setQueueWarningSize(IN const unsigned int maxSize) = 0;

   virtual void pushApp2BtsMessage(IN App2Bts_BaseMessage* ptrMessage) = 0;

   virtual void pushIpc2BtsMessage(IN Ipc2Bts_BaseMessage* ptrMessage) = 0;

   /**
    * Sets configuration data for local stack.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] userMode: user mode
    * @param[out] dbusInterfaces: list of needed DBUS interfaces
    * @param[in] configuration: local stack configuration data
    */
   virtual void setStackConfiguration(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
            OUT ::std::vector<BTSDbusInterfaceItem>& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration) = 0;

   /**
    * Triggers initialized callback.
    */
   virtual void triggerInitializedCallback(void) = 0;

   /**
    * Creates message for DBUS service availability.
    *
    * @param[in] interface: related interface/component/service
    * @param[in] availabilityEvent: availability event
    */
   virtual void createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent) = 0;

   /**
    * Creates message for DBUS service availability.
    *
    * @param[in] interface: related interface/component/service
    * @param[in] availabilityEvent: availability event
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type
    */
   virtual void createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent, IN const BTSBusName& busName, IN const BTSObjectPath& objPath, IN const BTSCommonEnumClass busType) = 0;

   /**
    * Send internal App2Bts message.
    *
    * @param[in] ptrMessage: App2Bts message
    * @param[in] highPrio: flag for high/low priority
    */
   virtual void sendInternalApp2BtsMessage(IN App2Bts_BaseMessage* ptrMessage, IN const bool highPrio = false) = 0;

   /**
    * Send internal Bts2App message.
    *
    * @param[in] ptrMessage: Bts2App message
    */
   virtual void sendInternalBts2AppMessage(IN Bts2App_BaseMessage* ptrMessage) = 0;

   /**
    * Send internal Ipc2Bts message.
    *
    * @param[in] ptrMessage: Ipc2Bts message
    * @param[in] highPrio: flag for high/low priority
    */
   virtual void sendInternalIpc2BtsMessage(IN Ipc2Bts_BaseMessage* ptrMessage, IN const bool highPrio = false) = 0;

   /**
    * Send internal Bts2Ipc message.
    *
    * @param[in] ptrMessage: Bts2Ipc message
    */
   virtual void sendInternalBts2IpcMessage(IN Bts2Ipc_BaseMessage* ptrMessage) = 0;

   /**
    * Send Bts2Ipc message list.
    *
    * @param[in] bts2IpcMsgList: Bts2Ipc message list
    * @param[in] item: message compare items
    */
   virtual void sendBts2IpcMessageList(IN ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSApp2BtsMessageCompareItem& item) = 0;

   /**
    * Send Bts2App message list.
    *
    * @param[in] bts2AppMsgList: Bts2App message list
    */
   virtual void sendBts2AppMessageList(IN ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList) = 0;

   /**
    * Handle timeout (of a timer).
    *
    * @param[in] handler: timeout handler
    * @param[in] timerId: timer id
    */
   virtual void handleTimeout(IN IExtendedTimeoutHandler* handler, IN const BTSTimerId timerId) = 0;

   /**
    * Add TTY monitor handler.
    *
    * @param[in] handler: handler
    * @param[in] filter: filter
    */
   virtual void addTTYMonitorHandler(IN ITTYDeviceMonitorHandler* handler, IN const ::std::string& filter) = 0;

   /**
    * Print queue statistics.
    */
   virtual void printQueueStatistics(void) = 0;

   /**
    * Sets a sub control test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setSubControlTestCommand(IN const char* testCommand, IN const unsigned int testData) = 0;

   /**
    * Sets a sub control test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setSubControlTestCommand(IN const char* testCommand, IN const unsigned char* testData) = 0;

   /**
    * Check waiting queue for messages that can be processed now.
    *
    * @param[in] ptrMessage: App2Bts message that was removed from working queue
    */
   virtual void checkWaitingQueue(IN const App2Bts_BaseMessage* ptrMessage) = 0;

   /**
    * Remove messages from waiting queue.
    *
    * @param[in] app2BtsMsgList: list with messages
    */
   virtual void removeApp2BtsWaitingMessages(IN const ::std::vector< App2Bts_BaseMessage* >& app2BtsMsgList) = 0;

   /**
    * Find matching waiting queue entries.
    *
    * @param[out] msgList: message list with found messages
    * @param[in] itemList: compare item list
    * @param[in] highPrioItemList: high prio compare item list
    *
    * @return = true: high prio item found,
    * @return = false: high prio item not found
    */
   virtual bool getMatchingWaitingQueueEntries(OUT ::std::vector< App2Bts_BaseMessage* >& msgList, IN const ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, IN const ::std::vector< BTSApp2BtsMessageMasterCompareItem >& highPrioItemList) = 0;

   /**
    * Find App2Bts message based on given message compare item.
    *
    * @param[in] item: App2Bts message compare item
    *
    * @return = App2Bts message if found else NULL
    */
   virtual App2Bts_BaseMessage* findApp2BtsWorkingMessageWrapper(IN const BTSApp2BtsMessageCompareItem& item) = 0;

   /**
    * Push App2Bts message to working queue.
    *
    * @param[in] message: App2Bts message
    * @param[in] withLock: flag for locking
    */
   virtual void pushApp2BtsMsgToWorkingQueue(IN App2Bts_BaseMessage* message, bool withLock = true) = 0;

   /**
    * Push App2Bts message to working queue (on top).
    *
    * @param[in] message: App2Bts message
    * @param[in] withLock: flag for locking
    */
   virtual void pushOnTopApp2BtsMsgToWorkingQueue(IN App2Bts_BaseMessage* message, bool withLock = true) = 0;

   /**
    * Push App2Bts message to waiting queue.
    *
    * @param[in] message: App2Bts message
    * @param[in] withLock: flag for locking
    */
   virtual void pushApp2BtsMsgToWaitingQueue(IN App2Bts_BaseMessage* message, bool withLock = true) = 0;

   /**
    * Push App2Bts message to waiting queue (on top).
    *
    * @param[in] message: App2Bts message
    * @param[in] withLock: flag for locking
    */
   virtual void pushOnTopApp2BtsMsgToWaitingQueue(IN App2Bts_BaseMessage* message, bool withLock = true) = 0;

   /**
    * Get App2Bts message working queue size.
    *
    * @return = queue size
    */
   virtual unsigned int getApp2BtsWorkingQueueSize(void) const = 0;

   /**
    * Get App2Bts message waiting queue size.
    *
    * @return = queue size
    */
   virtual unsigned int getApp2BtsWaitingQueueSize(void) const = 0;

   /**
    * Checks if one element in item list (opcode) is part of waiting queue.
    *
    * @param[in] itemList: item list
    *
    * @return = true: at least one element of item list is part of waiting queue,
    * @return = false: no element of item list is part of working queue
    */
   virtual bool isSimilarOpCodeInWaitingQueue(IN const ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList) const = 0;

   /**
    * Checks if one element in item list (opcode) is part of working queue.
    *
    * @param[in] itemList: item list
    *
    * @return = true: at least one element of item list is part of working queue,
    * @return = false: no element of item list is part of working queue
    */
   virtual bool isSimilarOpCodeInWorkingQueue(IN const ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList) const = 0;

   /**
    * Sends direct answers for App2Bts messages.
    *
    * @param[in] msgList: App2Bts message list
    * @param[in] resultCode: result code
    * @param[in] statusCode: status code
    */
   virtual void sendDirectAnswerForApp2BtsMessages(IN const ::std::vector<App2Bts_BaseMessage*>& msgList, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) = 0;

   /**
    * Sends direct answers for App2Bts messages (wrapper).
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] msgList: App2Bts message list
    * @param[in] resultCode: result code
    * @param[in] statusCode: status code
    */
   virtual void sendDirectAnswerForApp2BtsMessagesWrapper(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN const ::std::vector<App2Bts_BaseMessage*>& msgList, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) = 0;

   /**
    * Get working queue entries.
    *
    * @param[out] msgList: list of working queue entries
    * @param[in] withLock: flag for locking
    */
   virtual void getWorkingQueueEntries(OUT ::std::vector<App2Bts_BaseMessage*>& msgList, IN bool withLock = true) = 0;

   /**
    * Get waiting queue entries.
    *
    * @param[out] msgList: list of waiting queue entries
    * @param[in] withLock: flag for locking
    */
   virtual void getWaitingQueueEntries(OUT ::std::vector<App2Bts_BaseMessage*>& msgList, IN bool withLock = true) = 0;

   /**
    * Empty working queue.
    *
    * @param[in] withLock: flag for locking
    */
   virtual void emptyWorkingQueue(IN bool withLock = true) = 0;

   /**
    * Empty waiting queue.
    *
    * @param[in] withLock: flag for locking
    */
   virtual void emptyWaitingQueue(IN bool withLock = true) = 0;

   /**
    * Trigger reset of Dbus interface handler.
    */
   virtual void triggerResetOfDbusIfHandler(void) = 0;
};

} //btstackif

#endif //_I_BASIC_CONTROL_H_
