/**
 * @file Ipc2Bts_BaseMessage.h
 *
 * @par SW-Component
 * Messages
 *
 * @brief IPC to BtStackIf messages.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Base message declaration for IPC to BtStackIf messages.
 */

#ifndef _IPC_2_BTS_BASE_MESSAGE_H_
#define _IPC_2_BTS_BASE_MESSAGE_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

/**
 *
 */
class Ipc2Bts_BaseMessage
{
public:
   /**************************************************************************
    Constructor / destructor
    **************************************************************************/
   /**
    * Constructor. Do not use!!!
    */
   Ipc2Bts_BaseMessage();

   /**
    * Constructor.
    *
    * @param[in] functionBlock: function block
    * @param[in] opCode: opcode
    */
   Ipc2Bts_BaseMessage(IN const BTSCommonEnumClass functionBlock, IN const BTSCommonEnumClass opCode);

   /**
    * Copy constructor.
    *
    * @param[in] ref: reference of message to be copied
    */
   Ipc2Bts_BaseMessage(IN const Ipc2Bts_BaseMessage& ref);

   /**
    * Assignment operator.
    *
    * @param[in] ref: reference of message to be assigned
    */
   Ipc2Bts_BaseMessage& operator=(IN const Ipc2Bts_BaseMessage& ref);

   /**
    * Destructor.
    */
   virtual ~Ipc2Bts_BaseMessage();

   /**
    * Trace function for input parameters.
    */
   virtual void doInputTrace(void);

   /**
    * Returns function block identifier.
    *
    * @return = function block identifier
    */
   inline BTSCommonEnumClass getFunctionBlock(void) const { return _functionBlock; }

   /**
    * Returns message opcode.
    *
    * @return = message opcode
    */
   inline BTSCommonEnumClass getOpCode(void) const { return _opCode; }

   /**
    * Returns message opcode for tracing.
    *
    * @return = message opcode
    */
   inline BTSTraceOpCode getTraceOpCode(void) const { return (BTSTraceOpCode)_opCode; }

   /**
    * Set BTS destination function block identifier.
    *
    * @param[in] btsFunctionBlock: BTS function block
    */
   inline void setBtsDestinationFunctionBlock(IN const BTSFunctionBlock btsFunctionBlock) { _btsDestinationFunctionBlock = btsFunctionBlock; }

   /**
    * Returns BTS destination function block identifier.
    *
    * @return = BTS destination function block identifier
    */
   inline BTSFunctionBlock getBtsDestinationFunctionBlock(void) const { return _btsDestinationFunctionBlock; }

   /**
    * Set BTS opcode.
    *
    * @param[in] opCode: BTS opcode
    */
   inline void setApp2BtsOpcode(IN const BTSApp2BtsOpcode opCode) { _app2BtsOpcode = opCode; }

   /**
    * Returns BTS opcode.
    *
    * @return = BTS opcode
    */
   inline BTSApp2BtsOpcode getApp2BtsOpcode(void) const { return _app2BtsOpcode; }

   /**
    * Returns BTS opcode for tracing.
    *
    * @return = BTS opcode
    */
   inline BTSTraceOpCode getTraceApp2BtsOpCode(void) const { return (BTSTraceOpCode)_app2BtsOpcode; }

   /**
    * Sets component.
    *
    * @param[in] component: component
    */
   inline void setComponent(IN const BTSFunctionBlock component) { _component = component; }

   /**
    * Returns component.
    *
    * @return = component
    */
   inline BTSFunctionBlock getComponent(void) const { return _component; }

   /**
    * Sets stack interface.
    *
    * @param[in] stackInterface: stack interface
    */
   inline void setStackInterface(IN const BTSInterfaceType stackInterface) { _stackInterface = stackInterface; }

   /**
    * Returns stack interface
    *
    * @return = stack interface
    */
   inline BTSInterfaceType getStackInterface(void) const { return _stackInterface; }

   /**
    * Sets BT device address.
    *
    * @param[in] deviceAddress: BT device address
    */
   inline void setBDAddress(IN const BTSBDAddress& deviceAddress) { _deviceAddress = deviceAddress; }

   /**
    * Gets BT device address.
    *
    * @param[out] deviceAddress: BT device address
    */
   inline void getBDAddress(OUT BTSBDAddress& deviceAddress) const { deviceAddress = _deviceAddress; }

   /**
    * Gets BT device address.
    *
    * @return = BT device address
    */
   inline const BTSBDAddress& getBDAddress(void) const { return _deviceAddress; }

   /**
    * Gets BT device address.
    *
    * @return = BT device address
    */
   inline BTSBDAddress& getBDAddressMutable(void) { return _deviceAddress; }

   /**
    * Sets object path.
    *
    * @param[in] objectPath: object path
    */
   inline void setObjectPath(IN const BTSObjectPath& objectPath) { _objectPath = objectPath; }

   /**
    * Gets object path.
    *
    * @param[out] objectPath: object path
    */
   inline void getObjectPath(OUT BTSObjectPath& objectPath) const { objectPath = _objectPath; }

   /**
    * Gets object path.
    *
    * @return = object path
    */
   inline const BTSObjectPath& getObjectPath(void) const { return _objectPath; }

   /**
    * Returns copy of the message object.
    *
    * @return = copy of the message object (pointer, created by new)
    */
   virtual Ipc2Bts_BaseMessage* clone(void) const = 0;

   /**
    * Set common IPC error code.
    *
    * @param[in] opCode: common IPC error code
    */
   inline void setIpcCommonErrorCode(IN const BTSIpcCommonErrorCode errorCode) { _commonErrorCode = errorCode; }

   /**
    * Returns common IPC error code.
    *
    * @return = common IPC error code
    */
   inline BTSIpcCommonErrorCode getIpcCommonErrorCode(void) const { return _commonErrorCode; }

   /**
    * Sets D-Bus token (act).
    *
    * @param[in] token: D-Bus token (act)
    */
   inline void setDbusToken(IN const act_t token) { _token = token; }

   /**
    * Gets D-Bus token (act).
    *
    * @return = D-Bus token (act)
    */
   inline act_t getDbusToken(void) const { return _token; }

   /**
    * Sets error name.
    *
    * @param[in] errorName: error name
    */
   inline void setErrorName(IN const ::std::string& errorName) { _errorName = errorName; }

   /**
    * Gets error name.
    *
    * @param[out] errorName: error name
    */
   inline void getErrorName(OUT ::std::string& errorName) const { errorName = _errorName; }

   /**
    * Gets error name.
    *
    * @return = error name
    */
   inline const ::std::string& getErrorName(void) const { return _errorName; }

   /**
    * Sets error message.
    *
    * @param[in] errorMessage: error message
    */
   inline void setErrorMessage(IN const ::std::string& errorMessage) { _errorMessage = errorMessage; }

   /**
    * Gets error message.
    *
    * @param[out] errorMessage: error message
    */
   inline void getErrorMessage(OUT ::std::string& errorMessage) const { errorMessage = _errorMessage; }

   /**
    * Gets error message.
    *
    * @return = error message
    */
   inline const ::std::string& getErrorMessage(void) const { return _errorMessage; }

   /**
    * Sets App2Bts compare item.
    *
    * @param[in] item: compare item
    */
   inline void setApp2BtsCompareItem(IN const BTSApp2BtsMessageCompareItem& item) { _app2BtsCompareItem = item; }

   /**
    * Gets App2Bts compare item.
    *
    * @param[out] item: compare item
    */
   inline void getApp2BtsCompareItem(OUT BTSApp2BtsMessageCompareItem& item) const { item = _app2BtsCompareItem; }

   /**
    * Gets App2Bts compare item.
    *
    * @return = compare item
    */
   inline const BTSApp2BtsMessageCompareItem& getApp2BtsCompareItem(void) const { return _app2BtsCompareItem; }

   /**
    * Sets flag for response message.
    *
    * @param[in] enable: flag for response message
    */
   inline void setResponseMessageFlag(IN const bool enable) { _isResponseMessage = enable; }

   /**
    * Gets flag for response message.
    *
    * @return = flag for response message
    */
   inline bool getResponseMessageFlag(void) const { return _isResponseMessage; }

   /**
    * Sets object id.
    *
    * @param[in] objectId: object id
    */
   inline void setObjectId(IN const BTSObjectPath& objectId) { _objectId = objectId; }

   /**
    * Gets object id.
    *
    * @param[out] objectId: object id
    */
   inline void getObjectId(OUT BTSObjectPath& objectId) const { objectId = _objectId; }

   /**
    * Gets object id.
    *
    * @return = object id
    */
   inline const BTSObjectPath& getObjectId(void) const { return _objectId; }

   /**
    * Get message trace id.
    *
    * @return = message trace id
    */
   uint32_t getMessageTraceId(void) const;

   /**
    * Returns number of existing IPC 2 BTS messages.
    *
    * @return = number of existing IPC 2 BTS messages
    */
   static inline unsigned int getIpc2BtsMessageCount(void) { return _countIpc2BtsMessages; }

protected:
   BTSCommonEnumClass _functionBlock; /**< identifies the function block the message relates to, depends on used supplier Bluetooth stack */

   BTSCommonEnumClass _opCode; /**< identifies the message itself, depends on used supplier Bluetooth stack */

   BTSFunctionBlock _btsDestinationFunctionBlock; /**< identifies the BTS destination function block that should process this message */

   BTSApp2BtsOpcode _app2BtsOpcode; /**< identifies the BTS opcode the message relates to */

   BTSFunctionBlock _component; /**< component (function block) */

   BTSInterfaceType _stackInterface; /**< stack interface to be used (Bluetooth stack) */

   BTSBDAddress _deviceAddress; /**< parameter device address */

   BTSObjectPath _objectPath; /**< object path e.g. for device */

   BTSIpcCommonErrorCode _commonErrorCode; /**< common IPC error code */

   act_t _token; /**< token to be used to identify a callback invocation */

   ::std::string _errorName; /**< error name in case of request was answered with error */

   ::std::string _errorMessage; /**< error message in case of request was answered with error */

   BTSApp2BtsMessageCompareItem _app2BtsCompareItem; /**< compare item to be used to identify a message */

   bool _isResponseMessage; /**< flag indicating if this is a response message or not */

   BTSObjectPath _objectId; /**< used to identify a removed object path */

   static unsigned int _countIpc2BtsMessages; /**< number of existing IPC 2 BTS messages */
};

} //btstackif

#endif //_IPC_2_BTS_BASE_MESSAGE_H_
