/**
 * @file BtStackIfTypes.cpp
 *
 * @par SW-Component
 * Types
 *
 * @brief Basic types.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of all type classes needed for communication with BtStackIf component.
 */

#include "BtStackIfTypes.h"

namespace btstackif {

const BTSRssi defaultRssiValue = 127;
const BTSLinkQuality defaultLinkQualityValue = 255;
const BTSProfileVersion emptyProtocolVersion = 0x0000;
const BTSProfileVersion defaultProtocolVersion = 0xFFFF;

//------------------------------------------------------------------------------

BTSServiceInfo::BTSServiceInfo()
{
   service = BTS_SUPP_SRV_LAST;
   version = 0xFFFF;
}

BTSServiceInfo::BTSServiceInfo(IN const BTSServiceInfo& ref)
{
   service = ref.service;
   version = ref.version;
}

BTSServiceInfo& BTSServiceInfo::operator=(IN const BTSServiceInfo& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   service = ref.service;
   version = ref.version;

   return *this;
}

bool BTSServiceInfo::operator==(IN const BTSServiceInfo& ref) const
{
   bool result = true;

   result = (true == result) && (service == ref.service);
   result = (true == result) && (version == ref.version);

   return result;
}

bool BTSServiceInfo::operator!=(IN const BTSServiceInfo& ref) const
{
   return !(operator==(ref));
}

BTSServiceInfo::~BTSServiceInfo()
{
}

//------------------------------------------------------------------------------

BTSDiscoveredDeviceInfo::BTSDiscoveredDeviceInfo() :
address(),
name(),
rssi(0),
majorServiceClass(),
majorDeviceClass(BTS_COD_MAJORDC_MISCELLANEOUS),
minorDeviceClass(BTS_COD_MINORDC_UNCATEGORIZED),
services(),
sppCapabilities()
{
}

BTSDiscoveredDeviceInfo::BTSDiscoveredDeviceInfo(IN const BTSDiscoveredDeviceInfo& ref)
{
   address = ref.address;
   name = ref.name;
   rssi = ref.rssi;
   majorServiceClass = ref.majorServiceClass;
   majorDeviceClass = ref.majorDeviceClass;
   minorDeviceClass = ref.minorDeviceClass;
   services = ref.services;
   sppCapabilities = ref.sppCapabilities;
}

BTSDiscoveredDeviceInfo& BTSDiscoveredDeviceInfo::operator=(IN const BTSDiscoveredDeviceInfo& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   address = ref.address;
   name = ref.name;
   rssi = ref.rssi;
   majorServiceClass = ref.majorServiceClass;
   majorDeviceClass = ref.majorDeviceClass;
   minorDeviceClass = ref.minorDeviceClass;
   services = ref.services;
   sppCapabilities = ref.sppCapabilities;

   return *this;
}

bool BTSDiscoveredDeviceInfo::operator==(IN const BTSDiscoveredDeviceInfo& ref) const
{
   bool result = true;

   result = (true == result) && (address == ref.address);
   result = (true == result) && (name == ref.name);
   result = (true == result) && (rssi == ref.rssi);
   result = (true == result) && (majorServiceClass == ref.majorServiceClass);
   result = (true == result) && (majorDeviceClass == ref.majorDeviceClass);
   result = (true == result) && (minorDeviceClass == ref.minorDeviceClass);
   result = (true == result) && (services == ref.services);
   result = (true == result) && (sppCapabilities == ref.sppCapabilities);

   return result;
}

bool BTSDiscoveredDeviceInfo::operator!=(IN const BTSDiscoveredDeviceInfo& ref) const
{
   return !(operator==(ref));
}

BTSDiscoveredDeviceInfo::~BTSDiscoveredDeviceInfo()
{
}

//------------------------------------------------------------------------------

BTSDeviceIdServiceRecord::BTSDeviceIdServiceRecord()
{
   specificationID = 0;
   vendorID = 0;
   productID = 0;
   version = 0;
   primaryRecord = false;
   vendorIDSource = 0;
   // clientExecutableURL
   // serviceDescription
   // clientDocumentationURL
}

BTSDeviceIdServiceRecord::BTSDeviceIdServiceRecord(IN const BTSDeviceIdServiceRecord& ref)
{
   specificationID = ref.specificationID;
   vendorID = ref.vendorID;
   productID = ref.productID;
   version = ref.version;
   primaryRecord = ref.primaryRecord;
   vendorIDSource = ref.vendorIDSource;
   clientExecutableURL = ref.clientExecutableURL;
   serviceDescription = ref.serviceDescription;
   clientDocumentationURL = ref.clientDocumentationURL;
}

BTSDeviceIdServiceRecord& BTSDeviceIdServiceRecord::operator=(IN const BTSDeviceIdServiceRecord& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   specificationID = ref.specificationID;
   vendorID = ref.vendorID;
   productID = ref.productID;
   version = ref.version;
   primaryRecord = ref.primaryRecord;
   vendorIDSource = ref.vendorIDSource;
   clientExecutableURL = ref.clientExecutableURL;
   serviceDescription = ref.serviceDescription;
   clientDocumentationURL = ref.clientDocumentationURL;

   return *this;
}

bool BTSDeviceIdServiceRecord::operator==(IN const BTSDeviceIdServiceRecord& ref) const
{
   bool result = true;

   result = (true == result) && (specificationID == ref.specificationID);
   result = (true == result) && (vendorID == ref.vendorID);
   result = (true == result) && (productID == ref.productID);
   result = (true == result) && (version == ref.version);
   result = (true == result) && (primaryRecord == ref.primaryRecord);
   result = (true == result) && (vendorIDSource == ref.vendorIDSource);
   result = (true == result) && (clientExecutableURL == ref.clientExecutableURL);
   result = (true == result) && (serviceDescription == ref.serviceDescription);
   result = (true == result) && (clientDocumentationURL == ref.clientDocumentationURL);

   return result;
}

bool BTSDeviceIdServiceRecord::operator!=(IN const BTSDeviceIdServiceRecord& ref) const
{
   return !(operator==(ref));
}

BTSDeviceIdServiceRecord::~BTSDeviceIdServiceRecord()
{
}

//------------------------------------------------------------------------------

BTSSppCapability::BTSSppCapability()
{
   // uuidList
   // serviceName
}

BTSSppCapability::BTSSppCapability(IN const BTSSppCapability& ref)
{
   uuidList = ref.uuidList;
   serviceName = ref.serviceName;
}

BTSSppCapability& BTSSppCapability::operator=(IN const BTSSppCapability& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   uuidList = ref.uuidList;
   serviceName = ref.serviceName;

   return *this;
}

bool BTSSppCapability::operator==(IN const BTSSppCapability& ref) const
{
   bool result = true;

   result = (true == result) && (uuidList == ref.uuidList);
   result = (true == result) && (serviceName == ref.serviceName);

   return result;
}

bool BTSSppCapability::operator!=(IN const BTSSppCapability& ref) const
{
   return !(operator==(ref));
}

BTSSppCapability::~BTSSppCapability()
{
}

//------------------------------------------------------------------------------

BTSLocalStackConfiguration::BTSLocalStackConfiguration()
{
   // name
   // supportedServices
   // deviceIdServiceRecord
   wbsEnabled = true;
   pbdlViaPbapEnabled = true;
   pbdlViaHfpEnabled = false;
   pbdlViaSppEnabled = false;
   pbdlViaSyncMlEnabled= false;
   initialDiscoverableMode = BTS_MODE_DISABLED;
   initialConnectableMode = BTS_MODE_DISABLED;
   voiceRecognitionEnabled = true;
   enhancedCallControlEnabled = true;
   avrcpBrowsingEnabled = true;
   audioCodecMp3Enabled = false;
   audioCodecAacEnabled = false;
   mapMmsEnabled = true;
   mapEmailEnabled = false;
   bluetoothLowEnergyEnabled = false;
   carPlayWirelessEnabled = false;
   enableServiceSearchAfterFirstProtocol = true;
   stopInquiryDuringIncomingPairing = true;
   maxNumberConnectedDevices = 3;
#ifdef VARIANT_S_FTR_ENABLE_SPP_SERVER
   sppServerEnabled = true;
#else
   sppServerEnabled = false;
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPP_INSTANCE_TEST
   sppInstanceTestEnabled = true;
#else
   sppInstanceTestEnabled = false;
#endif
   // sppServiceInfoList
   defaultPageTimeout = 10240;
   agentCapability = BTS_AGENT_CAPABILITY_DISPLAY_YES_NO;
}

BTSLocalStackConfiguration::BTSLocalStackConfiguration(IN const BTSLocalStackConfiguration& ref)
{
   name = ref.name;
   supportedServices = ref.supportedServices;
   deviceIdServiceRecord = ref.deviceIdServiceRecord;
   wbsEnabled = ref.wbsEnabled;
   pbdlViaPbapEnabled = ref.pbdlViaPbapEnabled;
   pbdlViaHfpEnabled = ref.pbdlViaHfpEnabled;
   pbdlViaSppEnabled = ref.pbdlViaSppEnabled;
   pbdlViaSyncMlEnabled = ref.pbdlViaSyncMlEnabled;
   initialDiscoverableMode = ref.initialDiscoverableMode;
   initialConnectableMode = ref.initialConnectableMode;
   voiceRecognitionEnabled = ref.voiceRecognitionEnabled;
   enhancedCallControlEnabled = ref.enhancedCallControlEnabled;
   avrcpBrowsingEnabled = ref.avrcpBrowsingEnabled;
   audioCodecMp3Enabled = ref.audioCodecMp3Enabled;
   audioCodecAacEnabled = ref.audioCodecAacEnabled;
   mapMmsEnabled = ref.mapMmsEnabled;
   mapEmailEnabled = ref.mapEmailEnabled;
   bluetoothLowEnergyEnabled = ref.bluetoothLowEnergyEnabled;
   carPlayWirelessEnabled = ref.carPlayWirelessEnabled;
   enableServiceSearchAfterFirstProtocol = ref.enableServiceSearchAfterFirstProtocol;
   stopInquiryDuringIncomingPairing = ref.stopInquiryDuringIncomingPairing;
   maxNumberConnectedDevices = ref.maxNumberConnectedDevices;
   sppServerEnabled = ref.sppServerEnabled;
   sppInstanceTestEnabled = ref.sppInstanceTestEnabled;
   sppServiceInfoList = ref.sppServiceInfoList;
   defaultPageTimeout = ref.defaultPageTimeout;
   agentCapability = ref.agentCapability;
}

BTSLocalStackConfiguration& BTSLocalStackConfiguration::operator=(IN const BTSLocalStackConfiguration& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   name = ref.name;
   supportedServices = ref.supportedServices;
   deviceIdServiceRecord = ref.deviceIdServiceRecord;
   wbsEnabled = ref.wbsEnabled;
   pbdlViaPbapEnabled = ref.pbdlViaPbapEnabled;
   pbdlViaHfpEnabled = ref.pbdlViaHfpEnabled;
   pbdlViaSppEnabled = ref.pbdlViaSppEnabled;
   pbdlViaSyncMlEnabled = ref.pbdlViaSyncMlEnabled;
   initialDiscoverableMode = ref.initialDiscoverableMode;
   initialConnectableMode = ref.initialConnectableMode;
   voiceRecognitionEnabled = ref.voiceRecognitionEnabled;
   enhancedCallControlEnabled = ref.enhancedCallControlEnabled;
   avrcpBrowsingEnabled = ref.avrcpBrowsingEnabled;
   audioCodecMp3Enabled = ref.audioCodecMp3Enabled;
   audioCodecAacEnabled = ref.audioCodecAacEnabled;
   mapMmsEnabled = ref.mapMmsEnabled;
   mapEmailEnabled = ref.mapEmailEnabled;
   bluetoothLowEnergyEnabled = ref.bluetoothLowEnergyEnabled;
   carPlayWirelessEnabled = ref.carPlayWirelessEnabled;
   enableServiceSearchAfterFirstProtocol = ref.enableServiceSearchAfterFirstProtocol;
   stopInquiryDuringIncomingPairing = ref.stopInquiryDuringIncomingPairing;
   maxNumberConnectedDevices = ref.maxNumberConnectedDevices;
   sppServerEnabled = ref.sppServerEnabled;
   sppInstanceTestEnabled = ref.sppInstanceTestEnabled;
   sppServiceInfoList = ref.sppServiceInfoList;
   defaultPageTimeout = ref.defaultPageTimeout;
   agentCapability = ref.agentCapability;

   return *this;
}

bool BTSLocalStackConfiguration::operator==(IN const BTSLocalStackConfiguration& ref) const
{
   bool result = true;

   result = (true == result) && (name == ref.name);
   result = (true == result) && (supportedServices == ref.supportedServices);
   result = (true == result) && (deviceIdServiceRecord == ref.deviceIdServiceRecord);
   result = (true == result) && (wbsEnabled == ref.wbsEnabled);
   result = (true == result) && (pbdlViaPbapEnabled == ref.pbdlViaPbapEnabled);
   result = (true == result) && (pbdlViaHfpEnabled == ref.pbdlViaHfpEnabled);
   result = (true == result) && (pbdlViaSppEnabled == ref.pbdlViaSppEnabled);
   result = (true == result) && (pbdlViaSyncMlEnabled == ref.pbdlViaSyncMlEnabled);
   result = (true == result) && (initialDiscoverableMode == ref.initialDiscoverableMode);
   result = (true == result) && (initialConnectableMode == ref.initialConnectableMode);
   result = (true == result) && (voiceRecognitionEnabled == ref.voiceRecognitionEnabled);
   result = (true == result) && (enhancedCallControlEnabled == ref.enhancedCallControlEnabled);
   result = (true == result) && (avrcpBrowsingEnabled == ref.avrcpBrowsingEnabled);
   result = (true == result) && (audioCodecMp3Enabled == ref.audioCodecMp3Enabled);
   result = (true == result) && (audioCodecAacEnabled == ref.audioCodecAacEnabled);
   result = (true == result) && (mapMmsEnabled == ref.mapMmsEnabled);
   result = (true == result) && (mapEmailEnabled == ref.mapEmailEnabled);
   result = (true == result) && (bluetoothLowEnergyEnabled == ref.bluetoothLowEnergyEnabled);
   result = (true == result) && (carPlayWirelessEnabled == ref.carPlayWirelessEnabled);
   result = (true == result) && (enableServiceSearchAfterFirstProtocol == ref.enableServiceSearchAfterFirstProtocol);
   result = (true == result) && (stopInquiryDuringIncomingPairing == ref.stopInquiryDuringIncomingPairing);
   result = (true == result) && (maxNumberConnectedDevices == ref.maxNumberConnectedDevices);
   result = (true == result) && (sppServerEnabled == ref.sppServerEnabled);
   result = (true == result) && (sppInstanceTestEnabled == ref.sppInstanceTestEnabled);
   result = (true == result) && (sppServiceInfoList == ref.sppServiceInfoList);
   result = (true == result) && (defaultPageTimeout == ref.defaultPageTimeout);
   result = (true == result) && (agentCapability == ref.agentCapability);

   return result;
}

bool BTSLocalStackConfiguration::operator!=(IN const BTSLocalStackConfiguration& ref) const
{
   return !(operator==(ref));
}

BTSLocalStackConfiguration::~BTSLocalStackConfiguration()
{
}

//------------------------------------------------------------------------------

BTSLocalConfigConfiguration::BTSLocalConfigConfiguration() :
dummy(false)
{
}

BTSLocalConfigConfiguration::BTSLocalConfigConfiguration(IN const BTSLocalConfigConfiguration& ref) :
dummy(ref.dummy)
{
}

BTSLocalConfigConfiguration& BTSLocalConfigConfiguration::operator=(IN const BTSLocalConfigConfiguration& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   dummy = ref.dummy;

   return *this;
}

bool BTSLocalConfigConfiguration::operator==(IN const BTSLocalConfigConfiguration& ref) const
{
   bool result = true;

   result = (true == result) && (dummy == ref.dummy);

   return result;
}

bool BTSLocalConfigConfiguration::operator!=(IN const BTSLocalConfigConfiguration& ref) const
{
   return !(operator==(ref));
}

BTSLocalConfigConfiguration::~BTSLocalConfigConfiguration()
{
}

//------------------------------------------------------------------------------

BTSLocalConnectionConfiguration::BTSLocalConnectionConfiguration() :
configuration(),
pairedDevices()
{
}

BTSLocalConnectionConfiguration::BTSLocalConnectionConfiguration(IN const BTSLocalConnectionConfiguration& ref) :
configuration(ref.configuration),
pairedDevices(ref.pairedDevices)
{
}

BTSLocalConnectionConfiguration& BTSLocalConnectionConfiguration::operator=(IN const BTSLocalConnectionConfiguration& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   configuration = ref.configuration;
   pairedDevices = ref.pairedDevices;

   return *this;
}

bool BTSLocalConnectionConfiguration::operator==(IN const BTSLocalConnectionConfiguration& ref) const
{
   bool result = true;

   result = (true == result) && (configuration == ref.configuration);
   result = (true == result) && (pairedDevices == ref.pairedDevices);

   return result;
}

bool BTSLocalConnectionConfiguration::operator!=(IN const BTSLocalConnectionConfiguration& ref) const
{
   return !(operator==(ref));
}

BTSLocalConnectionConfiguration::~BTSLocalConnectionConfiguration()
{
}

//------------------------------------------------------------------------------

BTSLocalTelephonyConfiguration::BTSLocalTelephonyConfiguration() :
dummy(false)
{
}

BTSLocalTelephonyConfiguration::BTSLocalTelephonyConfiguration(IN const BTSLocalTelephonyConfiguration& ref) :
dummy(ref.dummy)
{
}

BTSLocalTelephonyConfiguration& BTSLocalTelephonyConfiguration::operator=(IN const BTSLocalTelephonyConfiguration& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   dummy = ref.dummy;

   return *this;
}

bool BTSLocalTelephonyConfiguration::operator==(IN const BTSLocalTelephonyConfiguration& ref) const
{
   bool result = true;

   result = (true == result) && (dummy == ref.dummy);

   return result;
}

bool BTSLocalTelephonyConfiguration::operator!=(IN const BTSLocalTelephonyConfiguration& ref) const
{
   return !(operator==(ref));
}

BTSLocalTelephonyConfiguration::~BTSLocalTelephonyConfiguration()
{
}

//------------------------------------------------------------------------------

BTSLocalPhonebookConfiguration::BTSLocalPhonebookConfiguration() :
dummy(false)
{
}

BTSLocalPhonebookConfiguration::BTSLocalPhonebookConfiguration(IN const BTSLocalPhonebookConfiguration& ref) :
dummy(ref.dummy)
{
}

BTSLocalPhonebookConfiguration& BTSLocalPhonebookConfiguration::operator=(IN const BTSLocalPhonebookConfiguration& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   dummy = ref.dummy;

   return *this;
}

bool BTSLocalPhonebookConfiguration::operator==(IN const BTSLocalPhonebookConfiguration& ref) const
{
   bool result = true;

   result = (true == result) && (dummy == ref.dummy);

   return result;
}

bool BTSLocalPhonebookConfiguration::operator!=(IN const BTSLocalPhonebookConfiguration& ref) const
{
   return !(operator==(ref));
}

BTSLocalPhonebookConfiguration::~BTSLocalPhonebookConfiguration()
{
}

//------------------------------------------------------------------------------

BTSLocalMessagingConfiguration::BTSLocalMessagingConfiguration() :
dummy(false)
{
}

BTSLocalMessagingConfiguration::BTSLocalMessagingConfiguration(IN const BTSLocalMessagingConfiguration& ref) :
dummy(ref.dummy)
{
}

BTSLocalMessagingConfiguration& BTSLocalMessagingConfiguration::operator=(IN const BTSLocalMessagingConfiguration& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   dummy = ref.dummy;

   return *this;
}

bool BTSLocalMessagingConfiguration::operator==(IN const BTSLocalMessagingConfiguration& ref) const
{
   bool result = true;

   result = (true == result) && (dummy == ref.dummy);

   return result;
}

bool BTSLocalMessagingConfiguration::operator!=(IN const BTSLocalMessagingConfiguration& ref) const
{
   return !(operator==(ref));
}

BTSLocalMessagingConfiguration::~BTSLocalMessagingConfiguration()
{
}

//------------------------------------------------------------------------------

BTSLocalMediaPlayerConfiguration::BTSLocalMediaPlayerConfiguration() :
dummy(false)
{
}

BTSLocalMediaPlayerConfiguration::BTSLocalMediaPlayerConfiguration(IN const BTSLocalMediaPlayerConfiguration& ref) :
dummy(ref.dummy)
{
}

BTSLocalMediaPlayerConfiguration& BTSLocalMediaPlayerConfiguration::operator=(IN const BTSLocalMediaPlayerConfiguration& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   dummy = ref.dummy;

   return *this;
}

bool BTSLocalMediaPlayerConfiguration::operator==(IN const BTSLocalMediaPlayerConfiguration& ref) const
{
   bool result = true;

   result = (true == result) && (dummy == ref.dummy);

   return result;
}

bool BTSLocalMediaPlayerConfiguration::operator!=(IN const BTSLocalMediaPlayerConfiguration& ref) const
{
   return !(operator==(ref));
}

BTSLocalMediaPlayerConfiguration::~BTSLocalMediaPlayerConfiguration()
{
}

//------------------------------------------------------------------------------

BTSLocalWblConfiguration::BTSLocalWblConfiguration() :
dummy(false)
{
}

BTSLocalWblConfiguration::BTSLocalWblConfiguration(IN const BTSLocalWblConfiguration& ref) :
dummy(ref.dummy)
{
}

BTSLocalWblConfiguration& BTSLocalWblConfiguration::operator=(IN const BTSLocalWblConfiguration& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   dummy = ref.dummy;

   return *this;
}

bool BTSLocalWblConfiguration::operator==(IN const BTSLocalWblConfiguration& ref) const
{
   bool result = true;

   result = (true == result) && (dummy == ref.dummy);

   return result;
}

bool BTSLocalWblConfiguration::operator!=(IN const BTSLocalWblConfiguration& ref) const
{
   return !(operator==(ref));
}

BTSLocalWblConfiguration::~BTSLocalWblConfiguration()
{
}

//------------------------------------------------------------------------------

BTSLocalEcnrConfiguration::BTSLocalEcnrConfiguration() :
dummy(false)
{
}

BTSLocalEcnrConfiguration::BTSLocalEcnrConfiguration(IN const BTSLocalEcnrConfiguration& ref) :
dummy(ref.dummy)
{
}

BTSLocalEcnrConfiguration& BTSLocalEcnrConfiguration::operator=(IN const BTSLocalEcnrConfiguration& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   dummy = ref.dummy;

   return *this;
}

bool BTSLocalEcnrConfiguration::operator==(IN const BTSLocalEcnrConfiguration& ref) const
{
   bool result = true;

   result = (true == result) && (dummy == ref.dummy);

   return result;
}

bool BTSLocalEcnrConfiguration::operator!=(IN const BTSLocalEcnrConfiguration& ref) const
{
   return !(operator==(ref));
}

BTSLocalEcnrConfiguration::~BTSLocalEcnrConfiguration()
{
}

//------------------------------------------------------------------------------

BTSLocalConfigurationContainer::BTSLocalConfigurationContainer() :
configConfiguration(),
configActive(false),
connectionConfiguration(),
connectionActive(false),
telephonyConfiguration(),
telephonyActive(false),
phonebookConfiguration(),
phonebookActive(false),
messagingConfiguration(),
messagingActive(false),
mediaPlayerConfiguration(),
mediaPlayerActive(false),
wblConfiguration(),
wblActive(false),
ecnrConfiguration(),
ecnrActive(false)
{
}

BTSLocalConfigurationContainer::BTSLocalConfigurationContainer(IN const BTSLocalConfigurationContainer& ref) :
configConfiguration(ref.configConfiguration),
configActive(ref.configActive),
connectionConfiguration(ref.connectionConfiguration),
connectionActive(ref.connectionActive),
telephonyConfiguration(ref.telephonyConfiguration),
telephonyActive(ref.telephonyActive),
phonebookConfiguration(ref.phonebookConfiguration),
phonebookActive(ref.phonebookActive),
messagingConfiguration(ref.messagingConfiguration),
messagingActive(ref.messagingActive),
mediaPlayerConfiguration(ref.mediaPlayerConfiguration),
mediaPlayerActive(ref.mediaPlayerActive),
wblConfiguration(ref.wblConfiguration),
wblActive(ref.wblActive),
ecnrConfiguration(ref.ecnrConfiguration),
ecnrActive(ref.ecnrActive)
{
}

BTSLocalConfigurationContainer& BTSLocalConfigurationContainer::operator=(IN const BTSLocalConfigurationContainer& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   configConfiguration = ref.configConfiguration;
   configActive = ref.configActive;
   connectionConfiguration = ref.connectionConfiguration;
   connectionActive = ref.connectionActive;
   telephonyConfiguration = ref.telephonyConfiguration;
   telephonyActive = ref.telephonyActive;
   phonebookConfiguration = ref.phonebookConfiguration;
   phonebookActive = ref.phonebookActive;
   messagingConfiguration = ref.messagingConfiguration;
   messagingActive = ref.messagingActive;
   mediaPlayerConfiguration = ref.mediaPlayerConfiguration;
   mediaPlayerActive = ref.mediaPlayerActive;
   wblConfiguration = ref.wblConfiguration;
   wblActive = ref.wblActive;
   ecnrConfiguration = ref.ecnrConfiguration;
   ecnrActive = ref.ecnrActive;

   return *this;
}

bool BTSLocalConfigurationContainer::operator==(IN const BTSLocalConfigurationContainer& ref) const
{
   bool result = true;

   result = (true == result) && (configConfiguration == ref.configConfiguration);
   result = (true == result) && (configActive == ref.configActive);
   result = (true == result) && (connectionConfiguration == ref.connectionConfiguration);
   result = (true == result) && (connectionActive == ref.connectionActive);
   result = (true == result) && (telephonyConfiguration == ref.telephonyConfiguration);
   result = (true == result) && (telephonyActive == ref.telephonyActive);
   result = (true == result) && (phonebookConfiguration == ref.phonebookConfiguration);
   result = (true == result) && (phonebookActive == ref.phonebookActive);
   result = (true == result) && (messagingConfiguration == ref.messagingConfiguration);
   result = (true == result) && (messagingActive == ref.messagingActive);
   result = (true == result) && (mediaPlayerConfiguration == ref.mediaPlayerConfiguration);
   result = (true == result) && (mediaPlayerActive == ref.mediaPlayerActive);
   result = (true == result) && (wblConfiguration == ref.wblConfiguration);
   result = (true == result) && (wblActive == ref.wblActive);
   result = (true == result) && (ecnrConfiguration == ref.ecnrConfiguration);
   result = (true == result) && (ecnrActive == ref.ecnrActive);

   return result;
}

bool BTSLocalConfigurationContainer::operator!=(IN const BTSLocalConfigurationContainer& ref) const
{
   return !(operator==(ref));
}

BTSLocalConfigurationContainer::~BTSLocalConfigurationContainer()
{
}

//------------------------------------------------------------------------------

BTSLocalVersionInfo::BTSLocalVersionInfo()
{
   stackType = BTS_IF_LAST;
   // stackVersion
   // supportedServices
   // moduleName
   btVersion = BTS_BT_CORE_LAST;
   // fwVersion
}

BTSLocalVersionInfo::BTSLocalVersionInfo(IN const BTSLocalVersionInfo& ref)
{
   stackType = ref.stackType;
   stackVersion = ref.stackVersion;
   supportedServices = ref.supportedServices;
   moduleName = ref.moduleName;
   btVersion = ref.btVersion;
   fwVersion = ref.fwVersion;
}

BTSLocalVersionInfo& BTSLocalVersionInfo::operator=(IN const BTSLocalVersionInfo& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   stackType = ref.stackType;
   stackVersion = ref.stackVersion;
   supportedServices = ref.supportedServices;
   moduleName = ref.moduleName;
   btVersion = ref.btVersion;
   fwVersion = ref.fwVersion;

   return *this;
}

bool BTSLocalVersionInfo::operator==(IN const BTSLocalVersionInfo& ref) const
{
   bool result = true;

   result = (true == result) && (stackType == ref.stackType);
   result = (true == result) && (stackVersion == ref.stackVersion);
   result = (true == result) && (supportedServices == ref.supportedServices);
   result = (true == result) && (moduleName == ref.moduleName);
   result = (true == result) && (btVersion == ref.btVersion);
   result = (true == result) && (fwVersion == ref.fwVersion);

   return result;
}

bool BTSLocalVersionInfo::operator!=(IN const BTSLocalVersionInfo& ref) const
{
   return !(operator==(ref));
}

BTSLocalVersionInfo::~BTSLocalVersionInfo()
{
}

//------------------------------------------------------------------------------

BTSPbapSdpRecord::BTSPbapSdpRecord()
{
   version = 0;
   // repositories
   // features
}

BTSPbapSdpRecord::BTSPbapSdpRecord(IN const BTSPbapSdpRecord& ref)
{
   version = ref.version;
   repositories = ref.repositories;
   features = ref.features;
}

BTSPbapSdpRecord& BTSPbapSdpRecord::operator=(IN const BTSPbapSdpRecord& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   version = ref.version;
   repositories = ref.repositories;
   features = ref.features;

   return *this;
}

bool BTSPbapSdpRecord::operator==(IN const BTSPbapSdpRecord& ref) const
{
   bool result = true;

   result = (true == result) && (version == ref.version);
   result = (true == result) && (repositories == ref.repositories);
   result = (true == result) && (features == ref.features);

   return result;
}

bool BTSPbapSdpRecord::operator!=(IN const BTSPbapSdpRecord& ref) const
{
   return !(operator==(ref));
}

BTSPbapSdpRecord::~BTSPbapSdpRecord()
{
}

//------------------------------------------------------------------------------

BTSMapSdpRecord::BTSMapSdpRecord()
{
   version = 0;
   instanceId = 0;
   // name
   // messageTypes
   // features
}

BTSMapSdpRecord::BTSMapSdpRecord(IN const BTSMapSdpRecord& ref)
{
   version = ref.version;
   instanceId = ref.instanceId;
   name = ref.name;
   messageTypes = ref.messageTypes;
   features = ref.features;
}

BTSMapSdpRecord& BTSMapSdpRecord::operator=(IN const BTSMapSdpRecord& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   version = ref.version;
   instanceId = ref.instanceId;
   name = ref.name;
   messageTypes = ref.messageTypes;
   features = ref.features;

   return *this;
}

bool BTSMapSdpRecord::operator==(IN const BTSMapSdpRecord& ref) const
{
   bool result = true;

   result = (true == result) && (version == ref.version);
   result = (true == result) && (instanceId == ref.instanceId);
   result = (true == result) && (name == ref.name);
   result = (true == result) && (messageTypes == ref.messageTypes);
   result = (true == result) && (features == ref.features);

   return result;
}

bool BTSMapSdpRecord::operator!=(IN const BTSMapSdpRecord& ref) const
{
   return !(operator==(ref));
}

BTSMapSdpRecord::~BTSMapSdpRecord()
{
}

//------------------------------------------------------------------------------

BTSDeviceCapabilities::BTSDeviceCapabilities()
{
   inbandRingtoneSupported = false;
}

BTSDeviceCapabilities::BTSDeviceCapabilities(IN const BTSDeviceCapabilities& ref)
{
   inbandRingtoneSupported = ref.inbandRingtoneSupported;
}

BTSDeviceCapabilities& BTSDeviceCapabilities::operator=(IN const BTSDeviceCapabilities& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   inbandRingtoneSupported = ref.inbandRingtoneSupported;

   return *this;
}

bool BTSDeviceCapabilities::operator==(IN const BTSDeviceCapabilities& ref) const
{
   bool result = true;

   result = (true == result) && (inbandRingtoneSupported == ref.inbandRingtoneSupported);

   return result;
}

bool BTSDeviceCapabilities::operator!=(IN const BTSDeviceCapabilities& ref) const
{
   return !(operator==(ref));
}

BTSDeviceCapabilities::~BTSDeviceCapabilities()
{
}

//------------------------------------------------------------------------------

BTSSppServiceInfo::BTSSppServiceInfo() :
serviceName(),
localUuid(),
remoteUuid()
{
}

BTSSppServiceInfo::BTSSppServiceInfo(IN const BTSSppServiceInfo& ref) :
serviceName(ref.serviceName),
localUuid(ref.localUuid),
remoteUuid(ref.remoteUuid)
{
}

BTSSppServiceInfo& BTSSppServiceInfo::operator=(IN const BTSSppServiceInfo& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   serviceName = ref.serviceName;
   localUuid = ref.localUuid;
   remoteUuid = ref.remoteUuid;

   return *this;
}

bool BTSSppServiceInfo::operator==(IN const BTSSppServiceInfo& ref) const
{
   bool result = true;

   result = (true == result) && (serviceName == ref.serviceName);
   result = (true == result) && (localUuid == ref.localUuid);
   result = (true == result) && (remoteUuid == ref.remoteUuid);

   return result;
}

bool BTSSppServiceInfo::operator!=(IN const BTSSppServiceInfo& ref) const
{
   return !(operator==(ref));
}

BTSSppServiceInfo::~BTSSppServiceInfo()
{
}

//------------------------------------------------------------------------------

BTSComponentEntry::BTSComponentEntry() :
component(BTS_FB_LAST),
stackInterface(BTS_IF_LAST),
subComponent(BTS_FB_LAST)
{
}

BTSComponentEntry::BTSComponentEntry(IN const BTSFunctionBlock inputComponent, IN const BTSInterfaceType inputStackInterface, IN const BTSFunctionBlock inputSubComponent) :
component(inputComponent),
stackInterface(inputStackInterface),
subComponent(inputSubComponent)
{
}

BTSComponentEntry::BTSComponentEntry(IN const BTSComponentEntry& ref) :
component(ref.component),
stackInterface(ref.stackInterface),
subComponent(ref.subComponent)
{
}

BTSComponentEntry& BTSComponentEntry::operator=(IN const BTSComponentEntry& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   component = ref.component;
   stackInterface = ref.stackInterface;
   subComponent = ref.subComponent;

   return *this;
}

bool BTSComponentEntry::operator==(IN const BTSComponentEntry& ref) const
{
   return (0 == compare(ref));
}

bool BTSComponentEntry::operator!=(IN const BTSComponentEntry& ref) const
{
   return (0 != compare(ref));
}

bool BTSComponentEntry::operator<(const BTSComponentEntry& ref) const
{
   return (0 > compare(ref));
}

bool BTSComponentEntry::operator>(const BTSComponentEntry& ref) const
{
   return (0 < compare(ref));
}

BTSComponentEntry::~BTSComponentEntry()
{
}

int BTSComponentEntry::compare(const BTSComponentEntry& ref) const
{
   if(component < ref.component)
   {
      return -1;
   }
   else if(component > ref.component)
   {
      return 1;
   }

   // component is equal

   if(stackInterface < ref.stackInterface)
   {
      return -1;
   }
   else if(stackInterface > ref.stackInterface)
   {
      return 1;
   }

   // stackInterface is equal

   if(subComponent < ref.subComponent)
   {
      return -1;
   }
   else if(subComponent > ref.subComponent)
   {
      return 1;
   }

   // subComponent is equal

   return 0;
}

//------------------------------------------------------------------------------

BTSWblWiFiSetupStatus::BTSWblWiFiSetupStatus() :
mode(BTS_WBL_WIFI_MODE_LAST),
powerState(BTS_WBL_WIFI_STATE_LAST),
frequency(BTS_WBL_WIFI_FREQUENCY_LAST),
type(BTS_WBL_AP_CONFIG_LAST)
{
}

BTSWblWiFiSetupStatus::BTSWblWiFiSetupStatus(IN const BTSWblWiFiSetupStatus& ref) :
mode(ref.mode),
powerState(ref.powerState),
frequency(ref.frequency),
type(ref.type)
{
}

BTSWblWiFiSetupStatus& BTSWblWiFiSetupStatus::operator=(IN const BTSWblWiFiSetupStatus& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   mode = ref.mode;
   powerState = ref.powerState;
   frequency = ref.frequency;
   type = ref.type;

   return *this;
}

bool BTSWblWiFiSetupStatus::operator==(IN const BTSWblWiFiSetupStatus& ref) const
{
   bool result = true;

   result = (true == result) && (mode == ref.mode);
   result = (true == result) && (powerState == ref.powerState);
   result = (true == result) && (frequency == ref.frequency);
   result = (true == result) && (type == ref.type);

   return result;
}

bool BTSWblWiFiSetupStatus::operator!=(IN const BTSWblWiFiSetupStatus& ref) const
{
   return !(operator==(ref));
}

BTSWblWiFiSetupStatus::~BTSWblWiFiSetupStatus()
{
}

void BTSWblWiFiSetupStatus::reset(void)
{
   mode = BTS_WBL_WIFI_MODE_LAST;
   powerState = BTS_WBL_WIFI_STATE_LAST;
   frequency = BTS_WBL_WIFI_FREQUENCY_LAST;
   type = BTS_WBL_AP_CONFIG_LAST;
}

//------------------------------------------------------------------------------

BTSWblRestriction::BTSWblRestriction() :
type(BTS_WBL_AP_CONFIG_LAST),
state(BTS_WBL_RESTRICTION_STATE_LAST)
{
}

BTSWblRestriction::BTSWblRestriction(IN const BTSWblRestriction& ref) :
type(ref.type),
state(ref.state)
{
}

BTSWblRestriction& BTSWblRestriction::operator=(IN const BTSWblRestriction& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   type = ref.type;
   state = ref.state;

   return *this;
}

bool BTSWblRestriction::operator==(IN const BTSWblRestriction& ref) const
{
   bool result = true;

   result = (true == result) && (type == ref.type);
   result = (true == result) && (state == ref.state);

   return result;
}

bool BTSWblRestriction::operator!=(IN const BTSWblRestriction& ref) const
{
   return !(operator==(ref));
}

BTSWblRestriction::~BTSWblRestriction()
{
}

//------------------------------------------------------------------------------

BTSWblSupportedTechnology::BTSWblSupportedTechnology() :
type(BTS_WBL_AP_CONFIG_LAST),
setting(BTS_WBL_SUPPORTED_SETTING_LAST)
{
}

BTSWblSupportedTechnology::BTSWblSupportedTechnology(IN const BTSWblSupportedTechnology& ref) :
type(ref.type),
setting(ref.setting)
{
}

BTSWblSupportedTechnology& BTSWblSupportedTechnology::operator=(IN const BTSWblSupportedTechnology& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   type = ref.type;
   setting = ref.setting;

   return *this;
}

bool BTSWblSupportedTechnology::operator==(IN const BTSWblSupportedTechnology& ref) const
{
   bool result = true;

   result = (true == result) && (type == ref.type);
   result = (true == result) && (setting == ref.setting);

   return result;
}

bool BTSWblSupportedTechnology::operator!=(IN const BTSWblSupportedTechnology& ref) const
{
   return !(operator==(ref));
}

BTSWblSupportedTechnology::~BTSWblSupportedTechnology()
{
}

//------------------------------------------------------------------------------

BTSWblSupportedRestriction::BTSWblSupportedRestriction() :
frequency(BTS_WBL_WIFI_FREQUENCY_LAST),
technologies()
{
}

BTSWblSupportedRestriction::BTSWblSupportedRestriction(IN const BTSWblSupportedRestriction& ref) :
frequency(ref.frequency),
technologies(ref.technologies)
{
}

BTSWblSupportedRestriction& BTSWblSupportedRestriction::operator=(IN const BTSWblSupportedRestriction& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   frequency = ref.frequency;
   technologies = ref.technologies;

   return *this;
}

bool BTSWblSupportedRestriction::operator==(IN const BTSWblSupportedRestriction& ref) const
{
   bool result = true;

   result = (true == result) && (frequency == ref.frequency);
   result = (true == result) && (technologies == ref.technologies);

   return result;
}

bool BTSWblSupportedRestriction::operator!=(IN const BTSWblSupportedRestriction& ref) const
{
   return !(operator==(ref));
}

BTSWblSupportedRestriction::~BTSWblSupportedRestriction()
{
}

//------------------------------------------------------------------------------

BTSWblSupportedRestrictionEntry::BTSWblSupportedRestrictionEntry() :
id(0),
supportedRestrictions()
{
}

BTSWblSupportedRestrictionEntry::BTSWblSupportedRestrictionEntry(IN const BTSWblSupportedRestrictionEntry& ref) :
id(ref.id),
supportedRestrictions(ref.supportedRestrictions)
{
}

BTSWblSupportedRestrictionEntry& BTSWblSupportedRestrictionEntry::operator=(IN const BTSWblSupportedRestrictionEntry& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   id = ref.id;
   supportedRestrictions = ref.supportedRestrictions;

   return *this;
}

bool BTSWblSupportedRestrictionEntry::operator==(IN const BTSWblSupportedRestrictionEntry& ref) const
{
   bool result = true;

   result = (true == result) && (id == ref.id);
   result = (true == result) && (supportedRestrictions == ref.supportedRestrictions);

   return result;
}

bool BTSWblSupportedRestrictionEntry::operator!=(IN const BTSWblSupportedRestrictionEntry& ref) const
{
   return !(operator==(ref));
}

BTSWblSupportedRestrictionEntry::~BTSWblSupportedRestrictionEntry()
{
}

//------------------------------------------------------------------------------

BTSWblWiFiSetupStatusEntry::BTSWblWiFiSetupStatusEntry() :
id(0),
status()
{
}

BTSWblWiFiSetupStatusEntry::BTSWblWiFiSetupStatusEntry(IN const BTSWblWiFiSetupStatusEntry& ref) :
id(ref.id),
status(ref.status)
{
}

BTSWblWiFiSetupStatusEntry& BTSWblWiFiSetupStatusEntry::operator=(IN const BTSWblWiFiSetupStatusEntry& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   id = ref.id;
   status = ref.status;

   return *this;
}

bool BTSWblWiFiSetupStatusEntry::operator==(IN const BTSWblWiFiSetupStatusEntry& ref) const
{
   bool result = true;

   result = (true == result) && (id == ref.id);
   result = (true == result) && (status == ref.status);

   return result;
}

bool BTSWblWiFiSetupStatusEntry::operator!=(IN const BTSWblWiFiSetupStatusEntry& ref) const
{
   return !(operator==(ref));
}

BTSWblWiFiSetupStatusEntry::~BTSWblWiFiSetupStatusEntry()
{
}

//------------------------------------------------------------------------------

BTSWblLastIntendedMode::BTSWblLastIntendedMode() :
id(0),
status()
{
}

BTSWblLastIntendedMode::BTSWblLastIntendedMode(IN const BTSWblLastIntendedMode& ref) :
id(ref.id),
status(ref.status)
{
}

BTSWblLastIntendedMode& BTSWblLastIntendedMode::operator=(IN const BTSWblLastIntendedMode& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   id = ref.id;
   status = ref.status;

   return *this;
}

bool BTSWblLastIntendedMode::operator==(IN const BTSWblLastIntendedMode& ref) const
{
   bool result = true;

   result = (true == result) && (id == ref.id);
   result = (true == result) && (status == ref.status);

   return result;
}

bool BTSWblLastIntendedMode::operator!=(IN const BTSWblLastIntendedMode& ref) const
{
   return !(operator==(ref));
}

BTSWblLastIntendedMode::~BTSWblLastIntendedMode()
{
}

//------------------------------------------------------------------------------

// +++add: class implementation here if needed

//------------------------------------------------------------------------------

bool isValidInterfaceType(IN const BTSInterfaceType stackInterface)
{
   if((BTS_IF_UNDEFINED < stackInterface) && (stackInterface < BTS_IF_LAST))
   {
      return true;
   }
   else
   {
      return false;
   }
}

bool isValidComponentType(IN const BTSFunctionBlock component)
{
   if((BTS_FB_NONE < component) && (component < BTS_FB_LAST))
   {
      return true;
   }
   else
   {
      return false;
   }
}

} //btstackif
