/**
 * @file BtStackIfFactory.cpp
 *
 * @par SW-Component
 * Factory
 *
 * @brief BtStackIf instance factory.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details BtStackIf instance factory class.
 * Major task of this class is to provide the possibility to set test interfaces (unit test).
 */

#include "BtStackIfFactory.h"
#include "BtStackIfConnectionRequest.h"
#include "BtStackIfTelephonyRequest.h"
#include "BtStackIfPhonebookRequest.h"
#include "BtStackIfMessagingRequest.h"
#include "BtStackIfMediaPlayerRequest.h"
#include "BtStackIfWblRequest.h"
#include "BtStackIfEcnrRequest.h"
#include "FwAssert.h"

namespace btstackif {

BtStackIfFactory::BtStackIfFactory() :
_localCreate(0),
_externCreate(1),
_lock(),
_connectionIfList(),
_connectionCreatorList(),
_telephonyIfList(),
_telephonyCreatorList(),
_phonebookIfList(),
_phonebookCreatorList(),
_messagingIfList(),
_messagingCreatorList(),
_mediaPlayerIfList(),
_mediaPlayerCreatorList(),
_wblIfList(),
_wblCreatorList(),
_ecnrIfList(),
_ecnrCreatorList()
{
}

BtStackIfFactory::~BtStackIfFactory()
{
}

BtStackIfFactory& BtStackIfFactory::getInstance(void)
{
   static BtStackIfFactory factory;
   return factory;
}

IBtStackIfConnectionRequest* BtStackIfFactory::getConnectionRequestIf(IN const BTSFunctionBlock componentFunctionBlock /*= BTS_FB_CONNECTION*/, IN const BTSInterfaceType stackInterface /*= BTS_IF_ALPS_EVOLUTION_GENIVI*/)
{
   IBtStackIfConnectionRequest* returnIf(0);

   if(false == isValidInterfaceType(stackInterface))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return returnIf;
   }

   _lock.lock();

   ::std::map< BTSInterfaceType, IBtStackIfConnectionRequest* >::iterator it = _connectionIfList.find(stackInterface);
   if(_connectionIfList.end() != it)
   {
      returnIf = it->second;
   }
   else
   {
      returnIf = new BtStackIfConnectionRequest(componentFunctionBlock, stackInterface);
      if(0 != returnIf)
      {
         _connectionIfList[stackInterface] = returnIf;
         _connectionCreatorList[returnIf] = _localCreate;
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }

   _lock.unlock();

   return returnIf;
}

void BtStackIfFactory::destroyConnectionRequestIf(IN const IBtStackIfConnectionRequest* requestIf)
{
   FW_IF_NULL_PTR_RETURN(requestIf);

   _lock.lock();

   for(::std::map< BTSInterfaceType, IBtStackIfConnectionRequest* >::iterator itIf = _connectionIfList.begin(); itIf != _connectionIfList.end(); ++itIf)
   {
      if(requestIf == itIf->second)
      {
         int creator(_localCreate);
         ::std::map< const IBtStackIfConnectionRequest*, int >::iterator itCreator = _connectionCreatorList.find(requestIf);
         if(_connectionCreatorList.end() != itCreator)
         {
            creator = itCreator->second;
            _connectionCreatorList.erase(itCreator);
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }

         if(_localCreate == creator)
         {
            delete requestIf;
         }

         _connectionIfList.erase(itIf);

         break;
      }
   }

   _lock.unlock();
}

IBtStackIfTelephonyRequest* BtStackIfFactory::getTelephonyRequestIf(IN const BTSFunctionBlock componentFunctionBlock /*= BTS_FB_TELEPHONY*/, IN const BTSInterfaceType stackInterface /*= BTS_IF_ALPS_EVOLUTION_GENIVI*/)
{
   IBtStackIfTelephonyRequest* returnIf(0);

   if(false == isValidInterfaceType(stackInterface))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return returnIf;
   }

   _lock.lock();

   ::std::map< BTSInterfaceType, IBtStackIfTelephonyRequest* >::iterator it = _telephonyIfList.find(stackInterface);
   if(_telephonyIfList.end() != it)
   {
      returnIf = it->second;
   }
   else
   {
      returnIf = new BtStackIfTelephonyRequest(componentFunctionBlock, stackInterface);
      if(0 != returnIf)
      {
         _telephonyIfList[stackInterface] = returnIf;
         _telephonyCreatorList[returnIf] = _localCreate;
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }

   _lock.unlock();

   return returnIf;
}

void BtStackIfFactory::destroyTelephonyRequestIf(IN const IBtStackIfTelephonyRequest* requestIf)
{
   FW_IF_NULL_PTR_RETURN(requestIf);

   _lock.lock();

   for(::std::map< BTSInterfaceType, IBtStackIfTelephonyRequest* >::iterator itIf = _telephonyIfList.begin(); itIf != _telephonyIfList.end(); ++itIf)
   {
      if(requestIf == itIf->second)
      {
         int creator(_localCreate);
         ::std::map< const IBtStackIfTelephonyRequest*, int >::iterator itCreator = _telephonyCreatorList.find(requestIf);
         if(_telephonyCreatorList.end() != itCreator)
         {
            creator = itCreator->second;
            _telephonyCreatorList.erase(itCreator);
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }

         if(_localCreate == creator)
         {
            delete requestIf;
         }

         _telephonyIfList.erase(itIf);

         break;
      }
   }

   _lock.unlock();
}

IBtStackIfPhonebookRequest* BtStackIfFactory::getPhonebookRequestIf(IN const BTSFunctionBlock componentFunctionBlock /*= BTS_FB_PHONEBOOK*/, IN const BTSInterfaceType stackInterface /*= BTS_IF_ALPS_EVOLUTION_GENIVI*/)
{
   IBtStackIfPhonebookRequest* returnIf(0);

   if(false == isValidInterfaceType(stackInterface))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return returnIf;
   }

   _lock.lock();

   ::std::map< BTSInterfaceType, IBtStackIfPhonebookRequest* >::iterator it = _phonebookIfList.find(stackInterface);
   if(_phonebookIfList.end() != it)
   {
      returnIf = it->second;
   }
   else
   {
      returnIf = new BtStackIfPhonebookRequest(componentFunctionBlock, stackInterface);
      if(0 != returnIf)
      {
         _phonebookIfList[stackInterface] = returnIf;
         _phonebookCreatorList[returnIf] = _localCreate;
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }

   _lock.unlock();

   return returnIf;
}

void BtStackIfFactory::destroyPhonebookRequestIf(IN const IBtStackIfPhonebookRequest* requestIf)
{
   FW_IF_NULL_PTR_RETURN(requestIf);

   _lock.lock();

   for(::std::map< BTSInterfaceType, IBtStackIfPhonebookRequest* >::iterator itIf = _phonebookIfList.begin(); itIf != _phonebookIfList.end(); ++itIf)
   {
      if(requestIf == itIf->second)
      {
         int creator(_localCreate);
         ::std::map< const IBtStackIfPhonebookRequest*, int >::iterator itCreator = _phonebookCreatorList.find(requestIf);
         if(_phonebookCreatorList.end() != itCreator)
         {
            creator = itCreator->second;
            _phonebookCreatorList.erase(itCreator);
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }

         if(_localCreate == creator)
         {
            delete requestIf;
         }

         _phonebookIfList.erase(itIf);

         break;
      }
   }

   _lock.unlock();
}

IBtStackIfMessagingRequest* BtStackIfFactory::getMessagingRequestIf(IN const BTSFunctionBlock componentFunctionBlock /*= BTS_FB_MESSAGING*/, IN const BTSInterfaceType stackInterface /*= BTS_IF_ALPS_EVOLUTION_GENIVI*/)
{
   IBtStackIfMessagingRequest* returnIf(0);

   if(false == isValidInterfaceType(stackInterface))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return returnIf;
   }

   _lock.lock();

   ::std::map< BTSInterfaceType, IBtStackIfMessagingRequest* >::iterator it = _messagingIfList.find(stackInterface);
   if(_messagingIfList.end() != it)
   {
      returnIf = it->second;
   }
   else
   {
      returnIf = new BtStackIfMessagingRequest(componentFunctionBlock, stackInterface);
      if(0 != returnIf)
      {
         _messagingIfList[stackInterface] = returnIf;
         _messagingCreatorList[returnIf] = _localCreate;
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }

   _lock.unlock();

   return returnIf;
}

void BtStackIfFactory::destroyMessagingRequestIf(IN const IBtStackIfMessagingRequest* requestIf)
{
   FW_IF_NULL_PTR_RETURN(requestIf);

   _lock.lock();

   for(::std::map< BTSInterfaceType, IBtStackIfMessagingRequest* >::iterator itIf = _messagingIfList.begin(); itIf != _messagingIfList.end(); ++itIf)
   {
      if(requestIf == itIf->second)
      {
         int creator(_localCreate);
         ::std::map< const IBtStackIfMessagingRequest*, int >::iterator itCreator = _messagingCreatorList.find(requestIf);
         if(_messagingCreatorList.end() != itCreator)
         {
            creator = itCreator->second;
            _messagingCreatorList.erase(itCreator);
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }

         if(_localCreate == creator)
         {
            delete requestIf;
         }

         _messagingIfList.erase(itIf);

         break;
      }
   }

   _lock.unlock();
}

IBtStackIfMediaPlayerRequest* BtStackIfFactory::getMediaPlayerRequestIf(IN const BTSFunctionBlock componentFunctionBlock /*= BTS_FB_MEDIAPLAYER*/, IN const BTSInterfaceType stackInterface /*= BTS_IF_ALPS_EVOLUTION_GENIVI*/)
{
   IBtStackIfMediaPlayerRequest* returnIf(0);

   if(false == isValidInterfaceType(stackInterface))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return returnIf;
   }

   _lock.lock();

   ::std::map< BTSInterfaceType, IBtStackIfMediaPlayerRequest* >::iterator it = _mediaPlayerIfList.find(stackInterface);
   if(_mediaPlayerIfList.end() != it)
   {
      returnIf = it->second;
   }
   else
   {
      returnIf = new BtStackIfMediaPlayerRequest(componentFunctionBlock, stackInterface);
      if(0 != returnIf)
      {
         _mediaPlayerIfList[stackInterface] = returnIf;
         _mediaPlayerCreatorList[returnIf] = _localCreate;
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }

   _lock.unlock();

   return returnIf;
}

void BtStackIfFactory::destroyMediaPlayerRequestIf(IN const IBtStackIfMediaPlayerRequest* requestIf)
{
   FW_IF_NULL_PTR_RETURN(requestIf);

   _lock.lock();

   for(::std::map< BTSInterfaceType, IBtStackIfMediaPlayerRequest* >::iterator itIf = _mediaPlayerIfList.begin(); itIf != _mediaPlayerIfList.end(); ++itIf)
   {
      if(requestIf == itIf->second)
      {
         int creator(_localCreate);
         ::std::map< const IBtStackIfMediaPlayerRequest*, int >::iterator itCreator = _mediaPlayerCreatorList.find(requestIf);
         if(_mediaPlayerCreatorList.end() != itCreator)
         {
            creator = itCreator->second;
            _mediaPlayerCreatorList.erase(itCreator);
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }

         if(_localCreate == creator)
         {
            delete requestIf;
         }

         _mediaPlayerIfList.erase(itIf);

         break;
      }
   }

   _lock.unlock();
}

IBtStackIfWblRequest* BtStackIfFactory::getWblRequestIf(IN const BTSFunctionBlock componentFunctionBlock /*= BTS_FB_CONNECTION*/, IN const BTSInterfaceType stackInterface /*= BTS_IF_ALPS_EVOLUTION_GENIVI*/)
{
   IBtStackIfWblRequest* returnIf(0);

   if(false == isValidInterfaceType(stackInterface))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return returnIf;
   }

   _lock.lock();

   // limit to 1 instance overall
   //::std::map< BTSInterfaceType, IBtStackIfWblRequest* >::iterator it = _wblIfList.find(stackInterface);
   ::std::map< BTSInterfaceType, IBtStackIfWblRequest* >::iterator it = _wblIfList.begin();
   if(_wblIfList.end() != it)
   {
      returnIf = it->second;
   }
   else
   {
      returnIf = new BtStackIfWblRequest(componentFunctionBlock, stackInterface);
      if(0 != returnIf)
      {
         _wblIfList[stackInterface] = returnIf;
         _wblCreatorList[returnIf] = _localCreate;
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }

   _lock.unlock();

   return returnIf;
}

void BtStackIfFactory::destroyWblRequestIf(IN const IBtStackIfWblRequest* requestIf)
{
   FW_IF_NULL_PTR_RETURN(requestIf);

   _lock.lock();

   for(::std::map< BTSInterfaceType, IBtStackIfWblRequest* >::iterator itIf = _wblIfList.begin(); itIf != _wblIfList.end(); ++itIf)
   {
      if(requestIf == itIf->second)
      {
         int creator(_localCreate);
         ::std::map< const IBtStackIfWblRequest*, int >::iterator itCreator = _wblCreatorList.find(requestIf);
         if(_wblCreatorList.end() != itCreator)
         {
            creator = itCreator->second;
            _wblCreatorList.erase(itCreator);
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }

         if(_localCreate == creator)
         {
            delete requestIf;
         }

         _wblIfList.erase(itIf);

         break;
      }
   }

   _lock.unlock();
}

IBtStackIfEcnrRequest* BtStackIfFactory::getEcnrRequestIf(IN const BTSFunctionBlock componentFunctionBlock /*= BTS_FB_TELEPHONY*/, IN const BTSInterfaceType stackInterface /*= BTS_IF_ALPS_EVOLUTION_GENIVI*/)
{
   IBtStackIfEcnrRequest* returnIf(0);

   if(false == isValidInterfaceType(stackInterface))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return returnIf;
   }

   _lock.lock();

   // limit to 1 instance overall
   //::std::map< BTSInterfaceType, IBtStackIfEcnrRequest* >::iterator it = _ecnrIfList.find(stackInterface);
   ::std::map< BTSInterfaceType, IBtStackIfEcnrRequest* >::iterator it = _ecnrIfList.begin();
   if(_ecnrIfList.end() != it)
   {
      returnIf = it->second;
   }
   else
   {
      returnIf = new BtStackIfEcnrRequest(componentFunctionBlock, stackInterface);
      if(0 != returnIf)
      {
         _ecnrIfList[stackInterface] = returnIf;
         _ecnrCreatorList[returnIf] = _localCreate;
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }

   _lock.unlock();

   return returnIf;
}

void BtStackIfFactory::destroyEcnrRequestIf(IN const IBtStackIfEcnrRequest* requestIf)
{
   FW_IF_NULL_PTR_RETURN(requestIf);

   _lock.lock();

   for(::std::map< BTSInterfaceType, IBtStackIfEcnrRequest* >::iterator itIf = _ecnrIfList.begin(); itIf != _ecnrIfList.end(); ++itIf)
   {
      if(requestIf == itIf->second)
      {
         int creator(_localCreate);
         ::std::map< const IBtStackIfEcnrRequest*, int >::iterator itCreator = _ecnrCreatorList.find(requestIf);
         if(_ecnrCreatorList.end() != itCreator)
         {
            creator = itCreator->second;
            _ecnrCreatorList.erase(itCreator);
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }

         if(_localCreate == creator)
         {
            delete requestIf;
         }

         _ecnrIfList.erase(itIf);

         break;
      }
   }

   _lock.unlock();
}

void BtStackIfFactory::setConnectionRequestIfInstanceForTesting(IN IBtStackIfConnectionRequest* testInstance, IN const BTSInterfaceType stackInterface /*= BTS_IF_ALPS_EVOLUTION_GENIVI*/)
{
   FW_IF_NULL_PTR_RETURN(testInstance);

   if(false == isValidInterfaceType(stackInterface))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   _lock.lock();

   ::std::map< BTSInterfaceType, IBtStackIfConnectionRequest* >::iterator it = _connectionIfList.find(stackInterface);
   if(_connectionIfList.end() != it)
   {
      // not allowed because already set => to be considered by unit test
   }
   else
   {
      _connectionIfList[stackInterface] = testInstance;
      _connectionCreatorList[testInstance] = _externCreate;
   }

   _lock.unlock();
}

void BtStackIfFactory::setTelephonyRequestIfInstanceForTesting(IN IBtStackIfTelephonyRequest* testInstance, IN const BTSInterfaceType stackInterface /*= BTS_IF_ALPS_EVOLUTION_GENIVI*/)
{
   FW_IF_NULL_PTR_RETURN(testInstance);

   if(false == isValidInterfaceType(stackInterface))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   _lock.lock();

   ::std::map< BTSInterfaceType, IBtStackIfTelephonyRequest* >::iterator it = _telephonyIfList.find(stackInterface);
   if(_telephonyIfList.end() != it)
   {
      // not allowed because already set => to be considered by unit test
   }
   else
   {
      _telephonyIfList[stackInterface] = testInstance;
      _telephonyCreatorList[testInstance] = _externCreate;
   }

   _lock.unlock();
}

void BtStackIfFactory::setPhonebookRequestIfInstanceForTesting(IN IBtStackIfPhonebookRequest* testInstance, IN const BTSInterfaceType stackInterface /*= BTS_IF_ALPS_EVOLUTION_GENIVI*/)
{
   FW_IF_NULL_PTR_RETURN(testInstance);

   if(false == isValidInterfaceType(stackInterface))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   _lock.lock();

   ::std::map< BTSInterfaceType, IBtStackIfPhonebookRequest* >::iterator it = _phonebookIfList.find(stackInterface);
   if(_phonebookIfList.end() != it)
   {
      // not allowed because already set => to be considered by unit test
   }
   else
   {
      _phonebookIfList[stackInterface] = testInstance;
      _phonebookCreatorList[testInstance] = _externCreate;
   }

   _lock.unlock();
}

void BtStackIfFactory::setMessagingRequestIfInstanceForTesting(IN IBtStackIfMessagingRequest* testInstance, IN const BTSInterfaceType stackInterface /*= BTS_IF_ALPS_EVOLUTION_GENIVI*/)
{
   FW_IF_NULL_PTR_RETURN(testInstance);

   if(false == isValidInterfaceType(stackInterface))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   _lock.lock();

   ::std::map< BTSInterfaceType, IBtStackIfMessagingRequest* >::iterator it = _messagingIfList.find(stackInterface);
   if(_messagingIfList.end() != it)
   {
      // not allowed because already set => to be considered by unit test
   }
   else
   {
      _messagingIfList[stackInterface] = testInstance;
      _messagingCreatorList[testInstance] = _externCreate;
   }

   _lock.unlock();
}

void BtStackIfFactory::setMediaPlayerRequestIfInstanceForTesting(IN IBtStackIfMediaPlayerRequest* testInstance, IN const BTSInterfaceType stackInterface /*= BTS_IF_ALPS_EVOLUTION_GENIVI*/)
{
   FW_IF_NULL_PTR_RETURN(testInstance);

   if(false == isValidInterfaceType(stackInterface))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   _lock.lock();

   ::std::map< BTSInterfaceType, IBtStackIfMediaPlayerRequest* >::iterator it = _mediaPlayerIfList.find(stackInterface);
   if(_mediaPlayerIfList.end() != it)
   {
      // not allowed because already set => to be considered by unit test
   }
   else
   {
      _mediaPlayerIfList[stackInterface] = testInstance;
      _mediaPlayerCreatorList[testInstance] = _externCreate;
   }

   _lock.unlock();
}

void BtStackIfFactory::setWblRequestIfInstanceForTesting(IN IBtStackIfWblRequest* testInstance, IN const BTSInterfaceType stackInterface /*= BTS_IF_ALPS_EVOLUTION_GENIVI*/)
{
   FW_IF_NULL_PTR_RETURN(testInstance);

   if(false == isValidInterfaceType(stackInterface))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   _lock.lock();

   // limit to 1 instance overall
   //::std::map< BTSInterfaceType, IBtStackIfWblRequest* >::iterator it = _wblIfList.find(stackInterface);
   ::std::map< BTSInterfaceType, IBtStackIfWblRequest* >::iterator it = _wblIfList.begin();
   if(_wblIfList.end() != it)
   {
      // not allowed because already set => to be considered by unit test
   }
   else
   {
      _wblIfList[stackInterface] = testInstance;
      _wblCreatorList[testInstance] = _externCreate;
   }

   _lock.unlock();
}

void BtStackIfFactory::setEcnrRequestIfInstanceForTesting(IN IBtStackIfEcnrRequest* testInstance, IN const BTSInterfaceType stackInterface /*= BTS_IF_ALPS_EVOLUTION_GENIVI*/)
{
   FW_IF_NULL_PTR_RETURN(testInstance);

   if(false == isValidInterfaceType(stackInterface))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   _lock.lock();

   // limit to 1 instance overall
   //::std::map< BTSInterfaceType, IBtStackIfEcnrRequest* >::iterator it = _ecnrIfList.find(stackInterface);
   ::std::map< BTSInterfaceType, IBtStackIfEcnrRequest* >::iterator it = _ecnrIfList.begin();
   if(_ecnrIfList.end() != it)
   {
      // not allowed because already set => to be considered by unit test
   }
   else
   {
      _ecnrIfList[stackInterface] = testInstance;
      _ecnrCreatorList[testInstance] = _externCreate;
   }

   _lock.unlock();
}

void BtStackIfFactory::setStackUserMode(IN BTSUserMode userMode)
{
   BtStackIfBaseRequest::setStackUserMode(userMode);
}

void BtStackIfFactory::setAsfLogConfigFile(IN const ::std::string& file)
{
   BtStackIfBaseRequest::setAsfLogConfigFile(file);
}

void BtStackIfFactory::printStatistics(void)
{
   BtStackIfBaseRequest::printStatistics();
}

} //btstackif
