/**
 * @file BtStackIfConnectionRequest.cpp
 *
 * @par SW-Component
 * Interface
 *
 * @brief Connection request interface.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of connection request functionality.
 */

#include "BtStackIfConnectionRequest.h"
#include "App2Bts_MessageWrapper.h"
#include "BtsUtils.h"
#include "FwAssert.h"

namespace btstackif {

BtStackIfConnectionRequest::BtStackIfConnectionRequest(IN const BTSFunctionBlock componentFunctionBlock /*= BTS_FB_CONNECTION*/, IN const BTSInterfaceType stackInterface /*= BTS_IF_ALPS_EVOLUTION_GENIVI*/) :
BtStackIfBaseRequest(BTS_USER_MODE_CONNECTION, componentFunctionBlock, BTS_FB_CONNECTION, stackInterface)
{
}

BtStackIfConnectionRequest::BtStackIfConnectionRequest() :
BtStackIfBaseRequest(BTS_USER_MODE_CONNECTION, BTS_FB_CONNECTION, BTS_FB_CONNECTION, BTS_IF_ALPS_EVOLUTION_GENIVI)
{
   // shall not be used
   FW_NORMAL_ASSERT_ALWAYS();
}

BtStackIfConnectionRequest::~BtStackIfConnectionRequest()
{
}

void BtStackIfConnectionRequest::registerCallback(IBtStackIfConnectionCallback* callback)
{
   if(0 == callback)
   {
      return;
   }

   if(_callback.getConnectionCallback())
   {
      // already registered
      return;
   }

   _callback.setConnectionCallback(callback);
   registerInternalCallback();
}

BTSSessionHandle BtStackIfConnectionRequest::getSessionHandle(void) const
{
   return getHandle();
}

BTSErrorCode BtStackIfConnectionRequest::init(IN const BTSLocalStackConfiguration& configuration, IN const BTSBDAddressList& pairedDevices) const
{
   BTSLocalConfigurationContainer container;
   container.connectionConfiguration.configuration = configuration;
   container.connectionConfiguration.pairedDevices = pairedDevices;
   container.connectionActive = true;
   return processInit(container);
}

BTSErrorCode BtStackIfConnectionRequest::deInit(void) const
{
   return processDeInit();
}

void BtStackIfConnectionRequest::switchBluetoothOnOff(IN const BTSBluetoothMode bluetoothMode, IN const BTSAdapterMode adapterMode)
{
   App2Bts_SwitchBluetoothOnOff* ptrMessage = ptrNew_App2Bts_SwitchBluetoothOnOff();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBluetoothMode(bluetoothMode);
      ptrMessage->setAdapterMode(adapterMode);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::getBluetoothOnOffStatus(void)
{
   App2Bts_GetBluetoothOnOffStatus* ptrMessage = ptrNew_App2Bts_GetBluetoothOnOffStatus();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::switchBluetoothHardwareOnOff(IN const BTSBluetoothMode bluetoothMode)
{
   App2Bts_SwitchBluetoothHardwareOnOff* ptrMessage = ptrNew_App2Bts_SwitchBluetoothHardwareOnOff();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBluetoothMode(bluetoothMode);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::setLocalBtName(IN const BTSDeviceName& adapterName)
{
   App2Bts_SetLocalBtName* ptrMessage = ptrNew_App2Bts_SetLocalBtName();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setDeviceName(adapterName);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::getLocalBtName(void)
{
   App2Bts_GetLocalBtName* ptrMessage = ptrNew_App2Bts_GetLocalBtName();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::setDeviceIdServiceRecord(IN const BTSDeviceIdServiceRecord& serviceRecord)
{
   App2Bts_SetDeviceIdServiceRecord* ptrMessage = ptrNew_App2Bts_SetDeviceIdServiceRecord();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setDeviceIdServiceRecord(serviceRecord);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::setDeviceForRestrictedPairingConnecting(IN const BTSBDAddress& expectedDeviceAddress, IN const BTSSessionHandle sessionHandle)
{
   if(true != isValidBdAddress(expectedDeviceAddress))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   App2Bts_SetDeviceForRestrictedPairingConnecting* ptrMessage = ptrNew_App2Bts_SetDeviceForRestrictedPairingConnecting();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBDAddress(expectedDeviceAddress);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::setLocalAdapterModes(IN const BTSLocalMode discoverableMode, IN const BTSDiscoverableTimeout discoverableTimeout, IN const BTSLocalMode connectableMode, IN const BTSConnectableTimeout connectableTimeout, IN const BTSSessionHandle sessionHandle)
{
   App2Bts_SetLocalAdapterModes* ptrMessage = ptrNew_App2Bts_SetLocalAdapterModes();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setDiscoverableMode(discoverableMode);
      ptrMessage->setDiscoverableTimeout(discoverableTimeout);
      ptrMessage->setConnectableMode(connectableMode);
      ptrMessage->setConnectableTimeout(connectableTimeout);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::getLocalAdapterModes(IN const BTSSessionHandle sessionHandle)
{
   App2Bts_GetLocalAdapterModes* ptrMessage = ptrNew_App2Bts_GetLocalAdapterModes();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::startDiscovery(IN const BTSSessionHandle sessionHandle)
{
   App2Bts_StartDiscovery* ptrMessage = ptrNew_App2Bts_StartDiscovery();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::stopDiscovery(IN const BTSSessionHandle sessionHandle)
{
   App2Bts_StopDiscovery* ptrMessage = ptrNew_App2Bts_StopDiscovery();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::getDiscoveringStatus(IN const BTSSessionHandle sessionHandle)
{
   App2Bts_GetDiscoveringStatus* ptrMessage = ptrNew_App2Bts_GetDiscoveringStatus();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::startPairing(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle)
{
   if(true != isValidBdAddress(deviceAddress, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   App2Bts_StartPairing* ptrMessage = ptrNew_App2Bts_StartPairing();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBDAddress(deviceAddress);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::cancelPairing(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle)
{
   if(true != isValidBdAddress(deviceAddress, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   App2Bts_CancelPairing* ptrMessage = ptrNew_App2Bts_CancelPairing();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBDAddress(deviceAddress);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::setPinCode(IN const BTSBDAddress& deviceAddress, IN const BTSPinCode& pinCode, IN const BTSSessionHandle sessionHandle)
{
   if(true != isValidBdAddress(deviceAddress, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   if(true != isValidPinCode(pinCode))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   App2Bts_SetPinCode* ptrMessage = ptrNew_App2Bts_SetPinCode();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBDAddress(deviceAddress);
      ptrMessage->setPinCode(pinCode);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::confirmSecureSimplePairing(IN const BTSBDAddress& deviceAddress, IN const BTSSspMode sspMode, IN const BTSNumericValue& numericValue, IN const BTSConfirmationMode confirmationMode, IN const BTSSessionHandle sessionHandle)
{
   if(true != isValidBdAddress(deviceAddress, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   if(true != isValidNumericValue(numericValue))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   App2Bts_ConfirmSecureSimplePairing* ptrMessage = ptrNew_App2Bts_ConfirmSecureSimplePairing();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBDAddress(deviceAddress);
      ptrMessage->setSspMode(sspMode);
      ptrMessage->setNumericValue(numericValue);
      ptrMessage->setConfirmationMode(confirmationMode);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::setLinkKey(IN const BTSBDAddress& deviceAddress, IN const BTSLinkKeyType linkKeyType, IN const BTSLinkKey& linkKey, IN const BTSDLinkKey& dLinkKey, IN const BTSSessionHandle sessionHandle)
{
   if(true != isValidBdAddress(deviceAddress, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   if(true != isValidLinkKey(linkKey))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   if(true != isValidDLinkKey(dLinkKey))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   App2Bts_SetLinkKey* ptrMessage = ptrNew_App2Bts_SetLinkKey();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBDAddress(deviceAddress);
      ptrMessage->setLinkKeyType(linkKeyType);
      ptrMessage->setLinkKey(linkKey);
      ptrMessage->setDLinkKey(dLinkKey);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::connectProtocol(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const BTSTimeValue pageTimeout, IN const BTSSessionHandle sessionHandle)
{
   if(true != isValidBdAddress(deviceAddress, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   if(true != isValidUuid(sppUuid))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   App2Bts_ConnectProtocol* ptrMessage = ptrNew_App2Bts_ConnectProtocol();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBDAddress(deviceAddress);
      ptrMessage->setProtocolId(protocolId);
      ptrMessage->setUuid(sppUuid);
      ptrMessage->setMasInstanceName(masInstance);
      ptrMessage->setTimeValue(pageTimeout);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::acceptRemoteProtocolConnect(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSSessionHandle sessionHandle)
{
   if(true != isValidBdAddress(deviceAddress, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   if(true != isValidUuid(sppUuid))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   App2Bts_AcceptRemoteProtocolConnect* ptrMessage = ptrNew_App2Bts_AcceptRemoteProtocolConnect();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBDAddress(deviceAddress);
      ptrMessage->setProtocolId(protocolId);
      ptrMessage->setUuid(sppUuid);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::rejectRemoteProtocolConnect(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSSessionHandle sessionHandle)
{
   if(true != isValidBdAddress(deviceAddress, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   if(true != isValidUuid(sppUuid))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   App2Bts_RejectRemoteProtocolConnect* ptrMessage = ptrNew_App2Bts_RejectRemoteProtocolConnect();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBDAddress(deviceAddress);
      ptrMessage->setProtocolId(protocolId);
      ptrMessage->setUuid(sppUuid);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::disconnectProtocol(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const bool pauseBtStreaming, IN const BTSSessionHandle sessionHandle)
{
   if(true != isValidBdAddress(deviceAddress, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   if(true != isValidUuid(sppUuid))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   App2Bts_DisconnectProtocol* ptrMessage = ptrNew_App2Bts_DisconnectProtocol();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBDAddress(deviceAddress);
      ptrMessage->setProtocolId(protocolId);
      ptrMessage->setUuid(sppUuid);
      ptrMessage->setMasInstanceName(masInstance);
      ptrMessage->setPauseBtStreaming(pauseBtStreaming);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::disconnectDevice(IN const BTSBDAddress& deviceAddress, IN const bool pauseBtStreaming, IN const BTSSessionHandle sessionHandle)
{
   if(true != isValidBdAddress(deviceAddress, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   App2Bts_DisconnectDevice* ptrMessage = ptrNew_App2Bts_DisconnectDevice();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBDAddress(deviceAddress);
      ptrMessage->setPauseBtStreaming(pauseBtStreaming);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::getProtocolConnectionStatus(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const BTSSessionHandle sessionHandle)
{
   if(true != isValidBdAddress(deviceAddress, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   if(true != isValidUuid(sppUuid))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   App2Bts_GetProtocolConnectionStatus* ptrMessage = ptrNew_App2Bts_GetProtocolConnectionStatus();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBDAddress(deviceAddress);
      ptrMessage->setProtocolId(protocolId);
      ptrMessage->setUuid(sppUuid);
      ptrMessage->setMasInstanceName(masInstance);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::getDeviceConnectionStatus(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle)
{
   if(true != isValidBdAddress(deviceAddress, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   App2Bts_GetDeviceConnectionStatus* ptrMessage = ptrNew_App2Bts_GetDeviceConnectionStatus();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBDAddress(deviceAddress);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::startRemoteServiceSearch(IN const BTSBDAddress& deviceAddress, IN const BTSSearchType searchType, IN const bool rejectIfNotConnected, IN const BTSSessionHandle sessionHandle)
{
   if(true != isValidBdAddress(deviceAddress, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   App2Bts_StartRemoteServiceSearch* ptrMessage = ptrNew_App2Bts_StartRemoteServiceSearch();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBDAddress(deviceAddress);
      ptrMessage->setSearchType(searchType);
      ptrMessage->setRejectIfNotConnected(rejectIfNotConnected);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::cancelRemoteServiceSearch(IN const BTSBDAddress& deviceAddress, IN const BTSSearchType searchType, IN const BTSSessionHandle sessionHandle)
{
   if(true != isValidBdAddress(deviceAddress, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   App2Bts_CancelRemoteServiceSearch* ptrMessage = ptrNew_App2Bts_CancelRemoteServiceSearch();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBDAddress(deviceAddress);
      ptrMessage->setSearchType(searchType);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::queryRemoteName(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle)
{
   if(true != isValidBdAddress(deviceAddress, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   App2Bts_QueryRemoteName* ptrMessage = ptrNew_App2Bts_QueryRemoteName();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBDAddress(deviceAddress);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::requestLinkQuality(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle)
{
   if(true != isValidBdAddress(deviceAddress, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   App2Bts_RequestLinkQuality* ptrMessage = ptrNew_App2Bts_RequestLinkQuality();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBDAddress(deviceAddress);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::startTestMode(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle)
{
   if(true != isValidBdAddress(deviceAddress, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   App2Bts_StartTestMode* ptrMessage = ptrNew_App2Bts_StartTestMode();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBDAddress(deviceAddress);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::stopTestMode(IN const BTSSessionHandle sessionHandle)
{
   App2Bts_StopTestMode* ptrMessage = ptrNew_App2Bts_StopTestMode();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::requestTestModeLinkQuality(IN const BTSSessionHandle sessionHandle)
{
   App2Bts_RequestTestModeLinkQuality* ptrMessage = ptrNew_App2Bts_RequestTestModeLinkQuality();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setSessionHandle(sessionHandle);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConnectionRequest::setSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData) const
{
   processSimulationTestCommand(testCommand, testData);
}

void BtStackIfConnectionRequest::setSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData) const
{
   processSimulationTestCommand(testCommand, testData);
}

void BtStackIfConnectionRequest::setSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData) const
{
   processSimulationTestCommand(testCommand, testData);
}

void BtStackIfConnectionRequest::setMainControlTestCommand(IN const char* testCommand, IN const unsigned int testData) const
{
   processMainControlTestCommand(testCommand, testData);
}

void BtStackIfConnectionRequest::setConnectionControlTestCommand(IN const char* testCommand, IN const unsigned int testData) const
{
   processConnectionControlTestCommand(testCommand, testData);
}

void BtStackIfConnectionRequest::setConnectionControlTestCommand(IN const char* testCommand, IN const unsigned char* testData) const
{
   processConnectionControlTestCommand(testCommand, testData);
}

} //btstackif
