/**
 * @file BtStackIfConfigRequest.cpp
 *
 * @par SW-Component
 * Interface
 *
 * @brief Config request interface.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of config request functionality.
 */

#include "BtStackIfConfigRequest.h"
#include "App2Bts_MessageWrapper.h"
#include "BtsUtils.h"
#include "FwAssert.h"

namespace btstackif {

BtStackIfConfigRequest::BtStackIfConfigRequest(IN const BTSFunctionBlock componentFunctionBlock /*= BTS_FB_CONFIG*/, IN const BTSInterfaceType stackInterface /*= BTS_IF_ALPS_EVOLUTION_GENIVI*/) :
BtStackIfBaseRequest(BTS_USER_MODE_CONNECTION, componentFunctionBlock, BTS_FB_CONFIG, stackInterface)
{
}

BtStackIfConfigRequest::BtStackIfConfigRequest() :
BtStackIfBaseRequest(BTS_USER_MODE_CONNECTION, BTS_FB_CONFIG, BTS_FB_CONFIG, BTS_IF_ALPS_EVOLUTION_GENIVI)
{
   // shall not be used
   FW_NORMAL_ASSERT_ALWAYS();
}

BtStackIfConfigRequest::~BtStackIfConfigRequest()
{
}

void BtStackIfConfigRequest::registerCallback(IBtStackIfConfigCallback* callback)
{
   if(0 == callback)
   {
      return;
   }

   if(_callback.getConfigCallback())
   {
      // already registered
      return;
   }

   _callback.setConfigCallback(callback);
   registerInternalCallback();
}

BTSSessionHandle BtStackIfConfigRequest::getSessionHandle(void) const
{
   return getHandle();
}

BTSErrorCode BtStackIfConfigRequest::init(IN const BTSLocalConfigConfiguration& configuration) const
{
   BTSLocalConfigurationContainer container;
   container.configConfiguration = configuration;
   container.configActive = true;
   return processInit(container);
}

BTSErrorCode BtStackIfConfigRequest::deInit(void) const
{
   return processDeInit();
}

void BtStackIfConfigRequest::setDummy(IN const BTSBDAddress& localDeviceAddress, IN const BTSBDAddress& remoteDeviceAddress)
{
   if(true != isValidBdAddress(localDeviceAddress, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   if(true != isValidBdAddress(remoteDeviceAddress, false))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   App2Bts_SetDummy* ptrMessage = ptrNew_App2Bts_SetDummy();
   if(ptrMessage)
   {
      ptrMessage->setCallback(&_callback);
      ptrMessage->setComponent(_componentFunctionBlock);
      ptrMessage->setStackInterface(_stackInterface);
      ptrMessage->setBDAddress(localDeviceAddress);
      ptrMessage->setRemoteDeviceAddress(remoteDeviceAddress);

      processApp2BtsMsg(ptrMessage);
   }
}

void BtStackIfConfigRequest::setSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData) const
{
   processSimulationTestCommand(testCommand, testData);
}

void BtStackIfConfigRequest::setSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData) const
{
   processSimulationTestCommand(testCommand, testData);
}

void BtStackIfConfigRequest::setSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData) const
{
   processSimulationTestCommand(testCommand, testData);
}

void BtStackIfConfigRequest::setMainControlTestCommand(IN const char* testCommand, IN const unsigned int testData) const
{
   processMainControlTestCommand(testCommand, testData);
}

void BtStackIfConfigRequest::setConnectionControlTestCommand(IN const char* testCommand, IN const unsigned int testData) const
{
   processConnectionControlTestCommand(testCommand, testData);
}

void BtStackIfConfigRequest::setConnectionControlTestCommand(IN const char* testCommand, IN const unsigned char* testData) const
{
   processConnectionControlTestCommand(testCommand, testData);
}

} //btstackif
