/**
 * @file BtStackIfBaseRequest.cpp
 *
 * @par SW-Component
 * Interface
 *
 * @brief Basic request interface.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of basic request functionality.
 */

#include "BtStackIfBaseRequest.h"
#include "InstanceFactory.h"
#include "App2Bts_BaseMessage.h"
#include "IGenericStackIf.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"
#include "FwUtils.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BtStackIfBaseRequest.cpp.trc.h"
#else
#include "BtStackIfTypesTrace.h"
#endif
#endif

namespace btstackif {

bool BtStackIfBaseRequest::_stackUserModeSet = false;
BTSUserMode BtStackIfBaseRequest::_stackUserMode = BTS_USER_MODE_CONNECTION;
::std::string BtStackIfBaseRequest::_asfConfigFile = "/var/opt/bosch/dynamic/connectivity/logging/asfLogConfigConnectivity.json"; // default ASF log configuration file
IGenericStackIf* BtStackIfBaseRequest::_genericStackIf = 0;
LockForeverAndNonReentrant BtStackIfBaseRequest::_sessionHandleLock;
BTSSessionHandle BtStackIfBaseRequest::_sessionHandleCounter = 1;
const ::std::string BtStackIfBaseRequest::_disableBtStackCommFile("/var/opt/bosch/dynamic/connectivity/logging/disable_bt.txt");

BtStackIfBaseRequest::BtStackIfBaseRequest(IN const BTSUserMode userMode, IN const BTSFunctionBlock componentFunctionBlock, IN const BTSFunctionBlock subComponentFunctionBlock, IN const BTSInterfaceType stackInterface) :
_callback(),
_componentFunctionBlock(componentFunctionBlock),
_subComponentFunctionBlock(subComponentFunctionBlock),
_stackInterface(stackInterface)
{
   _sessionHandleLock.lock();

   if(0 == _genericStackIf)
   {
      // set user mode (first one wins)
      setStackUserMode(getUserMode(userMode));

      // create general generic interface
      _genericStackIf = InstanceFactory::getInstance().getGenericStackIfInstance(_stackUserMode, _asfConfigFile);
   }

   if(_genericStackIf)
   {
      // register user
      _genericStackIf->registerUser(_componentFunctionBlock, _stackInterface, _subComponentFunctionBlock, this);
      // create environment for related stack interface and sub component
      _genericStackIf->createEnvironment(_componentFunctionBlock, _stackInterface, _subComponentFunctionBlock);
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   _sessionHandleLock.unlock();
}

BtStackIfBaseRequest::BtStackIfBaseRequest() :
_callback(),
_componentFunctionBlock(BTS_FB_CONNECTION),
_subComponentFunctionBlock(BTS_FB_CONNECTION),
_stackInterface(BTS_IF_ALPS_EVOLUTION_GENIVI)
{
   // shall not be used
   FW_NORMAL_ASSERT_ALWAYS();
}

BtStackIfBaseRequest::~BtStackIfBaseRequest()
{
   _sessionHandleLock.lock();

   if(_genericStackIf)
   {
      // unregister user
      _genericStackIf->unregisterUser(_componentFunctionBlock, _stackInterface, _subComponentFunctionBlock, this);

      // decreate depending on available registered users
      if(0 == _genericStackIf->getNumberRegisteredUsers())
      {
         (void)_genericStackIf->deInit(_componentFunctionBlock, _stackInterface, _subComponentFunctionBlock, true);
         InstanceFactory::getInstance().destroyGenericStackIfInstance();
         _genericStackIf = 0;
         _sessionHandleCounter = 1;
      }
   }

   _sessionHandleLock.unlock();
}

void BtStackIfBaseRequest::setStackUserMode(IN const BTSUserMode userMode)
{
   if(BTS_USER_MODE_LAST <= userMode)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(true == _stackUserModeSet)
   {
      // already set; ignore given value
      return;
   }

   _stackUserModeSet = true;
   _stackUserMode = userMode;

#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
   ETG_TRACE_USR1((" setStackUserMode(): _stackUserMode=%d", ETG_ENUM(TRC_BTSUserMode, _stackUserMode)));
#else
   BTS_TRACE_USR1((" setStackUserMode(): _stackUserMode=%s", getUserMode2String(_stackUserMode)));
#endif
}

void BtStackIfBaseRequest::setAsfLogConfigFile(IN const ::std::string& file)
{
   if(true == file.empty())
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   _asfConfigFile = file;
}

void BtStackIfBaseRequest::printStatistics(void)
{
   if(_genericStackIf)
   {
      _genericStackIf->printStatistics();
   }
}

BTSSessionHandle BtStackIfBaseRequest::getHandle(void) const
{
   BTSSessionHandle handle;

   _sessionHandleLock.lock();

   handle = _sessionHandleCounter++;

   _sessionHandleLock.unlock();

   return handle;
}

void BtStackIfBaseRequest::registerInternalCallback(void)
{
   if(_genericStackIf)
   {
      _genericStackIf->registerCallback(_componentFunctionBlock, _stackInterface, _subComponentFunctionBlock, this, &_callback);
   }
}

BTSErrorCode BtStackIfBaseRequest::processInit(IN const BTSLocalConfigurationContainer& configuration) const
{
   if(_genericStackIf)
   {
      return _genericStackIf->init(_componentFunctionBlock, _stackInterface, _subComponentFunctionBlock, configuration);
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return BTS_LIB_NOT_INITIALIZED;
   }
}

BTSErrorCode BtStackIfBaseRequest::processDeInit(void) const
{
   if(_genericStackIf)
   {
      return _genericStackIf->deInit(_componentFunctionBlock, _stackInterface, _subComponentFunctionBlock);
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return BTS_LIB_NOT_INITIALIZED;
   }
}

void BtStackIfBaseRequest::processApp2BtsMsg(IN App2Bts_BaseMessage* message)
{
   FW_IF_NULL_PTR_RETURN(message);

   if(_genericStackIf)
   {
      _genericStackIf->processApp2BtsMsg(message);
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
      delete message;
   }
}

void BtStackIfBaseRequest::processSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData) const
{
   if(_genericStackIf)
   {
      _genericStackIf->setSimulationTestCommand(_componentFunctionBlock, _stackInterface, _subComponentFunctionBlock, testCommand, testData);
   }
}

void BtStackIfBaseRequest::processSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData) const
{
   if(_genericStackIf)
   {
      _genericStackIf->setSimulationTestCommand(_componentFunctionBlock, _stackInterface, _subComponentFunctionBlock, testCommand, testData);
   }
}

void BtStackIfBaseRequest::processSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData) const
{
   if(_genericStackIf)
   {
      _genericStackIf->setSimulationTestCommand(_componentFunctionBlock, _stackInterface, _subComponentFunctionBlock, testCommand, testData);
   }
}

void BtStackIfBaseRequest::processMainControlTestCommand(IN const char* testCommand, IN const unsigned int testData) const
{
   if(_genericStackIf)
   {
      _genericStackIf->setMainControlTestCommand(_componentFunctionBlock, _stackInterface, _subComponentFunctionBlock, testCommand, testData);
   }
}

void BtStackIfBaseRequest::processConnectionControlTestCommand(IN const char* testCommand, IN const unsigned int testData) const
{
   if(_genericStackIf)
   {
      _genericStackIf->setConnectionControlTestCommand(_componentFunctionBlock, _stackInterface, _subComponentFunctionBlock, testCommand, testData);
   }
}

void BtStackIfBaseRequest::processConnectionControlTestCommand(IN const char* testCommand, IN const unsigned char* testData) const
{
   if(_genericStackIf)
   {
      _genericStackIf->setConnectionControlTestCommand(_componentFunctionBlock, _stackInterface, _subComponentFunctionBlock, testCommand, testData);
   }
}

BTSUserMode BtStackIfBaseRequest::getUserMode(IN const BTSUserMode currentMode) const
{
   // check if file to disable BT stack communication is available
   if(true == ::fw::doesFileExist(_disableBtStackCommFile))
   {
      return BTS_USER_MODE_BT_DISABLED;
   }
   else
   {
      return currentMode;
   }
}

} //btstackif
