/**
 * @file BtStackIfFactory.h
 *
 * @par SW-Component
 * BtStackIf instance factory
 *
 * @brief BtStackIf instance factory class.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Major task of this class is to provide the possibility to set
 *          test interfaces (unit test).
 */

#ifndef _BT_STACK_IF_FACTORY_H_
#define _BT_STACK_IF_FACTORY_H_

#include "BtStackIfTypes.h"
#include "FwLock.h"

namespace btstackif {

// class forward declarations
class IBtStackIfConnectionRequest;
// class IBtStackIfConnectionCallback;
class IBtStackIfTelephonyRequest;
// class IBtStackIfTelephonyCallback;
class IBtStackIfPhonebookRequest;
// class IBtStackIfPhonebookCallback;
class IBtStackIfMessagingRequest;
// class IBtStackIfMessagingCallback;
class IBtStackIfMediaPlayerRequest;
// class IBtStackIfMediaPlayerCallback;
class IBtStackIfWblRequest;
// class IBtStackIfWblCallback;
class IBtStackIfEcnrRequest;
// class IBtStackIfEcnrCallback;

/**
 *
 */
class BtStackIfFactory
{
public:
   /**
    * Get instance of instance factory (reference).
    *
    * @return = reference to instance factory object
    */
   static BtStackIfFactory& getInstance(void);

   /**
    * Get instance of connection request if (pointer).
    *
    * @param[in] componentFunctionBlock: function block related to general component functionality
    * @param[in] stackInterface: stack interface to be used (Bluetooth stack)
    *
    * @return = pointer to connection request if object
    */
   IBtStackIfConnectionRequest* getConnectionRequestIf(IN const BTSFunctionBlock componentFunctionBlock = BTS_FB_CONNECTION, IN const BTSInterfaceType stackInterface = BTS_IF_ALPS_EVOLUTION_GENIVI);

   /**
    * Destroy instance of connection request if (pointer).
    *
    * @param[in] requestIf: request interface to be destroyed
    */
   void destroyConnectionRequestIf(IN const IBtStackIfConnectionRequest* requestIf);

   /**
    * Get instance of telephony request if (pointer).
    *
    * @param[in] componentFunctionBlock: function block related to general component functionality
    * @param[in] stackInterface: stack interface to be used (Bluetooth stack)
    *
    * @return = pointer to telephony request if object
    */
   IBtStackIfTelephonyRequest* getTelephonyRequestIf(IN const BTSFunctionBlock componentFunctionBlock = BTS_FB_TELEPHONY, IN const BTSInterfaceType stackInterface = BTS_IF_ALPS_EVOLUTION_GENIVI);

   /**
    * Destroy instance of telephony request if (pointer).
    *
    * @param[in] requestIf: request interface to be destroyed
    */
   void destroyTelephonyRequestIf(IN const IBtStackIfTelephonyRequest* requestIf);

   /**
    * Get instance of phonebook request if (pointer).
    *
    * @param[in] componentFunctionBlock: function block related to general component functionality
    * @param[in] stackInterface: stack interface to be used (Bluetooth stack)
    *
    * @return = pointer to phonebook request if object
    */
   IBtStackIfPhonebookRequest* getPhonebookRequestIf(IN const BTSFunctionBlock componentFunctionBlock = BTS_FB_PHONEBOOK, IN const BTSInterfaceType stackInterface = BTS_IF_ALPS_EVOLUTION_GENIVI);

   /**
    * Destroy instance of phonebook request if (pointer).
    *
    * @param[in] requestIf: request interface to be destroyed
    */
   void destroyPhonebookRequestIf(IN const IBtStackIfPhonebookRequest* requestIf);

   /**
    * Get instance of messaging request if (pointer).
    *
    * @param[in] componentFunctionBlock: function block related to general component functionality
    * @param[in] stackInterface: stack interface to be used (Bluetooth stack)
    *
    * @return = pointer to messaging request if object
    */
   IBtStackIfMessagingRequest* getMessagingRequestIf(IN const BTSFunctionBlock componentFunctionBlock = BTS_FB_MESSAGING, IN const BTSInterfaceType stackInterface = BTS_IF_ALPS_EVOLUTION_GENIVI);

   /**
    * Destroy instance of messaging request if (pointer).
    *
    * @param[in] requestIf: request interface to be destroyed
    */
   void destroyMessagingRequestIf(IN const IBtStackIfMessagingRequest* requestIf);

   /**
    * Get instance of media player request if (pointer).
    *
    * @param[in] componentFunctionBlock: function block related to general component functionality
    * @param[in] stackInterface: stack interface to be used (Bluetooth stack)
    *
    * @return = pointer to media player request if object
    */
   IBtStackIfMediaPlayerRequest* getMediaPlayerRequestIf(IN const BTSFunctionBlock componentFunctionBlock = BTS_FB_MEDIAPLAYER, IN const BTSInterfaceType stackInterface = BTS_IF_ALPS_EVOLUTION_GENIVI);

   /**
    * Destroy instance of media player request if (pointer).
    *
    * @param[in] requestIf: request interface to be destroyed
    */
   void destroyMediaPlayerRequestIf(IN const IBtStackIfMediaPlayerRequest* requestIf);

   /**
    * Get instance of WBL request if (pointer).
    *
    * @param[in] componentFunctionBlock: function block related to general component functionality
    * @param[in] stackInterface: stack interface to be used (Bluetooth stack)
    *
    * @return = pointer to WBL request if object
    */
   IBtStackIfWblRequest* getWblRequestIf(IN const BTSFunctionBlock componentFunctionBlock = BTS_FB_CONNECTION, IN const BTSInterfaceType stackInterface = BTS_IF_ALPS_EVOLUTION_GENIVI);

   /**
    * Destroy instance of WBL request if (pointer).
    *
    * @param[in] requestIf: request interface to be destroyed
    */
   void destroyWblRequestIf(IN const IBtStackIfWblRequest* requestIf);

   /**
    * Get instance of ECNR request if (pointer).
    *
    * @param[in] componentFunctionBlock: function block related to general component functionality
    * @param[in] stackInterface: stack interface to be used (Bluetooth stack)
    *
    * @return = pointer to ECNR request if object
    */
   IBtStackIfEcnrRequest* getEcnrRequestIf(IN const BTSFunctionBlock componentFunctionBlock = BTS_FB_TELEPHONY, IN const BTSInterfaceType stackInterface = BTS_IF_ALPS_EVOLUTION_GENIVI);

   /**
    * Destroy instance of ECNR request if (pointer).
    *
    * @param[in] requestIf: request interface to be destroyed
    */
   void destroyEcnrRequestIf(IN const IBtStackIfEcnrRequest* requestIf);

   /************************ unit test section ****************************************************/
   /**
    * Set instance of connection request if (pointer).
    * For testing purpose (unit test).
    * If instance is set via this interface the caller is responsible to free the instance at the end of test run.
    *
    * @param[in] testInstance: test version of connection request if (pointer)
    * @param[in] stackInterface: stack interface to be used (Bluetooth stack)
    */
   void setConnectionRequestIfInstanceForTesting(IN IBtStackIfConnectionRequest* testInstance, IN const BTSInterfaceType stackInterface = BTS_IF_ALPS_EVOLUTION_GENIVI);

   /**
    * Set instance of telephony request if (pointer).
    * For testing purpose (unit test).
    * If instance is set via this interface the caller is responsible to free the instance at the end of test run.
    *
    * @param[in] testInstance: test version of telephony request if (pointer)
    * @param[in] stackInterface: stack interface to be used (Bluetooth stack)
    */
   void setTelephonyRequestIfInstanceForTesting(IN IBtStackIfTelephonyRequest* testInstance, IN const BTSInterfaceType stackInterface = BTS_IF_ALPS_EVOLUTION_GENIVI);

   /**
    * Set instance of phonebook request if (pointer).
    * For testing purpose (unit test).
    * If instance is set via this interface the caller is responsible to free the instance at the end of test run.
    *
    * @param[in] testInstance: test version of phonebook request if (pointer)
    * @param[in] stackInterface: stack interface to be used (Bluetooth stack)
    */
   void setPhonebookRequestIfInstanceForTesting(IN IBtStackIfPhonebookRequest* testInstance, IN const BTSInterfaceType stackInterface = BTS_IF_ALPS_EVOLUTION_GENIVI);

   /**
    * Set instance of messaging request if (pointer).
    * For testing purpose (unit test).
    * If instance is set via this interface the caller is responsible to free the instance at the end of test run.
    *
    * @param[in] testInstance: test version of messaging request if (pointer)
    * @param[in] stackInterface: stack interface to be used (Bluetooth stack)
    */
   void setMessagingRequestIfInstanceForTesting(IN IBtStackIfMessagingRequest* testInstance, IN const BTSInterfaceType stackInterface = BTS_IF_ALPS_EVOLUTION_GENIVI);

   /**
    * Set instance of media player request if (pointer).
    * For testing purpose (unit test).
    * If instance is set via this interface the caller is responsible to free the instance at the end of test run.
    *
    * @param[in] testInstance: test version of media player request if (pointer)
    * @param[in] stackInterface: stack interface to be used (Bluetooth stack)
    */
   void setMediaPlayerRequestIfInstanceForTesting(IN IBtStackIfMediaPlayerRequest* testInstance, IN const BTSInterfaceType stackInterface = BTS_IF_ALPS_EVOLUTION_GENIVI);

   /**
    * Set instance of WBL request if (pointer).
    * For testing purpose (unit test).
    * If instance is set via this interface the caller is responsible to free the instance at the end of test run.
    *
    * @param[in] testInstance: test version of WBL request if (pointer)
    * @param[in] stackInterface: stack interface to be used (Bluetooth stack)
    */
   void setWblRequestIfInstanceForTesting(IN IBtStackIfWblRequest* testInstance, IN const BTSInterfaceType stackInterface = BTS_IF_ALPS_EVOLUTION_GENIVI);

   /**
    * Set instance of ECNR request if (pointer).
    * For testing purpose (unit test).
    * If instance is set via this interface the caller is responsible to free the instance at the end of test run.
    *
    * @param[in] testInstance: test version of ECNR request if (pointer)
    * @param[in] stackInterface: stack interface to be used (Bluetooth stack)
    */
   void setEcnrRequestIfInstanceForTesting(IN IBtStackIfEcnrRequest* testInstance, IN const BTSInterfaceType stackInterface = BTS_IF_ALPS_EVOLUTION_GENIVI);

   /************************ general configuration section ****************************************/
   /**
    * Sets Bluetooth stack user mode to be used.
    * <BR><B>There is no need to call this interface. A default stack user mode is used if not set from outside.</B>
    * <BR><B>If user calls this method it must be called before constructor is invoked.</B>
    *
    * @param[in] userMode: stack user mode type to be used
    */
   void setStackUserMode(IN BTSUserMode userMode);

   /**
    * Sets ASF log configuration file.
    * <BR><B>There is no need to call this interface. A default ASF log configuration file is used if not set from outside.</B>
    * <BR><B>If user calls this method it must be called before constructor is invoked.</B>
    *
    * @param[in] file: configuration file
    */
   void setAsfLogConfigFile(IN const ::std::string& file);

   /**
    * Print statistics.
    */
   void printStatistics(void);

private:
   /**
    * Default constructor. Private!!!
    */
   BtStackIfFactory();

   /**
    * Destructor. Private!!!
    */
   virtual ~BtStackIfFactory();

   const int _localCreate; /**< value for local instance creation */
   const int _externCreate; /**< value for extern instance creation (unit test) */

   LockForeverAndNonReentrant _lock; /**< lock for accessing lists */

   ::std::map< BTSInterfaceType, IBtStackIfConnectionRequest* > _connectionIfList; /**< connection request if list */
   ::std::map< const IBtStackIfConnectionRequest*, int > _connectionCreatorList; /**< creator flag list for connection request if */

   ::std::map< BTSInterfaceType, IBtStackIfTelephonyRequest* > _telephonyIfList; /**< telephony request if list */
   ::std::map< const IBtStackIfTelephonyRequest*, int > _telephonyCreatorList; /**< creator flag list for telephony request if */

   ::std::map< BTSInterfaceType, IBtStackIfPhonebookRequest* > _phonebookIfList; /**< phonebook request if list */
   ::std::map< const IBtStackIfPhonebookRequest*, int > _phonebookCreatorList; /**< creator flag list for phonebook request if */

   ::std::map< BTSInterfaceType, IBtStackIfMessagingRequest* > _messagingIfList; /**< messaging request if list */
   ::std::map< const IBtStackIfMessagingRequest*, int > _messagingCreatorList; /**< creator flag list for messaging request if */

   ::std::map< BTSInterfaceType, IBtStackIfMediaPlayerRequest* > _mediaPlayerIfList; /**< media player request if list */
   ::std::map< const IBtStackIfMediaPlayerRequest*, int > _mediaPlayerCreatorList; /**< creator flag list for media player request if */

   ::std::map< BTSInterfaceType, IBtStackIfWblRequest* > _wblIfList; /**< wbl request if list */
   ::std::map< const IBtStackIfWblRequest*, int > _wblCreatorList; /**< creator flag list for wbl request if */

   ::std::map< BTSInterfaceType, IBtStackIfEcnrRequest* > _ecnrIfList; /**< ecnr request if list */
   ::std::map< const IBtStackIfEcnrRequest*, int > _ecnrCreatorList; /**< creator flag list for ecnr request if */
};

} //btstackif

#endif //_BT_STACK_IF_FACTORY_H_
