/**
 * @file BtStackIfConnectionRequest.h
 *
 * @par SW-Component
 * Interface
 *
 * @brief Connection request interface.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Class for connection request functionality.
 */

#ifndef _BT_STACK_IF_CONNECTION_REQUEST_H_
#define _BT_STACK_IF_CONNECTION_REQUEST_H_

#include "BtStackIfBaseRequest.h"
#include "IBtStackIfConnectionRequest.h"

namespace btstackif {

/**
 *
 */
class BtStackIfConnectionRequest : public BtStackIfBaseRequest, public IBtStackIfConnectionRequest
{
public:
   /**
    * Constructor.
    *
    * @param[in] componentFunctionBlock: function block related to general component functionality
    * @param[in] stackInterface: stack interface to be used (Bluetooth stack)
    */
   BtStackIfConnectionRequest(IN const BTSFunctionBlock componentFunctionBlock = BTS_FB_CONNECTION, IN const BTSInterfaceType stackInterface = BTS_IF_ALPS_EVOLUTION_GENIVI);

   /**
    * Destructor.
    */
   virtual ~BtStackIfConnectionRequest();

   virtual void registerCallback(IBtStackIfConnectionCallback* callback);

   virtual BTSSessionHandle getSessionHandle(void) const;

   virtual BTSErrorCode init(IN const BTSLocalStackConfiguration& configuration, IN const BTSBDAddressList& pairedDevices) const;

   virtual BTSErrorCode deInit(void) const;

   virtual void switchBluetoothOnOff(IN const BTSBluetoothMode bluetoothMode, IN const BTSAdapterMode adapterMode);

   virtual void getBluetoothOnOffStatus(void);

   virtual void switchBluetoothHardwareOnOff(IN const BTSBluetoothMode bluetoothMode);

   virtual void setLocalBtName(IN const BTSDeviceName& adapterName);

   virtual void getLocalBtName(void);

   virtual void setDeviceIdServiceRecord(IN const BTSDeviceIdServiceRecord& serviceRecord);

   virtual void setDeviceForRestrictedPairingConnecting(IN const BTSBDAddress& expectedDeviceAddress, IN const BTSSessionHandle sessionHandle = 0);

   virtual void setLocalAdapterModes(IN const BTSLocalMode discoverableMode, IN const BTSDiscoverableTimeout discoverableTimeout, IN const BTSLocalMode connectableMode, IN const BTSConnectableTimeout connectableTimeout, IN const BTSSessionHandle sessionHandle = 0);

   virtual void getLocalAdapterModes(IN const BTSSessionHandle sessionHandle = 0);

   virtual void startDiscovery(IN const BTSSessionHandle sessionHandle = 0);

   virtual void stopDiscovery(IN const BTSSessionHandle sessionHandle = 0);

   virtual void getDiscoveringStatus(IN const BTSSessionHandle sessionHandle = 0);

   virtual void startPairing(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle = 0);

   virtual void cancelPairing(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle = 0);

   virtual void setPinCode(IN const BTSBDAddress& deviceAddress, IN const BTSPinCode& pinCode, IN const BTSSessionHandle sessionHandle = 0);

   virtual void confirmSecureSimplePairing(IN const BTSBDAddress& deviceAddress, IN const BTSSspMode sspMode, IN const BTSNumericValue& numericValue, IN const BTSConfirmationMode confirmationMode, IN const BTSSessionHandle sessionHandle = 0);

   virtual void setLinkKey(IN const BTSBDAddress& deviceAddress, IN const BTSLinkKeyType linkKeyType, IN const BTSLinkKey& linkKey, IN const BTSDLinkKey& dLinkKey, IN const BTSSessionHandle sessionHandle = 0);

   virtual void connectProtocol(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const BTSTimeValue pageTimeout = 10240, IN const BTSSessionHandle sessionHandle = 0);

   virtual void acceptRemoteProtocolConnect(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSSessionHandle sessionHandle = 0);

   virtual void rejectRemoteProtocolConnect(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSSessionHandle sessionHandle = 0);

   virtual void disconnectProtocol(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const bool pauseBtStreaming = true, IN const BTSSessionHandle sessionHandle = 0);

   virtual void disconnectDevice(IN const BTSBDAddress& deviceAddress, IN const bool pauseBtStreaming = true, IN const BTSSessionHandle sessionHandle = 0);

   virtual void getProtocolConnectionStatus(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const BTSSessionHandle sessionHandle = 0);

   virtual void getDeviceConnectionStatus(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle = 0);

   virtual void startRemoteServiceSearch(IN const BTSBDAddress& deviceAddress, IN const BTSSearchType searchType, IN const bool rejectIfNotConnected = true, IN const BTSSessionHandle sessionHandle = 0);

   virtual void cancelRemoteServiceSearch(IN const BTSBDAddress& deviceAddress, IN const BTSSearchType searchType, IN const BTSSessionHandle sessionHandle = 0);

   virtual void queryRemoteName(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle = 0);

   virtual void requestLinkQuality(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle = 0);

   virtual void startTestMode(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle = 0);

   virtual void stopTestMode(IN const BTSSessionHandle sessionHandle = 0);

   virtual void requestTestModeLinkQuality(IN const BTSSessionHandle sessionHandle = 0);

   virtual void setSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData) const;

   virtual void setSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData) const;

   virtual void setSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData) const;

   virtual void setMainControlTestCommand(IN const char* testCommand, IN const unsigned int testData) const;

   virtual void setConnectionControlTestCommand(IN const char* testCommand, IN const unsigned int testData) const;

   virtual void setConnectionControlTestCommand(IN const char* testCommand, IN const unsigned char* testData) const;

private:
   /**
    * Default constructor.
    */
   BtStackIfConnectionRequest();
};

} //btstackif

#endif //_BT_STACK_IF_CONNECTION_REQUEST_H_
