/**
 * @file BtStackIfBaseRequest.h
 *
 * @par SW-Component
 * Interface
 *
 * @brief Basic request interface.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Class for basic request functionality.
 */

#ifndef _BT_STACK_IF_BASE_REQUEST_H_
#define _BT_STACK_IF_BASE_REQUEST_H_

#include "BtStackIfCallback.h"
#include "BtStackIfTypes.h"
#include "FwLock.h"

namespace btstackif {

// class forward declarations
class App2Bts_BaseMessage;
class Ipc2Bts_BaseMessage;
class IGenericStackIf;

/**
 *
 */
class BtStackIfBaseRequest
{
public:
   /**
    * Sets Bluetooth stack user mode to be used.
    * <BR><B>There is no need to call this interface. A default stack user mode is used if not set from outside.</B>
    * <BR><B>If user calls this method it must be called before constructor is invoked.</B>
    *
    * @param[in] userMode: stack user mode type to be used
    */
   static void setStackUserMode(IN const BTSUserMode userMode);

   /**
    * Sets ASF log configuration file.
    * <BR><B>There is no need to call this interface. A default ASF log configuration file is used if not set from outside.</B>
    * <BR><B>If user calls this method it must be called before constructor is invoked.</B>
    *
    * @param[in] file: configuration file
    */
   static void setAsfLogConfigFile(IN const ::std::string& file);

   /**
    * Print statistics.
    */
   static void printStatistics(void);

protected:
   /**
    * Constructor.
    *
    * @param[in] userMode: stack user mode type to be used
    * @param[in] componentFunctionBlock: function block related to general component functionality
    * @param[in] subComponentFunctionBlock: function block related to specific sub component functionality
    * @param[in] stackInterface: stack interface to be used (Bluetooth stack)
    */
   BtStackIfBaseRequest(IN const BTSUserMode userMode, IN const BTSFunctionBlock componentFunctionBlock, IN const BTSFunctionBlock subComponentFunctionBlock, IN const BTSInterfaceType stackInterface);

   /**
    * Destructor.
    */
   virtual ~BtStackIfBaseRequest();

   /**
    * Get session handle.
    *
    * @return = session handle
    */
   BTSSessionHandle getHandle(void) const;

   /**
    * Register callback.
    */
   void registerInternalCallback(void);

   /**
    * Init the interface library.
    *
    * @param[in] configuration: local configuration data
    *
    * @return != BTS_OK: error,
    * @return  = BTS_OK: OK
    */
   BTSErrorCode processInit(IN const BTSLocalConfigurationContainer& configuration) const;

   /**
    * DeInit the interface library.
    *
    * @return != BTS_OK: error,
    * @return  = BTS_OK: OK
    */
   BTSErrorCode processDeInit(void) const;

   /**
    * Process App2Bts message.
    *
    * @param[in] message: App2Bts message
    */
   void processApp2BtsMsg(IN App2Bts_BaseMessage* message);

   /**
    * Sets a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   void processSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData) const;

   /**
    * Sets a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   void processSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData) const;

   /**
    * Sets a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   void processSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData) const;

   /**
    * Sets a main control test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   void processMainControlTestCommand(IN const char* testCommand, IN const unsigned int testData) const;

   /**
    * Sets a connection control test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   void processConnectionControlTestCommand(IN const char* testCommand, IN const unsigned int testData) const;

   /**
    * Sets a connection control test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   void processConnectionControlTestCommand(IN const char* testCommand, IN const unsigned char* testData) const;

   BtStackIfCallback _callback; /**< callback handler */
   BTSFunctionBlock _componentFunctionBlock; /**< component function block */
   BTSFunctionBlock _subComponentFunctionBlock; /**< sub-component function block => part of message opcode */
   BTSInterfaceType _stackInterface; /**< stack interface to be used (Bluetooth stack) */

   static bool _stackUserModeSet; /**< flag indicating that stack user mode was set from outside (unit test) */
   static BTSUserMode _stackUserMode; /**< stack user mode */
   static ::std::string _asfConfigFile; /**< ASF configuration file */
   static IGenericStackIf* _genericStackIf; /**< internal generic stack interface */

private:
   /**
    * Default constructor.
    */
   BtStackIfBaseRequest();

   /**
    * Get user mode.
    *
    * @param[in] currentMode: current user mode
    *
    * @return = user mode to be used
    */
   BTSUserMode getUserMode(IN const BTSUserMode currentMode) const;

   static LockForeverAndNonReentrant _sessionHandleLock; /**< lock for session handle creation */
   static BTSSessionHandle _sessionHandleCounter; /**< session handle counter */
   static const ::std::string _bluetoothServiceOutFile; /**< Bluetooth service binary file name */
   static const ::std::string _disableBtStackCommFile; /**< name of file to disable BT stack communication */
};

} //btstackif

#endif //_BT_STACK_IF_BASE_REQUEST_H_
