/**
 * @file InstanceFactory.cpp
 *
 * @par SW-Component
 * Factory
 *
 * @brief Instance factory.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Instance factory class.
 * Major task of this class is to provide the possibility to set test interfaces (unit test).
 */

#include "InstanceFactory.h"
#include "GenericStackIf.h"
#include "MainControl.h"
#include "DbusIncludes.h"
#include "ControlIncludes.h"
#include "FwAssert.h"

namespace btstackif {

InstanceFactory::InstanceFactory() :
_genericStackIfPtr(0),
_genericStackIfCreator(BTS_LOCAL_CREATE),
_mainControlPtr(0),
_mainControlCreator(BTS_LOCAL_CREATE),
_dbusAccessPtrList(),
_dbusAccessCreatorList(),
_configurationBasicControlPtrList(),
_configurationBasicControlCreatorList(),
_connectionBasicControlPtrList(),
_connectionBasicControlCreatorList(),
_telephonyBasicControlPtrList(),
_telephonyBasicControlCreatorList(),
_phonebookBasicControlPtrList(),
_phonebookBasicControlCreatorList(),
_messagingBasicControlPtrList(),
_messagingBasicControlCreatorList(),
_mediaPlayerBasicControlPtrList(),
_mediaPlayerBasicControlCreatorList(),
_wblBasicControlPtrList(),
_wblBasicControlCreatorList()
{
}

InstanceFactory::~InstanceFactory()
{
   _genericStackIfPtr = 0;
   _mainControlPtr = 0;
}

InstanceFactory& InstanceFactory::getInstance(void)
{
   static InstanceFactory factory;
   return factory;
}

IGenericStackIf* InstanceFactory::getGenericStackIfInstance(IN const BTSUserMode userMode, IN const ::std::string& file)
{
   if(0 != _genericStackIfPtr)
   {
      // already created
   }
   else
   {
      _genericStackIfPtr = new GenericStackIf(userMode, file);
      _genericStackIfCreator = BTS_LOCAL_CREATE;
   }

   return _genericStackIfPtr;
}

void InstanceFactory::destroyGenericStackIfInstance(void)
{
   if((0 != _genericStackIfPtr) && (BTS_LOCAL_CREATE == _genericStackIfCreator))
   {
      delete _genericStackIfPtr;
      _genericStackIfPtr = 0;
   }
}

void InstanceFactory::setGenericStackIfInstanceForTesting(IGenericStackIf* testInstance)
{
   if(0 != _genericStackIfPtr)
   {
      // not allowed because already set => to be considered by unit test
   }
   else
   {
      _genericStackIfPtr = testInstance;
      _genericStackIfCreator = BTS_EXTERN_CREATE;
   }
}

IMainControl* InstanceFactory::getMainControlInstance(void)
{
   if(0 != _mainControlPtr)
   {
      // already created
   }
   else
   {
      _mainControlPtr = new MainControl();
      _mainControlCreator = BTS_LOCAL_CREATE;
   }

   return _mainControlPtr;
}

void InstanceFactory::destroyMainControlInstance(void)
{
   if((0 != _mainControlPtr) && (BTS_LOCAL_CREATE == _mainControlCreator))
   {
      delete _mainControlPtr;
      _mainControlPtr = 0;
   }
}

void InstanceFactory::setMainControlInstanceForTesting(IMainControl* testInstance)
{
   if(0 != _mainControlPtr)
   {
      // not allowed because already set => to be considered by unit test
   }
   else
   {
      _mainControlPtr = testInstance;
      _mainControlCreator = BTS_EXTERN_CREATE;
   }
}

IDbusBase* InstanceFactory::getDbusAccessInstance(IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent)
{
   IDbusBase* returnIf(0);

   if(false == isValidInterfaceType(stackInterface))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return returnIf;
   }

   if(false == isValidComponentType(subComponent))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return returnIf;
   }

   const BTSFunctionBlock component(BTS_FB_NONE);
   BTSComponentEntry entry;
   entry = getDbusComponentKey(entry, component, stackInterface, subComponent);
   ::std::map< BTSComponentEntry, IDbusBase* >::iterator it = _dbusAccessPtrList.find(entry);
   if(_dbusAccessPtrList.end() != it)
   {
      returnIf = it->second;
   }
   else
   {
      switch(subComponent)
      {
         case BTS_FB_CONFIG:
         case BTS_FB_CONNECTION:
         case BTS_FB_TELEPHONY:
         case BTS_FB_PHONEBOOK:
         case BTS_FB_MESSAGING:
         case BTS_FB_MEDIAPLAYER:
         {
            switch(stackInterface)
            {
               case BTS_IF_SIMULATION:
#ifdef VARIANT_S_FTR_ENABLE_CONN_SIM
                  returnIf = new simulation::DbusAccess();
#endif
                  break;
               case BTS_IF_ALPS_EVOLUTION_GENIVI:
                  returnIf = new genivi::DbusAccess();
                  break;
               case BTS_IF_UNDEFINED:
               case BTS_IF_LAST:
               default:
                  break;
            }
            break;
         }
         case BTS_FB_WBL:
         case BTS_FB_ECNR:
         {
            if(BTS_FB_WBL == subComponent)
            {
               returnIf = new wbl::WblDbusAccessMain();
            }
            else
            {
               // implement if needed --- returnIf = new ECNR
               FW_NORMAL_ASSERT_ALWAYS();
            }
            break;
         }
         case BTS_FB_NONE:
         case BTS_FB_LAST:
         default:
            break;
      }

      if(0 != returnIf)
      {
         _dbusAccessPtrList[entry] = returnIf;
         _dbusAccessCreatorList[returnIf] = BTS_LOCAL_CREATE;
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }

   return returnIf;
}

void InstanceFactory::destroyDbusAccessInstance(IN const IDbusBase* dbusIf)
{
   FW_IF_NULL_PTR_RETURN(dbusIf);

   for(::std::map< BTSComponentEntry, IDbusBase* >::iterator itIf = _dbusAccessPtrList.begin(); itIf != _dbusAccessPtrList.end(); ++itIf)
   {
      if(dbusIf == itIf->second)
      {
         BTSFactoryCreator creator(BTS_LOCAL_CREATE);
         ::std::map< const IDbusBase*, BTSFactoryCreator >::iterator itCreator = _dbusAccessCreatorList.find(dbusIf);
         if(_dbusAccessCreatorList.end() != itCreator)
         {
            creator = itCreator->second;
            _dbusAccessCreatorList.erase(itCreator);
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }

         if(BTS_LOCAL_CREATE == creator)
         {
            delete dbusIf;
         }

         _dbusAccessPtrList.erase(itIf);

         break;
      }
   }
}

void InstanceFactory::setDbusAccessInstanceForTesting(IDbusBase* testInstance, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent)
{
   FW_IF_NULL_PTR_RETURN(testInstance);

   if(false == isValidInterfaceType(stackInterface))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(false == isValidComponentType(subComponent))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   const BTSFunctionBlock component(BTS_FB_NONE);
   BTSComponentEntry entry;
   entry = getDbusComponentKey(entry, component, stackInterface, subComponent);
   ::std::map< BTSComponentEntry, IDbusBase* >::iterator it = _dbusAccessPtrList.find(entry);
   if(_dbusAccessPtrList.end() != it)
   {
      // not allowed because already set => to be considered by unit test
   }
   else
   {
      _dbusAccessPtrList[entry] = testInstance;
      _dbusAccessCreatorList[testInstance] = BTS_EXTERN_CREATE;
   }
}

IBasicControl* InstanceFactory::getBasicControlInstance(IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent)
{
   IBasicControl* returnIf(0);

   if(false == isValidInterfaceType(stackInterface))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return returnIf;
   }

   if(false == isValidComponentType(subComponent))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return returnIf;
   }

   const BTSFunctionBlock component(BTS_FB_NONE);
   BTSComponentEntry entry;
   entry = getControlComponentKey(entry, component, stackInterface, subComponent);

   switch(subComponent)
   {
      case BTS_FB_CONFIG:
      {
         ::std::map< BTSComponentEntry, IBasicControl* >::iterator it = _configurationBasicControlPtrList.find(entry);
         if(_configurationBasicControlPtrList.end() != it)
         {
            returnIf = it->second;
         }
         else
         {
            switch(stackInterface)
            {
               case BTS_IF_SIMULATION:
#ifdef VARIANT_S_FTR_ENABLE_CONN_SIM
                  returnIf = new simulation::ConfigurationControl();
#endif
                  break;
               case BTS_IF_ALPS_EVOLUTION_GENIVI:
                  returnIf = new genivi::ConfigurationControl();
                  break;
               case BTS_IF_UNDEFINED:
               case BTS_IF_LAST:
               default:
                  break;
            }

            if(0 != returnIf)
            {
               _configurationBasicControlPtrList[entry] = returnIf;
               _configurationBasicControlCreatorList[returnIf] = BTS_LOCAL_CREATE;
            }
            else
            {
               FW_NORMAL_ASSERT_ALWAYS();
            }
         }
         break;
      }
      case BTS_FB_CONNECTION:
      {
         ::std::map< BTSComponentEntry, IBasicControl* >::iterator it = _connectionBasicControlPtrList.find(entry);
         if(_connectionBasicControlPtrList.end() != it)
         {
            returnIf = it->second;
         }
         else
         {
            switch(stackInterface)
            {
               case BTS_IF_SIMULATION:
#ifdef VARIANT_S_FTR_ENABLE_CONN_SIM
                  returnIf = new simulation::ConnectionControl();
#endif
                  break;
               case BTS_IF_ALPS_EVOLUTION_GENIVI:
                  returnIf = new genivi::ConnectionControl();
                  break;
               case BTS_IF_UNDEFINED:
               case BTS_IF_LAST:
               default:
                  break;
            }

            if(0 != returnIf)
            {
               _connectionBasicControlPtrList[entry] = returnIf;
               _connectionBasicControlCreatorList[returnIf] = BTS_LOCAL_CREATE;
            }
            else
            {
               FW_NORMAL_ASSERT_ALWAYS();
            }
         }
         break;
      }
      case BTS_FB_TELEPHONY:
      {
         ::std::map< BTSComponentEntry, IBasicControl* >::iterator it = _telephonyBasicControlPtrList.find(entry);
         if(_telephonyBasicControlPtrList.end() != it)
         {
            returnIf = it->second;
         }
         else
         {
            switch(stackInterface)
            {
               case BTS_IF_SIMULATION:
#ifdef VARIANT_S_FTR_ENABLE_CONN_SIM
                  returnIf = new simulation::TelephonyControl();
#endif
                  break;
               case BTS_IF_ALPS_EVOLUTION_GENIVI:
                  returnIf = new genivi::TelephonyControl();
                  break;
               case BTS_IF_UNDEFINED:
               case BTS_IF_LAST:
               default:
                  break;
            }

            if(0 != returnIf)
            {
               _telephonyBasicControlPtrList[entry] = returnIf;
               _telephonyBasicControlCreatorList[returnIf] = BTS_LOCAL_CREATE;
            }
            else
            {
               FW_NORMAL_ASSERT_ALWAYS();
            }
         }
         break;
      }
      case BTS_FB_PHONEBOOK:
      {
         ::std::map< BTSComponentEntry, IBasicControl* >::iterator it = _phonebookBasicControlPtrList.find(entry);
         if(_phonebookBasicControlPtrList.end() != it)
         {
            returnIf = it->second;
         }
         else
         {
            switch(stackInterface)
            {
               case BTS_IF_SIMULATION:
#ifdef VARIANT_S_FTR_ENABLE_CONN_SIM
                  returnIf = new simulation::PhonebookControl();
#endif
                  break;
               case BTS_IF_ALPS_EVOLUTION_GENIVI:
                  returnIf = new genivi::PhonebookControl();
                  break;
               case BTS_IF_UNDEFINED:
               case BTS_IF_LAST:
               default:
                  break;
            }

            if(0 != returnIf)
            {
               _phonebookBasicControlPtrList[entry] = returnIf;
               _phonebookBasicControlCreatorList[returnIf] = BTS_LOCAL_CREATE;
            }
            else
            {
               FW_NORMAL_ASSERT_ALWAYS();
            }
         }
         break;
      }
      case BTS_FB_MESSAGING:
      {
         ::std::map< BTSComponentEntry, IBasicControl* >::iterator it = _messagingBasicControlPtrList.find(entry);
         if(_messagingBasicControlPtrList.end() != it)
         {
            returnIf = it->second;
         }
         else
         {
            switch(stackInterface)
            {
               case BTS_IF_SIMULATION:
#ifdef VARIANT_S_FTR_ENABLE_CONN_SIM
                  returnIf = new simulation::MessagingControl();
#endif
                  break;
               case BTS_IF_ALPS_EVOLUTION_GENIVI:
                  returnIf = new genivi::MessagingControl();
                  break;
               case BTS_IF_UNDEFINED:
               case BTS_IF_LAST:
               default:
                  break;
            }

            if(0 != returnIf)
            {
               _messagingBasicControlPtrList[entry] = returnIf;
               _messagingBasicControlCreatorList[returnIf] = BTS_LOCAL_CREATE;
            }
            else
            {
               FW_NORMAL_ASSERT_ALWAYS();
            }
         }
         break;
      }
      case BTS_FB_MEDIAPLAYER:
      {
         ::std::map< BTSComponentEntry, IBasicControl* >::iterator it = _mediaPlayerBasicControlPtrList.find(entry);
         if(_mediaPlayerBasicControlPtrList.end() != it)
         {
            returnIf = it->second;
         }
         else
         {
            switch(stackInterface)
            {
               case BTS_IF_SIMULATION:
#ifdef VARIANT_S_FTR_ENABLE_CONN_SIM
                  returnIf = new simulation::MediaPlayerControl();
#endif
                  break;
               case BTS_IF_ALPS_EVOLUTION_GENIVI:
                  returnIf = new genivi::MediaPlayerControl();
                  break;
               case BTS_IF_UNDEFINED:
               case BTS_IF_LAST:
               default:
                  break;
            }

            if(0 != returnIf)
            {
               _mediaPlayerBasicControlPtrList[entry] = returnIf;
               _mediaPlayerBasicControlCreatorList[returnIf] = BTS_LOCAL_CREATE;
            }
            else
            {
               FW_NORMAL_ASSERT_ALWAYS();
            }
         }
         break;
      }
      case BTS_FB_WBL:
      {
         ::std::map< BTSComponentEntry, IBasicControl* >::iterator it = _wblBasicControlPtrList.find(entry);
         if(_wblBasicControlPtrList.end() != it)
         {
            returnIf = it->second;
         }
         else
         {
            returnIf = new wbl::WblControlMain();

            if(0 != returnIf)
            {
               _wblBasicControlPtrList[entry] = returnIf;
               _wblBasicControlCreatorList[returnIf] = BTS_LOCAL_CREATE;
            }
            else
            {
               FW_NORMAL_ASSERT_ALWAYS();
            }
         }
         break;
      }
      case BTS_FB_ECNR:
      {
         // implement if needed
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      }
      case BTS_FB_NONE:
      case BTS_FB_LAST:
      default:
      {
         break;
      }
   }

   return returnIf;
}

void InstanceFactory::destroyBasicControlInstance(IN const IBasicControl* controlHandler)
{
   FW_IF_NULL_PTR_RETURN(controlHandler);

   if(true == destroyControlInstance(_configurationBasicControlPtrList, _configurationBasicControlCreatorList, controlHandler)) { return; }
   if(true == destroyControlInstance(_connectionBasicControlPtrList, _connectionBasicControlCreatorList, controlHandler)) { return; }
   if(true == destroyControlInstance(_telephonyBasicControlPtrList, _telephonyBasicControlCreatorList, controlHandler)) { return; }
   if(true == destroyControlInstance(_phonebookBasicControlPtrList, _phonebookBasicControlCreatorList, controlHandler)) { return; }
   if(true == destroyControlInstance(_messagingBasicControlPtrList, _messagingBasicControlCreatorList, controlHandler)) { return; }
   if(true == destroyControlInstance(_mediaPlayerBasicControlPtrList, _mediaPlayerBasicControlCreatorList, controlHandler)) { return; }
   if(true == destroyControlInstance(_wblBasicControlPtrList, _wblBasicControlCreatorList, controlHandler)) { return; }
}

void InstanceFactory::setBasicControlInstanceForTesting(IBasicControl* testInstance, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent)
{
   FW_IF_NULL_PTR_RETURN(testInstance);

   switch(subComponent)
   {
      case BTS_FB_CONFIG:
         setControlInstanceForTesting(stackInterface, subComponent, _configurationBasicControlPtrList, _configurationBasicControlCreatorList, testInstance);
         break;
      case BTS_FB_CONNECTION:
         setControlInstanceForTesting(stackInterface, subComponent, _connectionBasicControlPtrList, _connectionBasicControlCreatorList, testInstance);
         break;
      case BTS_FB_TELEPHONY:
         setControlInstanceForTesting(stackInterface, subComponent, _telephonyBasicControlPtrList, _telephonyBasicControlCreatorList, testInstance);
         break;
      case BTS_FB_PHONEBOOK:
         setControlInstanceForTesting(stackInterface, subComponent, _phonebookBasicControlPtrList, _phonebookBasicControlCreatorList, testInstance);
         break;
      case BTS_FB_MESSAGING:
         setControlInstanceForTesting(stackInterface, subComponent, _messagingBasicControlPtrList, _messagingBasicControlCreatorList, testInstance);
         break;
      case BTS_FB_MEDIAPLAYER:
         setControlInstanceForTesting(stackInterface, subComponent, _mediaPlayerBasicControlPtrList, _mediaPlayerBasicControlCreatorList, testInstance);
         break;
      case BTS_FB_WBL:
         setControlInstanceForTesting(stackInterface, subComponent, _wblBasicControlPtrList, _wblBasicControlCreatorList, testInstance);
         break;
      case BTS_FB_ECNR:
         // implement if needed
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_FB_NONE:
      case BTS_FB_LAST:
      default:
      {
         break;
      }
   }
}

bool InstanceFactory::destroyControlInstance(::std::map< BTSComponentEntry, IBasicControl* >& controlList, ::std::map< const IBasicControl*, BTSFactoryCreator >& creatorList, IN const IBasicControl* controlHandler) const
{
   FW_IF_NULL_PTR_RETURN_FALSE(controlHandler);

   for(::std::map< BTSComponentEntry, IBasicControl* >::iterator itIf = controlList.begin(); itIf != controlList.end(); ++itIf)
   {
      if(controlHandler == itIf->second)
      {
         BTSFactoryCreator creator(BTS_LOCAL_CREATE);
         ::std::map< const IBasicControl*, BTSFactoryCreator >::iterator itCreator = creatorList.find(controlHandler);
         if(creatorList.end() != itCreator)
         {
            creator = itCreator->second;
            creatorList.erase(itCreator);
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }

         if(BTS_LOCAL_CREATE == creator)
         {
            delete controlHandler;
         }

         controlList.erase(itIf);

         return true;
      }
   }

   return false;
}

void InstanceFactory::setControlInstanceForTesting(IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, ::std::map< BTSComponentEntry, IBasicControl* >& controlList, ::std::map< const IBasicControl*, BTSFactoryCreator >& creatorList, IBasicControl* testInstance) const
{
   FW_IF_NULL_PTR_RETURN(testInstance);

   if(false == isValidInterfaceType(stackInterface))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(false == isValidComponentType(subComponent))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   const BTSFunctionBlock component(BTS_FB_NONE);
   BTSComponentEntry entry;
   entry = getControlComponentKey(entry, component, stackInterface, subComponent);

   ::std::map< BTSComponentEntry, IBasicControl* >::iterator it = controlList.find(entry);
   if(controlList.end() != it)
   {
      // not allowed because already set => to be considered by unit test
   }
   else
   {
      controlList[entry] = testInstance;
      creatorList[testInstance] = BTS_EXTERN_CREATE;
   }
}

} //btstackif
