/**
 * @file InstanceFactory.h
 *
 * @par SW-Component
 * Factory
 *
 * @brief Instance factory.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Instance factory class.
 * Major task of this class is to provide the possibility to set test interfaces (unit test).
 */

#ifndef _INSTANCE_FACTORY_H_
#define _INSTANCE_FACTORY_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IGenericStackIf;
class IMainControl;
class IDbusBase;
class IBasicControl;

/**
 *
 */
class InstanceFactory
{
public:
   /**
    * Get instance of instance factory (reference).
    *
    * @return = reference to instance factory object
    */
   static InstanceFactory& getInstance(void);

   /**
    * Get instance of generic stack if (pointer).
    *
    * @param[in] userMode: stack user mode to be used
    * @param[in] file: configuration file
    *
    * @return = pointer to generic stack if object
    */
   IGenericStackIf* getGenericStackIfInstance(IN const BTSUserMode userMode, IN const ::std::string& file);

   /**
    * Destroy instance of generic stack if (pointer).
    */
   void destroyGenericStackIfInstance(void);

   /**
    * Set instance of generic stack if (pointer).
    * For testing purpose (unit test).
    * If Instance is set from via this interface the caller is responsible to free the instance at the end of test run.
    *
    * @param[in] testInstance: test version of generic stack if (pointer)
    */
   void setGenericStackIfInstanceForTesting(IGenericStackIf* testInstance);

   /**
    * Get instance of main control (pointer).
    *
    * @return = pointer to main control object
    */
   IMainControl* getMainControlInstance(void);

   /**
    * Destroy instance of main control (pointer).
    */
   void destroyMainControlInstance(void);

   /**
    * Set instance of main control (pointer).
    * For testing purpose (unit test).
    * If Instance is set from via this interface the caller is responsible to free the instance at the end of test run.
    *
    * @param[in] testInstance: test version of main control (pointer)
    */
   void setMainControlInstanceForTesting(IMainControl* testInstance);

   /**
    * Get instance of DBUS access (pointer).
    *
    * @param[in] stackInterface: stack interface type to be used
    * @param[in] subComponent: sub component
    *
    * @return = pointer to DBUS access object
    */
   IDbusBase* getDbusAccessInstance(IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent);

   /**
    * Destroy instance of DBUS access (pointer).
    *
    * @param[in] dbusIf: dbus instance to be destroyed
    */
   void destroyDbusAccessInstance(IN const IDbusBase* dbusIf);

   /**
    * Set instance of DBUS access (pointer).
    * For testing purpose (unit test).
    * If Instance is set from via this interface the caller is responsible to free the instance at the end of test run.
    *
    * @param[in] testInstance: test version of DBUS access (pointer)
    * @param[in] stackInterface: stack interface type to be used
    * @param[in] subComponent: sub component
    */
   void setDbusAccessInstanceForTesting(IN IDbusBase* testInstance, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent);

   /**
    * Get instance of basic control (pointer).
    *
    * @param[in] stackInterface: stack interface type to be used
    * @param[in] subComponent: sub component
    *
    * @return = pointer to basic control object
    */
   IBasicControl* getBasicControlInstance(IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent);

   /**
    * Destroy instance of basic control (pointer).
    *
    * @param[in] controlHandler: control handler
    */
   void destroyBasicControlInstance(IN const IBasicControl* controlHandler);

   /**
    * Set instance of basic control (pointer).
    * For testing purpose (unit test).
    * If Instance is set from via this interface the caller is responsible to free the instance at the end of test run.
    *
    * @param[in] testInstance: test version of basic control (pointer)
    * @param[in] stackInterface: stack interface type to be used
    * @param[in] subComponent: sub component
    */
   void setBasicControlInstanceForTesting(IBasicControl* testInstance, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent);

private:
   /**
    * Default constructor. Private!!!
    */
   InstanceFactory();

   /**
    * Destructor. Private!!!
    */
   virtual ~InstanceFactory();

   IGenericStackIf* _genericStackIfPtr;
   BTSFactoryCreator _genericStackIfCreator;

   IMainControl* _mainControlPtr;
   BTSFactoryCreator _mainControlCreator;

   ::std::map< BTSComponentEntry, IDbusBase* > _dbusAccessPtrList;
   ::std::map< const IDbusBase*, BTSFactoryCreator > _dbusAccessCreatorList;

   ::std::map< BTSComponentEntry, IBasicControl* > _configurationBasicControlPtrList;
   ::std::map< const IBasicControl*, BTSFactoryCreator > _configurationBasicControlCreatorList;
   ::std::map< BTSComponentEntry, IBasicControl* > _connectionBasicControlPtrList;
   ::std::map< const IBasicControl*, BTSFactoryCreator > _connectionBasicControlCreatorList;
   ::std::map< BTSComponentEntry, IBasicControl* > _telephonyBasicControlPtrList;
   ::std::map< const IBasicControl*, BTSFactoryCreator > _telephonyBasicControlCreatorList;
   ::std::map< BTSComponentEntry, IBasicControl* > _phonebookBasicControlPtrList;
   ::std::map< const IBasicControl*, BTSFactoryCreator > _phonebookBasicControlCreatorList;
   ::std::map< BTSComponentEntry, IBasicControl* > _messagingBasicControlPtrList;
   ::std::map< const IBasicControl*, BTSFactoryCreator > _messagingBasicControlCreatorList;
   ::std::map< BTSComponentEntry, IBasicControl* > _mediaPlayerBasicControlPtrList;
   ::std::map< const IBasicControl*, BTSFactoryCreator > _mediaPlayerBasicControlCreatorList;
   ::std::map< BTSComponentEntry, IBasicControl* > _wblBasicControlPtrList;
   ::std::map< const IBasicControl*, BTSFactoryCreator > _wblBasicControlCreatorList;

   bool destroyControlInstance(::std::map< BTSComponentEntry, IBasicControl* >& controlList, ::std::map< const IBasicControl*, BTSFactoryCreator >& creatorList, IN const IBasicControl* controlHandler) const;
   void setControlInstanceForTesting(IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, ::std::map< BTSComponentEntry, IBasicControl* >& controlList, ::std::map< const IBasicControl*, BTSFactoryCreator >& creatorList, IBasicControl* testInstance) const;
};

} //btstackif

#endif //_INSTANCE_FACTORY_H_
