/**
 * @file DbusServiceAvailability.cpp
 *
 * @par SW-Component
 * IPC
 *
 * @brief DBUS service availability handling.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details DBUS service availability handling.
 */

#include "DbusServiceAvailability.h"
#include "IBasicControl.h"
#include "BtsTimer.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_DBUS
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/DbusServiceAvailability.cpp.trc.h"
#else
#include "BtStackIfTypesTrace.h"
#include "BtStackInternalTypesTrace.h"
#endif
#endif

namespace btstackif {

BTSTimeValue DbusServiceInfo::_dbusServiceTimeout = 10000;

DbusServiceEntry::DbusServiceEntry()
{
   // busName
   // objPath
   busType = ::ccdbusif::BUS_TYPE_SYSTEM;
}

DbusServiceEntry::DbusServiceEntry(IN const DbusServiceEntry& ref)
: BTSProtocolBaseEntry(ref)
{
   busName = ref.busName;
   objPath = ref.objPath;
   busType = ref.busType;
}

DbusServiceEntry& DbusServiceEntry::operator=(IN const DbusServiceEntry& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   BTSProtocolBaseEntry::operator=(ref);

   busName = ref.busName;
   objPath = ref.objPath;
   busType = ref.busType;

   return *this;
}

bool DbusServiceEntry::operator==(IN const DbusServiceEntry& ref) const
{
   return (0 == entryCompare(ref));
}

bool DbusServiceEntry::operator!=(IN const DbusServiceEntry& ref) const
{
   return (0 != entryCompare(ref));
}

bool DbusServiceEntry::operator<(const DbusServiceEntry& ref) const
{
   return (0 > entryCompare(ref));
}

bool DbusServiceEntry::operator>(const DbusServiceEntry& ref) const
{
   return (0 < entryCompare(ref));
}

DbusServiceEntry::~DbusServiceEntry()
{
}

int DbusServiceEntry::entryCompare(const DbusServiceEntry& ref) const
{
   int result;

   result = compare(ref);
   if(result < 0)
   {
      return -1;
   }
   else if(result > 0)
   {
      return 1;
   }

   // BTSProtocolBaseEntry is equal

   result = busName.compare(ref.busName);
   if(result < 0)
   {
      return -1;
   }
   else if(result > 0)
   {
      return 1;
   }

   // bus name is equal

   result = objPath.compare(ref.objPath);
   if(result < 0)
   {
      return -1;
   }
   else if(result > 0)
   {
      return 1;
   }

   // object path is equal

   if(busType < ref.busType)
   {
      return -1;
   }
   else if(busType > ref.busType)
   {
      return 1;
   }

   // bus type is equal

   return 0;
}

//###########################################################################

DbusServiceInfo::DbusServiceInfo()
{
   // busName
   // objPath
   busType = ::ccdbusif::BUS_TYPE_SYSTEM;
   serviceAvailability = BTS_DBUS_SERVICE_NOT_AVAILABLE;
   _ipc2BtsSendIf = NULL;
   _interface = BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE;
   _timer = NULL;
}

DbusServiceInfo::DbusServiceInfo(IN const DbusServiceInfo& ref)
{
   busName = ref.busName;
   objPath = ref.objPath;
   busType = ref.busType;
   serviceAvailability = ref.serviceAvailability;
   _ipc2BtsSendIf = NULL;
   setIpc2BtsSendIf(ref._ipc2BtsSendIf);
   _interface = ref._interface;
   _timer = NULL;
   setTimer(ref._timer);
}

DbusServiceInfo& DbusServiceInfo::operator=(IN const DbusServiceInfo& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   busName = ref.busName;
   objPath = ref.objPath;
   busType = ref.busType;
   serviceAvailability = ref.serviceAvailability;
   _ipc2BtsSendIf = NULL;
   setIpc2BtsSendIf(ref._ipc2BtsSendIf);
   _interface = ref._interface;
   _timer = NULL;
   setTimer(ref._timer);

   return *this;
}

bool DbusServiceInfo::operator==(IN const DbusServiceInfo& ref) const
{
   bool result = true;

   result = (true == result) && (busName == ref.busName);
   result = (true == result) && (objPath == ref.objPath);
   result = (true == result) && (busType == ref.busType);
   result = (true == result) && (serviceAvailability == ref.serviceAvailability);
   result = (true == result) && (_ipc2BtsSendIf == ref._ipc2BtsSendIf);
   result = (true == result) && (_interface == ref._interface);
   result = (true == result) && (_timer == ref._timer);

   return result;
}

bool DbusServiceInfo::operator!=(IN const DbusServiceInfo& ref) const
{
   return !(operator==(ref));
}

DbusServiceInfo::~DbusServiceInfo()
{
   _ipc2BtsSendIf = NULL;
   _timer = 0;
}

void DbusServiceInfo::handleTimeout(IN const BTSTimerId timerId)
{
   (void)(timerId);

   ETG_TRACE_USR3((" handleTimeout()"));

   // timeout happened => sent event
   if(NULL != _ipc2BtsSendIf)
   {
      _ipc2BtsSendIf->createDbusServiceAvailabilityMessage(_interface, BTS_DBUS_SERVICE_NOT_AVAILABLE, busName, objPath, busType);
   }
}

void DbusServiceInfo::startTimer(void)
{
   ETG_TRACE_USR3((" startTimer()"));

   if(NULL == _timer)
   {
      // create timer
      _timer = new Timer();
   }

   if(NULL == _timer)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   _timer->start(_dbusServiceTimeout, this, false);
}

void DbusServiceInfo::stopTimer(void)
{
   ETG_TRACE_USR3((" stopTimer()"));

   if(NULL == _timer)
   {
      return;
   }

   _timer->stop();
}

void DbusServiceInfo::releaseTimer(void)
{
   ETG_TRACE_USR3((" releaseTimer()"));

   if(NULL == _timer)
   {
      return;
   }

   _timer->stop();
   delete _timer;
   _timer = NULL;
}

void DbusServiceInfo::setTimeout(IN const BTSTimeValue timeout)
{
   _dbusServiceTimeout = timeout;
}

//###########################################################################

DbusServiceStatus::DbusServiceStatus()
{
   _enabled = false;
   // _name
   // _info
   _timerEnabled = true;
   _ipc2BtsSendIf = NULL;
   _interface = BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE;
}

DbusServiceStatus::DbusServiceStatus(IN const ::std::string& name)
{
   _enabled = false;
   _name = name;
   // _info
   _timerEnabled = true;
   _ipc2BtsSendIf = NULL;
   _interface = BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE;
}

DbusServiceStatus::~DbusServiceStatus()
{
   _info.clear();
   _ipc2BtsSendIf = NULL;
}

BTSDbusServiceAvailability DbusServiceStatus::getGroupAvailability(void) const
{
   BTSDbusServiceAvailability returnAvailability = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   // consider following rules (in given order):
   // - if at least one service is NOT_AVAILABLE then set group status to NOT_AVAILABLE
   // - if at least one service is WAITING then set group status to WAITING
   // - if all services are AVAILABLE then set group status to AVAILABLE

   size_t countNotAvailable = 0;
   size_t countWaiting = 0;
   size_t countAvailable = 0;

   for(::std::map<DbusServiceEntry, DbusServiceInfo>::const_iterator it = _info.begin(); it != _info.end(); ++it)
   {
      if(BTS_DBUS_SERVICE_NOT_AVAILABLE == it->second.serviceAvailability)
      {
         countNotAvailable++;
      }
      else if(BTS_DBUS_SERVICE_WAITING == it->second.serviceAvailability)
      {
         countWaiting++;
      }
      else if(BTS_DBUS_SERVICE_AVAILABLE == it->second.serviceAvailability)
      {
         countAvailable++;
      }
   }

   if(0 < countNotAvailable)
   {
      returnAvailability = BTS_DBUS_SERVICE_NOT_AVAILABLE;
   }
   else if(0 < countWaiting)
   {
      returnAvailability = BTS_DBUS_SERVICE_WAITING;
   }
   else if(0 < countAvailable)
   {
      returnAvailability = BTS_DBUS_SERVICE_AVAILABLE;
   }
   else
   {
      // not matching entries
      FW_NORMAL_ASSERT_ALWAYS();
   }

   return returnAvailability;
}

void DbusServiceStatus::removeAllServiceInfos(void)
{
   for(::std::map<DbusServiceEntry, DbusServiceInfo>::iterator it = _info.begin(); it != _info.end(); ++it)
   {
      it->second.releaseTimer();
   }

   _info.clear();
}

//###########################################################################
// --- for all interfaces with fixed object path ---
void DbusServiceStatus::addServiceInfo(IN const BTSObjectPath& objPath)
{
   ETG_TRACE_USR2((" addServiceInfo(): [%20s] objPath=%s", _name.c_str(), objPath.c_str()));
   addServiceInfo("", BTS_PROTO_LAST, "", 0, objPath);
}

void DbusServiceStatus::removeServiceInfo(void)
{
   ETG_TRACE_USR2((" removeServiceInfo(): [%20s]", _name.c_str()));
   removeServiceInfo("", BTS_PROTO_LAST, "", 0);
}

void DbusServiceStatus::setAvailability(IN const BTSDbusServiceAvailability availability)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
   ETG_TRACE_USR2((" setAvailability(): [%20s] availability=%d", _name.c_str(), ETG_ENUM(TRC_BTSDbusServiceAvailability, availability)));
#else
   BTS_TRACE_USR2((" setAvailability(): [%20s] availability=%s", _name.c_str(), getDbusServiceAvailability2String(availability)));
#endif
   setAvailability("", BTS_PROTO_LAST, "", 0, availability);
}

BTSDbusServiceAvailability DbusServiceStatus::getAvailability(void) const
{
   return getAvailability("", BTS_PROTO_LAST, "", 0);
}

//###########################################################################
// --- for all interfaces based on bus type, bus name and object path ---
void DbusServiceStatus::addServiceInfo(const ::ccdbusif::DbusBusType busType, IN const BTSBusName& busName, IN const BTSObjectPath& objPath)
{
   if(::ccdbusif::BUS_TYPE_SYSTEM == busType)
   {
      ETG_TRACE_USR2((" addServiceInfo(): [%20s] busType=SYSTEM busName=%s", _name.c_str(), busName.c_str()));
   }
   else
   {
      ETG_TRACE_USR2((" addServiceInfo(): [%20s] busType=SESSION busName=%s", _name.c_str(), busName.c_str()));
   }
   ETG_TRACE_USR2((" addServiceInfo(): [%20s] objPath=%s", _name.c_str(), objPath.c_str()));

   DbusServiceEntry entry;
   // entry.deviceAddress
   // entry.protocolId
   // entry.sppUuid
   // entry.masInstanceId
   entry.busName = busName;
   entry.objPath = objPath;
   entry.busType = busType;

   ::std::map<DbusServiceEntry, DbusServiceInfo>::const_iterator it = _info.find(entry);

   if(_info.end() != it)
   {
      // entry is already stored
      FW_NORMAL_ASSERT_ALWAYS();
   }
   else
   {
      DbusServiceInfo info;
      info.busName = busName;
      info.objPath = objPath;
      info.busType = busType;
      info.serviceAvailability = BTS_DBUS_SERVICE_NOT_AVAILABLE;
      info.setIpc2BtsSendIf(_ipc2BtsSendIf);
      info.setInterface(_interface);

      _info[entry] = info;
   }
}

void DbusServiceStatus::removeServiceInfo(const ::ccdbusif::DbusBusType busType, IN const BTSBusName& busName, IN const BTSObjectPath& objPath)
{
   if(::ccdbusif::BUS_TYPE_SYSTEM == busType)
   {
      ETG_TRACE_USR2((" removeServiceInfo(): [%20s] busType=SYSTEM busName=%s", _name.c_str(), busName.c_str()));
   }
   else
   {
      ETG_TRACE_USR2((" removeServiceInfo(): [%20s] busType=SESSION busName=%s", _name.c_str(), busName.c_str()));
   }
   ETG_TRACE_USR2((" removeServiceInfo(): [%20s] objPath=%s", _name.c_str(), objPath.c_str()));

   DbusServiceEntry entry;
   // entry.deviceAddress
   // entry.protocolId
   // entry.sppUuid
   // entry.masInstanceId
   entry.busName = busName;
   entry.objPath = objPath;
   entry.busType = busType;

   ::std::map<DbusServiceEntry, DbusServiceInfo>::iterator it = _info.find(entry);

   if(_info.end() == it)
   {
      // entry is already erased
      FW_NORMAL_ASSERT_ALWAYS();
   }
   else
   {
      // release timer before removing entry
      it->second.releaseTimer();
      _info.erase(it);
   }
}

void DbusServiceStatus::setAvailability(const ::ccdbusif::DbusBusType busType, IN const BTSBusName& busName, IN const BTSObjectPath& objPath, IN const BTSDbusServiceAvailability availability)
{
   if(::ccdbusif::BUS_TYPE_SYSTEM == busType)
   {
      ETG_TRACE_USR2((" setAvailability(): [%20s] busType=SYSTEM busName=%s", _name.c_str(), busName.c_str()));
   }
   else
   {
      ETG_TRACE_USR2((" setAvailability(): [%20s] busType=SESSION busName=%s", _name.c_str(), busName.c_str()));
   }
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
   ETG_TRACE_USR2((" setAvailability(): [%20s] availability=%d objPath=%s", _name.c_str(), ETG_ENUM(TRC_BTSDbusServiceAvailability, availability), objPath.c_str()));
#else
   BTS_TRACE_USR2((" setAvailability(): [%20s] availability=%s objPath=%s", _name.c_str(), getDbusServiceAvailability2String(availability), objPath.c_str()));
#endif

   if(BTS_DBUS_SERVICE_LAST <= availability)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   DbusServiceEntry entry;
   // entry.deviceAddress
   // entry.protocolId
   // entry.sppUuid
   // entry.masInstanceId
   entry.busName = busName;
   entry.objPath = objPath;
   entry.busType = busType;

   ::std::map<DbusServiceEntry, DbusServiceInfo>::iterator it = _info.find(entry);

   if(_info.end() == it)
   {
      // entry is already erased
      FW_NORMAL_ASSERT_ALWAYS();
   }
   else
   {
      setAvailability(it->second, availability);
   }
}

BTSDbusServiceAvailability DbusServiceStatus::getAvailability(const ::ccdbusif::DbusBusType busType, IN const BTSBusName& busName, IN const BTSObjectPath& objPath) const
{
   if(::ccdbusif::BUS_TYPE_SYSTEM == busType)
   {
      ETG_TRACE_USR2((" getAvailability(): [%20s] busType=SYSTEM busName=%s", _name.c_str(), busName.c_str()));
   }
   else
   {
      ETG_TRACE_USR2((" getAvailability(): [%20s] busType=SESSION busName=%s", _name.c_str(), busName.c_str()));
   }
   ETG_TRACE_USR2((" getAvailability(): [%20s] objPath=%s", _name.c_str(), objPath.c_str()));

   BTSDbusServiceAvailability currentAvailability = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   DbusServiceEntry entry;
   // entry.deviceAddress
   // entry.protocolId
   // entry.sppUuid
   // entry.masInstanceId
   entry.busName = busName;
   entry.objPath = objPath;
   entry.busType = busType;

   ::std::map<DbusServiceEntry, DbusServiceInfo>::const_iterator it = _info.find(entry);

   if(_info.end() == it)
   {
      // entry is already erased
      FW_NORMAL_ASSERT_ALWAYS();
   }
   else
   {
      currentAvailability = it->second.serviceAvailability;
   }

   return currentAvailability;
}

//###########################################################################
// --- for all interfaces with variable object path based on device address ---
void DbusServiceStatus::addServiceInfo(IN const BTSBDAddress& address, IN const BTSObjectPath& objPath)
{
   BTSBDAddressTrace traceAddr(address);

   ETG_TRACE_USR2((" addServiceInfo(): [%20s] address=%08X%04X objPath=%s", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), objPath.c_str()));
   addServiceInfo(address, BTS_PROTO_LAST, "", 0, objPath);
}

void DbusServiceStatus::removeServiceInfo(IN const BTSBDAddress& address)
{
   BTSBDAddressTrace traceAddr(address);

   ETG_TRACE_USR2((" removeServiceInfo(): [%20s] address=%08X%04X", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower()));
   removeServiceInfo(address, BTS_PROTO_LAST, "", 0);
}

void DbusServiceStatus::setAvailability(IN const BTSBDAddress& address, IN const BTSDbusServiceAvailability availability)
{
   BTSBDAddressTrace traceAddr(address);

#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
   ETG_TRACE_USR2((" setAvailability(): [%20s] address=%08X%04X availability=%d", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), ETG_ENUM(TRC_BTSDbusServiceAvailability, availability)));
#else
   BTS_TRACE_USR2((" setAvailability(): [%20s] address=%08X%04X availability=%s", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), getDbusServiceAvailability2String(availability)));
#endif
   setAvailability(address, BTS_PROTO_LAST, "", 0, availability);
}

BTSDbusServiceAvailability DbusServiceStatus::getAvailability(IN const BTSBDAddress& address) const
{
   return getAvailability(address, BTS_PROTO_LAST, "", 0);
}

bool DbusServiceStatus::isEntryAvailable(IN const BTSBDAddress& address) const
{
   return isEntryAvailable(address, BTS_PROTO_LAST, "", 0);
}

//###########################################################################
// --- for all interfaces with variable object path based on device address + protocol id ---
void DbusServiceStatus::addServiceInfo(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSObjectPath& objPath)
{
   BTSBDAddressTrace traceAddr(address);

#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
   ETG_TRACE_USR2((" addServiceInfo(): [%20s] address=%08X%04X protocolId=%d objPath=%s", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), ETG_ENUM(TRC_BTSProtocolId, protocolId), objPath.c_str()));
#else
   BTS_TRACE_USR2((" addServiceInfo(): [%20s] address=%08X%04X protocolId=%s objPath=%s", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), getProtocolId2String(protocolId), objPath.c_str()));
#endif
   addServiceInfo(address, protocolId, "", 0, objPath);
}

void DbusServiceStatus::removeServiceInfo(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId)
{
   BTSBDAddressTrace traceAddr(address);

#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
   ETG_TRACE_USR2((" removeServiceInfo(): [%20s] address=%08X%04X protocolId=%d", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), ETG_ENUM(TRC_BTSProtocolId, protocolId)));
#else
   BTS_TRACE_USR2((" removeServiceInfo(): [%20s] address=%08X%04X protocolId=%s", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), getProtocolId2String(protocolId)));
#endif
   removeServiceInfo(address, protocolId, "", 0);
}

void DbusServiceStatus::setAvailability(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSDbusServiceAvailability availability)
{
   BTSBDAddressTrace traceAddr(address);

#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
   ETG_TRACE_USR2((" setAvailability(): [%20s] address=%08X%04X protocolId=%d availability=%d", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), ETG_ENUM(TRC_BTSProtocolId, protocolId), ETG_ENUM(TRC_BTSDbusServiceAvailability, availability)));
#else
   BTS_TRACE_USR2((" setAvailability(): [%20s] address=%08X%04X protocolId=%s availability=%s", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), getProtocolId2String(protocolId), getDbusServiceAvailability2String(availability)));
#endif
   setAvailability(address, protocolId, "", 0, availability);
}

BTSDbusServiceAvailability DbusServiceStatus::getAvailability(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId) const
{
   return getAvailability(address, protocolId, "", 0);
}

//###########################################################################
// --- for all interfaces with variable object path based on device address + protocol id + SPP uuid ---
void DbusServiceStatus::addServiceInfo(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid, IN const BTSObjectPath& objPath)
{
   BTSBDAddressTrace traceAddr(address);

#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
   ETG_TRACE_USR2((" addServiceInfo(): [%20s] address=%08X%04X protocolId=%d uuid=%s", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), ETG_ENUM(TRC_BTSProtocolId, protocolId), uuid.c_str()));
   ETG_TRACE_USR2((" addServiceInfo(): [%20s] objPath=%s", _name.c_str(), objPath.c_str()));
#else
   BTS_TRACE_USR2((" addServiceInfo(): [%20s] address=%08X%04X protocolId=%s uuid=%s", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), getProtocolId2String(protocolId), uuid.c_str()));
   BTS_TRACE_USR2((" addServiceInfo(): [%20s] objPath=%s", _name.c_str(), objPath.c_str()));
#endif
   addServiceInfo(address, protocolId, uuid, 0, objPath);
}

void DbusServiceStatus::removeServiceInfo(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid)
{
   BTSBDAddressTrace traceAddr(address);

#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
   ETG_TRACE_USR2((" removeServiceInfo(): [%20s] address=%08X%04X protocolId=%d uuid=%s", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), ETG_ENUM(TRC_BTSProtocolId, protocolId), uuid.c_str()));
#else
   BTS_TRACE_USR2((" removeServiceInfo(): [%20s] address=%08X%04X protocolId=%s uuid=%s", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), getProtocolId2String(protocolId), uuid.c_str()));
#endif
   removeServiceInfo(address, protocolId, uuid, 0);
}

void DbusServiceStatus::setAvailability(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid, IN const BTSDbusServiceAvailability availability)
{
   BTSBDAddressTrace traceAddr(address);

#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
   ETG_TRACE_USR2((" setAvailability(): [%20s] address=%08X%04X protocolId=%d uuid=%s", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), ETG_ENUM(TRC_BTSProtocolId, protocolId), uuid.c_str()));
   ETG_TRACE_USR2((" setAvailability(): [%20s] availability=%d", _name.c_str(), ETG_ENUM(TRC_BTSDbusServiceAvailability, availability)));
#else
   BTS_TRACE_USR2((" setAvailability(): [%20s] address=%08X%04X protocolId=%s uuid=%s", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), getProtocolId2String(protocolId), uuid.c_str()));
   BTS_TRACE_USR2((" setAvailability(): [%20s] availability=%s", _name.c_str(), getDbusServiceAvailability2String(availability)));
#endif
   setAvailability(address, protocolId, uuid, 0, availability);
}

BTSDbusServiceAvailability DbusServiceStatus::getAvailability(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid) const
{
   return getAvailability(address, protocolId, uuid, 0);
}

//###########################################################################
// --- for all interfaces with variable object path based on device address + protocol id + MAS instance id ---
void DbusServiceStatus::addServiceInfo(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSMasInstanceId instanceId, IN const BTSObjectPath& objPath)
{
   BTSBDAddressTrace traceAddr(address);

#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
   ETG_TRACE_USR2((" addServiceInfo(): [%20s] address=%08X%04X protocolId=%d instanceId=%u objPath=%s", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), ETG_ENUM(TRC_BTSProtocolId, protocolId), instanceId, objPath.c_str()));
#else
   BTS_TRACE_USR2((" addServiceInfo(): [%20s] address=%08X%04X protocolId=%s instanceId=%u objPath=%s", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), getProtocolId2String(protocolId), instanceId, objPath.c_str()));
#endif
   addServiceInfo(address, protocolId, "", instanceId, objPath);
}

void DbusServiceStatus::removeServiceInfo(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSMasInstanceId instanceId)
{
   BTSBDAddressTrace traceAddr(address);

#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
   ETG_TRACE_USR2((" removeServiceInfo(): [%20s] address=%08X%04X protocolId=%d instanceId=%u", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), ETG_ENUM(TRC_BTSProtocolId, protocolId), instanceId));
#else
   BTS_TRACE_USR2((" removeServiceInfo(): [%20s] address=%08X%04X protocolId=%s instanceId=%u", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), getProtocolId2String(protocolId), instanceId));
#endif
   removeServiceInfo(address, protocolId, "", instanceId);
}

void DbusServiceStatus::setAvailability(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSMasInstanceId instanceId, IN const BTSDbusServiceAvailability availability)
{
   BTSBDAddressTrace traceAddr(address);

#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
   ETG_TRACE_USR2((" setAvailability(): [%20s] address=%08X%04X protocolId=%d instanceId=%u availability=%d", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), ETG_ENUM(TRC_BTSProtocolId, protocolId), instanceId, ETG_ENUM(TRC_BTSDbusServiceAvailability, availability)));
#else
   BTS_TRACE_USR2((" setAvailability(): [%20s] address=%08X%04X protocolId=%s instanceId=%u availability=%s", _name.c_str(), traceAddr.getUpper(), traceAddr.getLower(), getProtocolId2String(protocolId), instanceId, getDbusServiceAvailability2String(availability)));
#endif
   setAvailability(address, protocolId, "", instanceId, availability);
}

BTSDbusServiceAvailability DbusServiceStatus::getAvailability(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSMasInstanceId instanceId) const
{
   return getAvailability(address, protocolId, "", instanceId);
}

//###########################################################################
// --- internal ---
void DbusServiceStatus::addServiceInfo(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid, IN const BTSMasInstanceId instanceId, IN const BTSObjectPath& objPath)
{
   DbusServiceEntry entry;
   entry.deviceAddress = address;
   entry.protocolId = protocolId;
   entry.sppUuid = uuid;
   entry.masInstanceId = instanceId;
   // entry.busName
   // entry.objPath
   // entry.busType

   ::std::map<DbusServiceEntry, DbusServiceInfo>::const_iterator it = _info.find(entry);

   if(_info.end() != it)
   {
      // entry is already stored
      FW_NORMAL_ASSERT_ALWAYS();
   }
   else
   {
      DbusServiceInfo info;
      // info.busName
      info.objPath = objPath;
      // info.busType
      info.serviceAvailability = BTS_DBUS_SERVICE_NOT_AVAILABLE;
      info.setIpc2BtsSendIf(_ipc2BtsSendIf);
      info.setInterface(_interface);

      _info[entry] = info;
   }
}

void DbusServiceStatus::removeServiceInfo(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid, IN const BTSMasInstanceId instanceId)
{
   DbusServiceEntry entry;
   entry.deviceAddress = address;
   entry.protocolId = protocolId;
   entry.sppUuid = uuid;
   entry.masInstanceId = instanceId;
   // entry.busName
   // entry.objPath
   // entry.busType

   ::std::map<DbusServiceEntry, DbusServiceInfo>::iterator it = _info.find(entry);

   if(_info.end() == it)
   {
      // entry is already erased
      FW_NORMAL_ASSERT_ALWAYS();
   }
   else
   {
      // release timer before removing entry
      it->second.releaseTimer();
      _info.erase(it);
   }
}

void DbusServiceStatus::setAvailability(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid, IN const BTSMasInstanceId instanceId, IN const BTSDbusServiceAvailability availability)
{
   if(BTS_DBUS_SERVICE_LAST <= availability)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   DbusServiceEntry entry;
   entry.deviceAddress = address;
   entry.protocolId = protocolId;
   entry.sppUuid = uuid;
   entry.masInstanceId = instanceId;
   // entry.busName
   // entry.objPath
   // entry.busType

   ::std::map<DbusServiceEntry, DbusServiceInfo>::iterator it = _info.find(entry);

   if(_info.end() == it)
   {
      // entry is already erased
      FW_NORMAL_ASSERT_ALWAYS();
   }
   else
   {
      setAvailability(it->second, availability);

#if 0
      // consider following rules:
      // - NOT_AVAILABLE can be overwritten by WAITING or AVAILABLE
      // - WAITING can be overwritten by NOT_AVAILABLE or AVAILABLE
      // - AVAILABLE can be overwritten by NOT_AVAILABLE only => transition from AVAILABLE to WAITING is not valid
      BTSDbusServiceAvailability currentAvailability = it->second.serviceAvailability;

      if(BTS_DBUS_SERVICE_NOT_AVAILABLE == currentAvailability)
      {
         if(currentAvailability != availability)
         {
            it->second.serviceAvailability = availability;

            if(BTS_DBUS_SERVICE_WAITING == availability)
            {
               // start timer
               if(true == _timerEnabled)
               {
                  it->second.startTimer();
               }
            }
         }
      }
      else if(BTS_DBUS_SERVICE_WAITING == currentAvailability)
      {
         if(currentAvailability != availability)
         {
            it->second.serviceAvailability = availability;
            // stop timer
            it->second.stopTimer();
         }
      }
      else if(BTS_DBUS_SERVICE_AVAILABLE == currentAvailability)
      {
         if((currentAvailability != availability) && (BTS_DBUS_SERVICE_WAITING != availability))
         {
            it->second.serviceAvailability = availability;
         }
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
#endif
   }
}

void DbusServiceStatus::setAvailability(IN DbusServiceInfo& info, IN const BTSDbusServiceAvailability availability)
{
   if(BTS_DBUS_SERVICE_LAST <= availability)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // consider following rules:
   // - NOT_AVAILABLE can be overwritten by WAITING or AVAILABLE
   // - WAITING can be overwritten by NOT_AVAILABLE or AVAILABLE
   // - AVAILABLE can be overwritten by NOT_AVAILABLE only => transition from AVAILABLE to WAITING is not valid
   BTSDbusServiceAvailability currentAvailability = info.serviceAvailability;

   if(BTS_DBUS_SERVICE_NOT_AVAILABLE == currentAvailability)
   {
      if(currentAvailability != availability)
      {
         info.serviceAvailability = availability;

         if(BTS_DBUS_SERVICE_WAITING == availability)
         {
            // start timer
            if(true == _timerEnabled)
            {
               info.startTimer();
            }
         }
      }
   }
   else if(BTS_DBUS_SERVICE_WAITING == currentAvailability)
   {
      if(currentAvailability != availability)
      {
         info.serviceAvailability = availability;
         // stop timer
         info.stopTimer();
      }
   }
   else if(BTS_DBUS_SERVICE_AVAILABLE == currentAvailability)
   {
      if((currentAvailability != availability) && (BTS_DBUS_SERVICE_WAITING != availability))
      {
         info.serviceAvailability = availability;
      }
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

BTSDbusServiceAvailability DbusServiceStatus::getAvailability(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid, IN const BTSMasInstanceId instanceId) const
{
   BTSDbusServiceAvailability currentAvailability = BTS_DBUS_SERVICE_NOT_AVAILABLE;

   DbusServiceEntry entry;
   entry.deviceAddress = address;
   entry.protocolId = protocolId;
   entry.sppUuid = uuid;
   entry.masInstanceId = instanceId;
   // entry.busName
   // entry.objPath
   // entry.busType

   ::std::map<DbusServiceEntry, DbusServiceInfo>::const_iterator it = _info.find(entry);

   if(_info.end() == it)
   {
      // entry is already erased
      FW_NORMAL_ASSERT_ALWAYS();
   }
   else
   {
      currentAvailability = it->second.serviceAvailability;
   }

   return currentAvailability;
}

bool DbusServiceStatus::isEntryAvailable(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid, IN const BTSMasInstanceId instanceId) const
{
   DbusServiceEntry entry;
   entry.deviceAddress = address;
   entry.protocolId = protocolId;
   entry.sppUuid = uuid;
   entry.masInstanceId = instanceId;
   // entry.busName
   // entry.objPath
   // entry.busType

   return (_info.end() != _info.find(entry));
}

//###########################################################################

DbusServiceGroupStatus::DbusServiceGroupStatus()
{
   _groupInfo.reserve(50);
}

DbusServiceGroupStatus::DbusServiceGroupStatus(IN const ::std::string& name) : DbusServiceStatus(name)
{
   _groupInfo.reserve(50);
}

DbusServiceGroupStatus::~DbusServiceGroupStatus()
{
   _groupInfo.clear();
}

void DbusServiceGroupStatus::addServiceStatus(IN DbusServiceStatus* status)
{
   _groupInfo.push_back(status);
}

void DbusServiceGroupStatus::checkAvailability(void)
{
   // consider following rules (in given order):
   // - if at least one service is NOT_AVAILABLE then set group status to NOT_AVAILABLE
   // - if at least one service is WAITING then set group status to WAITING
   // - if all services are AVAILABLE then set group status to AVAILABLE

   size_t countNotAvailable = 0;
   size_t countWaiting = 0;
   size_t countAvailable = 0;

   for(size_t i = 0; i < _groupInfo.size(); i++)
   {
      if(NULL != _groupInfo[i])
      {
         if(BTS_DBUS_SERVICE_NOT_AVAILABLE == _groupInfo[i]->getGroupAvailability())
         {
            countNotAvailable++;
         }
         else if(BTS_DBUS_SERVICE_WAITING == _groupInfo[i]->getGroupAvailability())
         {
            countWaiting++;
         }
         else if(BTS_DBUS_SERVICE_AVAILABLE == _groupInfo[i]->getGroupAvailability())
         {
            countAvailable++;
         }
      }
   }

   if(0 < countNotAvailable)
   {
      setAvailability(BTS_DBUS_SERVICE_NOT_AVAILABLE);
   }
   else if(0 < countWaiting)
   {
      setAvailability(BTS_DBUS_SERVICE_WAITING);
   }
   else if(0 < countAvailable)
   {
      setAvailability(BTS_DBUS_SERVICE_AVAILABLE);
   }
   else
   {
      // not matching entries
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

} //btstackif
