/**
 * @file DbusCallbackIf.cpp
 *
 * @par SW-Component
 * IPC
 *
 * @brief DBUS basic callback handling.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Base class for DBUS callback handling.
 */

#include "asf/dbus/DBusTypes.h"

#include "DbusCallbackIf.h"
#include "IDbusRecHandler.h"
#include "Ipc2Bts_BaseMessage.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_DBUS
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/DbusCallbackIf.cpp.trc.h"
#endif
#endif

namespace btstackif {

DbusCallbackIf::DbusCallbackIf()
{
   _recHandler = NULL;
}

DbusCallbackIf::DbusCallbackIf(IDbusRecHandler* recHandler)
{
   _recHandler = recHandler;
}

DbusCallbackIf::~DbusCallbackIf()
{
   _recHandler = NULL;
}

void DbusCallbackIf::onResponse(Ipc2Bts_BaseMessage* ptrMsg, const ::boost::shared_ptr< ::asf::dbus::DBusTypes::DefaultMessage >& response)
{
   if(NULL == _recHandler)
   {
      if(NULL != ptrMsg)
      {
         delete ptrMsg;
      }
      return;
   }

   if(NULL != ptrMsg)
   {
      ::ccdbusif::ActInfo info(response->getAct());
      ETG_TRACE_USR3((" onResponse(): Ipc2Bts 0x%04X received: token=%s", ptrMsg->getTraceOpCode(), info.getInfo()));

      ptrMsg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
      ptrMsg->setDbusToken(response->getAct());

      _recHandler->handleReceivedDbusMessage(ptrMsg);
   }
}

void DbusCallbackIf::onResponse(Ipc2Bts_BaseMessage* ptrMsg, act_t token)
{
   if(NULL == _recHandler)
   {
      if(NULL != ptrMsg)
      {
         delete ptrMsg;
      }
      return;
   }

   if(NULL != ptrMsg)
   {
      ::ccdbusif::ActInfo info(token);
      ETG_TRACE_USR3((" onResponse(): Ipc2Bts 0x%04X received: token=%s", ptrMsg->getTraceOpCode(), info.getInfo()));

      ptrMsg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
      ptrMsg->setDbusToken(token);

      _recHandler->handleReceivedDbusMessage(ptrMsg);
   }
}

void DbusCallbackIf::onSignal(Ipc2Bts_BaseMessage* ptrMsg, const ::boost::shared_ptr< ::asf::dbus::DBusTypes::DefaultMessage >& signal)
{
   if(NULL == _recHandler)
   {
      if(NULL != ptrMsg)
      {
         delete ptrMsg;
      }
      return;
   }

   if(NULL != ptrMsg)
   {
      ::ccdbusif::ActInfo info(signal->getAct());
      ETG_TRACE_USR3((" onSignal(): Ipc2Bts 0x%04X received: token=%s", ptrMsg->getTraceOpCode(), info.getInfo()));

      ptrMsg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
      ptrMsg->setDbusToken(signal->getAct());

      _recHandler->handleReceivedDbusMessage(ptrMsg);
   }
}

void DbusCallbackIf::onSignal(Ipc2Bts_BaseMessage* ptrMsg, act_t token, const bool highPrio /*= false*/)
{
   if(NULL == _recHandler)
   {
      if(NULL != ptrMsg)
      {
         delete ptrMsg;
      }
      return;
   }

   if(NULL != ptrMsg)
   {
      ::ccdbusif::ActInfo info(token);
      ETG_TRACE_USR3((" onSignal(): Ipc2Bts 0x%04X received: token=%s", ptrMsg->getTraceOpCode(), info.getInfo()));

      ptrMsg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
      ptrMsg->setDbusToken(token);

      _recHandler->handleReceivedDbusMessage(ptrMsg, highPrio);
   }
}

void DbusCallbackIf::onError(Ipc2Bts_BaseMessage* ptrMsg, const ::boost::shared_ptr< ::asf::dbus::DBusTypes::DBusError >& error)
{
   if(NULL == _recHandler)
   {
      if(NULL != ptrMsg)
      {
         delete ptrMsg;
      }
      return;
   }

   if(NULL != ptrMsg)
   {
      ::ccdbusif::ActInfo info(error->getAct());
      ETG_TRACE_USR3((" onError(): Ipc2Bts 0x%04X received: token=%s", ptrMsg->getTraceOpCode(), info.getInfo()));

      ptrMsg->setIpcCommonErrorCode(BTS_IPC_ERROR);
      ptrMsg->setDbusToken(error->getAct());
      ptrMsg->setErrorName(error->getName());
      ptrMsg->setErrorMessage(error->getMessage());

      _recHandler->handleReceivedDbusMessage(ptrMsg);
   }
}

void DbusCallbackIf::onError(Ipc2Bts_BaseMessage* ptrMsg, const ::boost::shared_ptr< ::asf::dbus::DBusTypes::DBusError >& error, const BTSIpcCommonErrorCode errorCode)
{
   if(NULL == _recHandler)
   {
      if(NULL != ptrMsg)
      {
         delete ptrMsg;
      }
      return;
   }

   if(NULL != ptrMsg)
   {
      ::ccdbusif::ActInfo info(error->getAct());
      ETG_TRACE_USR3((" onError(): Ipc2Bts 0x%04X received: token=%20s errorCode=%d", ptrMsg->getTraceOpCode(), info.getInfo(), errorCode));

      ptrMsg->setIpcCommonErrorCode(errorCode);
      ptrMsg->setDbusToken(error->getAct());
      ptrMsg->setErrorName(error->getName());
      ptrMsg->setErrorMessage(error->getMessage());

      _recHandler->handleReceivedDbusMessage(ptrMsg);
   }
}

void DbusCallbackIf::onUpdate(Ipc2Bts_BaseMessage* ptrMsg, act_t token)
{
   if(NULL == _recHandler)
   {
      if(NULL != ptrMsg)
      {
         delete ptrMsg;
      }
      return;
   }

   if(NULL != ptrMsg)
   {
      ::ccdbusif::ActInfo info(token);
      ETG_TRACE_USR3((" onUpdate(): Ipc2Bts 0x%04X received: token=%s", ptrMsg->getTraceOpCode(), info.getInfo()));

      ptrMsg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
      ptrMsg->setDbusToken(token);

      _recHandler->handleReceivedDbusMessage(ptrMsg);
   }
}

void DbusCallbackIf::onRequest(Ipc2Bts_BaseMessage* ptrMsg, act_t token)
{
   if(NULL == _recHandler)
   {
      if(NULL != ptrMsg)
      {
         delete ptrMsg;
      }
      return;
   }

   if(NULL != ptrMsg)
   {
      ::ccdbusif::ActInfo info(token);
      ETG_TRACE_USR3((" onRequest(): Ipc2Bts 0x%04X received: token=%s", ptrMsg->getTraceOpCode(), info.getInfo()));

      ptrMsg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
      ptrMsg->setDbusToken(token);

      _recHandler->handleReceivedDbusMessage(ptrMsg);
   }
}

} //btstackif
