/**
 * @file IDbusIfHandler.h
 *
 * @par SW-Component
 * Interface for DBus interface handler
 *
 * @brief Interface definition for DBus interface handler.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the interface for DBus interface handler.
 */

#ifndef _I_DBUS_IF_HANDLER_H_
#define _I_DBUS_IF_HANDLER_H_

#include "BtStackInternalTypes.h"

// class forward declarations
namespace ccdbusif {
class ICcDbusIfController;
} //ccdbusif

namespace btstackif {

// class forward declarations
class IDbusBase;
class IDbusRecHandler;
class Bts2Ipc_BaseMessage;
class Ipc2Bts_BaseMessage;

/**
 * Interface definition.
 */
class IDbusIfHandler
{
public:
   /**
    * Destructor.
    */
   virtual ~IDbusIfHandler() {}

   /**
    * Set DBus base interface.
    *
    * @param[in] dbusBase: Bus base interface
    */
   virtual void setDbusBaseIf(IN IDbusBase* dbusBase) = 0;

   /**
    * Set DBus receive handler interface.
    *
    * @param[in] dbusRecHandler: DBus receive handler interface
    */
   virtual void setDbusRecHandlerIf(IN IDbusRecHandler* dbusRecHandler) = 0;

   /**
    * Enable/disable dbus if handler. TODO: double check if this function is really needed
    *
    * @param[in] enable: enable flag
    */
   virtual void enable(IN const bool enable) = 0;

   /**
    * Initialize. TODO: double check if this function is really needed
    */
   virtual void initialize(void) = 0;

   /**
    * Destroy. TODO: double check if this function is really needed
    */
   virtual void destroy(void) = 0;

   /**
    * Set interface for CcDbusIf controller.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] userMode: user mode
    * @param[in] controller: CcDbusIf controller (pointer)
    * @param[in] dbusInterfaces: list of needed DBUS interfaces
    * @param[in] configuration: local stack configuration data
    */
   virtual void setCcDbusIfControllerIf(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
            IN ::ccdbusif::ICcDbusIfController* controller, IN const ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration) = 0;

   /**
    * Reset interface for CcDbusIf controller.
    */
   virtual void resetCcDbusIfControllerIf(void) = 0;

   /**
    * Set a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("dbus if handler is responsible for this message request"),
    * @return = false: given message was not handled ("dbus if handler is not responsible for this message request")
    */
   virtual bool setSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData) = 0;

   /**
    * Set a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("dbus if handler is responsible for this message request"),
    * @return = false: given message was not handled ("dbus if handler is not responsible for this message request")
    */
   virtual bool setSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData) = 0;

   /**
    * Set a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("dbus if handler is responsible for this message request"),
    * @return = false: given message was not handled ("dbus if handler is not responsible for this message request")
    */
   virtual bool setSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData) = 0;

   /**
    * Send given message.
    *
    * @param[out] deleteMsg: flag indicating whether given message shall be deleted or not
    * @param[out] sendErrorMsg: flag indicating whether error message shall be generated and sent (sending failed) or not
    * @param[in] message: message
    *
    * @return = true: given message was handled ("dbus if handler is responsible for this message request"),
    * @return = false: given message was not handled ("dbus if handler is not responsible for this message request")
    */
   virtual bool sendMessage(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_BaseMessage* message) = 0;

   /**
    * Create error message.
    *
    * @param[out] ipc2BtsMessage: error message
    * @param[in] bts2IpcMessage: message failed to be sent
    * @param[in] errorCode: error code
    *
    * @return = true: given message was handled ("dbus if handler is responsible for this message request"),
    * @return = false: given message was not handled ("dbus if handler is not responsible for this message request")
    */
   virtual bool createErrorMessage(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage, IN const BTSIpcCommonErrorCode errorCode) const = 0;

   /**
    * Transfer message data.
    *
    * @param[out] ipc2BtsMessage: error message
    * @param[in] bts2IpcMessage: message failed to be sent
    *
    * @return = true: given message was handled ("dbus if handler is responsible for this message request"),
    * @return = false: given message was not handled ("dbus if handler is not responsible for this message request")
    */
   virtual bool transferMessageData(OUT Ipc2Bts_BaseMessage* ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const = 0;
};

} //btstackif

#endif //_I_DBUS_IF_HANDLER_H_
