/**
 * @file IDbusBase.h
 *
 * @par SW-Component
 * Interface
 *
 * @brief DBUS basic handling interface.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface class for basic DBUS handling.
 */

#ifndef _I_DBUS_BASE_H_
#define _I_DBUS_BASE_H_

#include "BtStackInternalTypes.h"

// class forward declarations
namespace ccdbusif {
class ICcDbusIfController;
} //ccdbusif

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
class IMainControl;
class Ipc2Bts_BaseMessage;

/**
 *
 */
class IDbusBase
{
public:
   virtual ~IDbusBase() {}

   /**
    * Sets main control (pointer).
    *
    * @param[in] mainControl: main control (pointer)
    */
   virtual void setMainControl(IN IMainControl* mainControl) = 0;

   /**
    * Sets component.
    *
    * @param[in] component: component
    */
   virtual void setComponent(IN const BTSFunctionBlock component) = 0;

   /**
    * Sets stack interface.
    *
    * @param[in] stackInterface: stack interface
    */
   virtual void setStackInterface(IN const BTSInterfaceType stackInterface) = 0;

   /**
    * Enables/disables timer handling.
    *
    * @param[in] enable: enable flag
    */
   virtual void setTimerHandling(IN const bool enable) = 0;

   /**
    * Sends a BTS 2 IPC message.
    *
    * @param[in] ptrMessage: message
    */
   virtual void sendBts2IpcMessage(IN Bts2Ipc_BaseMessage* ptrMessage) = 0;

   /**
    * Stops execution.
    */
   virtual void stop(void) = 0;

   /**
    * Starts execution.
    */
   virtual void run(void) = 0;

   /**
    * Resets handler.
    */
   virtual void resetHandler(void) = 0;

   /**
    * Sets interface for CcDbusIf controller.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] userMode: user mode
    * @param[in] controller: CcDbusIf controller (pointer)
    * @param[in] dbusInterfaces: list of needed DBUS interfaces
    * @param[in] configuration: local stack configuration data
    *
    * @return = BTS_INVALID_PARAM: at least one of the given parameters is invalid,
    * @return = BTS_ERROR: general error,
    * @return = BTS_OK: OK
    */
   virtual BTSErrorCode setCcDbusIfControllerIf(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
            IN ::ccdbusif::ICcDbusIfController* controller, IN const ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration) = 0;

   /**
    * Resets interface for CcDbusIf controller.
    */
   virtual void resetCcDbusIfControllerIf(void) = 0;

   /**
    * Handle timer tick.
    */
   virtual void handleTimerTick(void) = 0;

   /**
    * Print queue statistics.
    */
   virtual void printQueueStatistics(void) = 0;

   /**
    * Sets a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData) = 0;

   /**
    * Sets a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData) = 0;

   /**
    * Sets a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData) = 0;

   /**
    * Create service availability message.
    *
    * @param[in] interface: interface
    * @param[in] availabilityEvent: availability event
    */
   virtual void createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent) = 0;

   /**
    * Create service availability message.
    *
    * @param[in] interface: interface
    * @param[in] availabilityEvent: availability event
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] busType: bus type
    */
   virtual void createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent, IN const BTSBusName& busName, IN const BTSObjectPath& objPath, IN const BTSCommonEnumClass busType) = 0;
};

} //btstackif

#endif //_I_DBUS_BASE_H_
