/**
 * @file DbusBase.h
 *
 * @par SW-Component
 * IPC
 *
 * @brief DBUS basic handling.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Base class for DBUS handling.
 */

#ifndef _DBUS_BASE_H_
#define _DBUS_BASE_H_

#include "IDbusBase.h"
#include "FwISingleThreadFunction.h"
#include "BtsMessageQueue.h"
#include "EventQueue.h"
#include "BtsTimerMaster.h"

// class forward declarations
namespace ccdbusif {
class ICcDbusIfControllerClient;
} //ccdbusif

// class forward declarations
namespace fw {
class SingleThread;
} //fw

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
class Ipc2Bts_BaseMessage;
class IDbusIfHandler;
class DbusTimeoutControl;
class IDbusResponseTimeoutHandler;

/**
 *
 */
class DbusBase : public IDbusBase, public ::fw::ISingleThreadFunction
{
public:
   /**************************************************************************
    Constructor / destructor
    **************************************************************************/
   /**
    * Default constructor.
    */
   DbusBase();

protected:
   /**
    * Copy constructor.
    *
    * DO NOT USE!!!
    *
    * @param[in] ref: reference of object to be copied
    */
   DbusBase(const DbusBase& ref);

   /**
    * Assignment operator.
    *
    * DO NOT USE!!!
    *
    * @param[in] ref: reference of object to be assigned
    */
   DbusBase& operator=(const DbusBase& ref);

public:
   /**
    * Destructor.
    */
   virtual ~DbusBase();

   virtual void setMainControl(IN IMainControl* mainControl);

   virtual void setComponent(IN const BTSFunctionBlock component);

   virtual void setStackInterface(IN const BTSInterfaceType stackInterface);

   virtual void setTimerHandling(IN const bool enable);

   virtual void sendBts2IpcMessage(IN Bts2Ipc_BaseMessage* ptrMessage);

   virtual void stop(void);

   virtual void run(void);

   virtual void resetHandler(void);

   virtual BTSErrorCode setCcDbusIfControllerIf(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
            IN ::ccdbusif::ICcDbusIfController* controller, IN const ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration);

   virtual void resetCcDbusIfControllerIf(void);

   virtual void handleTimerTick(void);

   virtual void printQueueStatistics(void);

   virtual void setSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData);

   virtual void setSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData);

   virtual void setSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData);

   virtual void createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent) = 0;

   virtual void createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent, IN const BTSBusName& busName, IN const BTSObjectPath& objPath, IN const BTSCommonEnumClass busType) = 0;

protected:
   enum
   {
      NMB_MESSAGES_WARNING_THRESHOLD = 128
   };

   enum
   {
      MAX_NUMBER_BTS2IPC_MGS_PROCESSING = 5,
      MAX_NUMBER_IPC2BTS_MGS_PROCESSING = 5
   };

   enum
   {
      EV_TERMINATE =          0x00000001,

      EV_BTS2IPC_OUTPUT =     0x00000010
   };

   MessageQueue<Bts2Ipc_BaseMessage> _bts2IpcWaitingQueue;
   MessageQueue<Bts2Ipc_BaseMessage> _bts2IpcWorkingQueue;
   MessageQueue<Ipc2Bts_BaseMessage> _ipc2BtsCallbackQueue;
   EventQueue _eventQueue;
   bool _terminateWorkerThread;
   bool _workerThreadTerminated;
   ::fw::SingleThread* _singleThread;
   IMainControl* _mainControl;
   bool _dbusIfAvailable;
   TimerMaster _timerMaster;
   ::std::vector< IDbusIfHandler* > _configurationExtensions; /**< list of extensions TODO: replace with _extensions */
   ::std::vector< IDbusIfHandler* > _connectionExtensions; /**< list of extensions TODO: replace with _extensions */
   ::std::vector< IDbusIfHandler* > _phonecallExtensions; /**< list of extensions TODO: replace with _extensions */
   ::std::vector< IDbusIfHandler* > _phonebookExtensions; /**< list of extensions TODO: replace with _extensions */
   ::std::vector< IDbusIfHandler* > _messagingExtensions; /**< list of extensions TODO: replace with _extensions */
   ::std::vector< IDbusIfHandler* > _mediaPlayerExtensions; /**< list of extensions TODO: replace with _extensions */
   ::std::vector< IDbusIfHandler* > _extensions; /**< list of extensions */
   BTSFunctionBlock _component; /**< component (function) block => tracing only */
   // part of message opcode, equals _functionBlock in BasicControl --- BTSFunctionBlock _subComponent; /**< sub-component (function) block */
   BTSInterfaceType _stackInterface; /**< stack interface to be used (Bluetooth stack) */
   bool _timerHandlingEnabled; /**< flag indicating that timer handling is enabled */
   ::std::vector< DbusTimeoutControl* > _timeoutControlList; /**< timeout control list */
   ::ccdbusif::ICcDbusIfControllerClient* _controllerClient; /**< controller client for DBus interface */
   IDbusResponseTimeoutHandler* _responseTimeoutHandler; /**< response timeout handler */

   virtual void threadFunction(void* arguments) = 0;
   virtual void setTerminate(void* arguments) = 0;

   void startThread(void);
   void stopThread(void);
   void pushIpc2BtsMessage(IN Ipc2Bts_BaseMessage* ptrMessage, IN const bool highPrio = false);
   void sendIpc2BtsMessageList(IN ::std::vector<Ipc2Bts_BaseMessage*>& ipc2BtsMsgList);
   Bts2Ipc_BaseMessage* findAndRemoveBts2IpcWorkingMessage(IN const Ipc2Bts_BaseMessage* ptrMessage, IN const bool withLock = true);
   void findAndRemoveBts2IpcWorkingMessages(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const Ipc2Bts_BaseMessage* ptrMessage, IN const bool withLock = true);
   void removeBts2IpcWorkingMessage(IN const Bts2Ipc_BaseMessage* ptrMessage, IN const bool withLock = true);

   /**
    * Add DBus interface handler.
    *
    * @param[in] handler: DBus interface handler
    */
   void addDbusIfHandler(IN IDbusIfHandler* handler);

   /**
    * Set response timeout handler.
    *
    * @param[in] handler: response timeout handler
    */
   void setResponseTimeoutHandler(IN IDbusResponseTimeoutHandler* handler);

   /**
    * Create list with timeout control entries.
    *
    * @param[in] count: number of timeout control entries to be created
    */
   void createTimeoutControlList(IN const unsigned int count);

   /**
    * Create timeout control entry.
    *
    * @return = timeout control entry
    */
   DbusTimeoutControl* createTimeoutControlEntry(void);

   /**
    * Destroy list with timeout control entries.
    */
   void destroyTimeoutControlList(void);

   /**
    * Stop all timer part of timeout control list.
    */
   void stopAllTimeoutControlEntries(void);

   /**
    * Get timeout control entry.
    *
    * @return = timeout control entry
    */
   DbusTimeoutControl* getTimeoutControlEntry(void);

   /**
    * Release timeout control entry.
    *
    * @param[in] ptrMessage: message
    */
   void releaseTimeoutControlEntry(IN const Bts2Ipc_BaseMessage* ptrMessage);

   /**
    * Release all timeout control entries.
    */
   void releaseAllTimeoutControlEntries(void);

   /**
    * Handle successful sending of a message.
    *
    * @param[in] ptrMessage: message
    */
   void handleSendingSuccess(IN Bts2Ipc_BaseMessage* ptrMessage);

   /**
    * Execute set interface for CcDbusIf controller.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] userMode: user mode
    * @param[in] controller: CcDbusIf controller (pointer)
    * @param[in] dbusInterfaces: list of needed DBUS interfaces
    * @param[in] configuration: local stack configuration data
    *
    * @return = BTS_INVALID_PARAM: at least one of the given parameters is invalid,
    * @return = BTS_ERROR: general error,
    * @return = BTS_OK: OK
    */
   BTSErrorCode executeSetCcDbusIfControllerIf(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
            IN ::ccdbusif::ICcDbusIfController* controller, IN const ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration);

   /**
    * Execute reset interface for CcDbusIf controller.
    */
   void executeResetCcDbusIfControllerIf(void);

   /**
    * Execute a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("dbus if handler is responsible for this message request"),
    * @return = false: given message was not handled ("dbus if handler is not responsible for this message request")
    */
   bool executeSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData);

   /**
    * Execute a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("dbus if handler is responsible for this message request"),
    * @return = false: given message was not handled ("dbus if handler is not responsible for this message request")
    */
   bool executeSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData);

   /**
    * Execute a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("dbus if handler is responsible for this message request"),
    * @return = false: given message was not handled ("dbus if handler is not responsible for this message request")
    */
   bool executeSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData);

   /**
    * Process given message.
    *
    * @param[out] deleteMsg: flag indicating whether given message shall be deleted or not
    * @param[out] sendErrorMsg: flag indicating whether error message shall be generated and sent (sending failed) or not
    * @param[in] message: message
    *
    * @return = true: given message was handled ("dbus if handler is responsible for this message request"),
    * @return = false: given message was not handled ("dbus if handler is not responsible for this message request")
    */
   bool processMessage(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_BaseMessage* message);

   /**
    * Create error message.
    *
    * @param[out] ipc2BtsMessage: error message
    * @param[in] bts2IpcMessage: message failed to be sent
    * @param[in] errorCode: error code
    *
    * @return = true: given message was handled ("dbus if handler is responsible for this message request"),
    * @return = false: given message was not handled ("dbus if handler is not responsible for this message request")
    */
   bool createErrorMessage(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage, IN const BTSIpcCommonErrorCode errorCode) const;

   /**
    * Transfer message data.
    *
    * @param[out] ipc2BtsMessage: error message
    * @param[in] bts2IpcMessage: message failed to be sent
    *
    * @return = true: given message was handled ("dbus if handler is responsible for this message request"),
    * @return = false: given message was not handled ("dbus if handler is not responsible for this message request")
    */
   bool transferMessageData(OUT Ipc2Bts_BaseMessage* ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const;

   /**
    * Destroy dbus if handler list.
    */
   void destroyDbusIfHandlerList(void);

private:
   void cleanBts2IpcWorkingQueue(IN Bts2Ipc_BaseMessage* ptrMessage, IN const bool withLock = true);
};

} //btstackif

#endif //_DBUS_BASE_H_
