/**
 * @file SppTestApp.cpp
 *
 * @par SW-Component
 * Test
 *
 * @brief SPP test application.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details SPP test application.
 */

#include <cstdio>
#include <cstring>
#include <cerrno>
#include <pthread.h>
#include <unistd.h>
#include <sys/prctl.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <poll.h>
#include <termios.h>

#include "SppTestApp.h"
#include "TraceBase.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/SppTestApp.cpp.trc.h"
#endif
#endif

namespace btstackif {

SppTestApp::SppTestApp() :
_defaultFd(-1),
_defaultTimeout(50)
{
   _fd = _defaultFd;
   // _lock
   _terminate = false;
   _send = true;
   _rawMode = false;
   _toggle = true;
   _dataMode = true;
   _sendLoopCounter = 0;
   _maxSendLoopCounter = 0;
}

SppTestApp::~SppTestApp()
{
}

SppTestApp& SppTestApp::getInstance(void)
{
   static SppTestApp instance;
   return instance;
}

void SppTestApp::openDevice(const ::std::string& device, const unsigned int nmbSendingLoops /*= 0*/, const unsigned int delayBetweenSending /*= 50*/)
{
   ETG_TRACE_USR1((" open(): device=%100s nmbSendingLoops=%u delayBetweenSending=%u", device.c_str(), nmbSendingLoops, delayBetweenSending));

   if(device.empty())
   {
      return;
   }

   _sendLoopCounter = 0;
   _maxSendLoopCounter = nmbSendingLoops;
   _defaultTimeout = delayBetweenSending;

   _send = true;

   _lock.lock();

   if(_defaultFd == _fd)
   {
      int errorCode;

      _fd = open(device.c_str(), O_RDWR | O_NOCTTY | O_SYNC);

      if(0 > _fd)
      {
         _fd = _defaultFd;
      }

      if(_defaultFd == _fd)
      {
         errorCode = errno;
         ETG_TRACE_ERR((" open(): open failed: ERROR=%d (%s)", errorCode, strerror(errorCode)));
      }
      else
      {
         ETG_TRACE_USR1((" open(): open success"));

         bool goOn = true;

         if(true == _rawMode)
         {
            ETG_TRACE_USR1((" open(): configure raw mode"));

            struct termios terminalParams;
            if(0 > tcgetattr(_fd, &terminalParams))
            {
               errorCode = errno;
               ETG_TRACE_ERR((" open(): tcgetattr failed: ERROR=%d (%s)", errorCode, strerror(errorCode)));
               goOn = false;
            }
            else
            {
               // configure character device:
               // control characters: MIN value
               terminalParams.c_cc[VMIN] = 1;
               // control characters: TIME value
               terminalParams.c_cc[VTIME] = 0;
               // local modes: disable echo, disable canonical input (erase and kill processing), disable signals, disable echo erase character as error-correcting backspace, disable echo kill, disable echo NL
               terminalParams.c_lflag &= ~(ECHO | ICANON | ISIG | ECHOE | ECHOK | ECHONL);
               // output modes: disable post-process output
               terminalParams.c_oflag &= ~(OPOST);

               // flush both data received but not read and data written but not transmitted
               (void)tcflush(_fd, TCIOFLUSH); // ignore return value

               // set raw mode:
               /* cfmakeraw() sets the terminal to something like the "raw" mode of the old Version 7 terminal driver: input is available character by character,
                * echoing is disabled, and all special processing of terminal input and output characters is disabled. The terminal attributes are set as follows:
                  termios_p->c_iflag &= ~(IGNBRK | BRKINT | PARMRK | ISTRIP | INLCR | IGNCR | ICRNL | IXON);
                  termios_p->c_oflag &= ~OPOST;
                  termios_p->c_lflag &= ~(ECHO | ECHONL | ICANON | ISIG | IEXTEN);
                  termios_p->c_cflag &= ~(CSIZE | PARENB);
                  termios_p->c_cflag |= CS8;
                */
               cfmakeraw(&terminalParams);

               // set parameters associated with the terminal
               if(0 > tcsetattr(_fd, TCSANOW, &terminalParams))
               {
                  errorCode = errno;
                  ETG_TRACE_ERR((" open(): tcsetattr failed: ERROR=%d (%s)", errorCode, strerror(errorCode)));
                  // just print the error and continue
               }
            }
         }

         if(true == goOn)
         {
            _terminate = false;

            pthread_t threadHandle;
            int result = pthread_create(&threadHandle, NULL /* no attributes */, &deviceHandler, (void*)this);

            if(0 == result)
            {
               ETG_TRACE_USR1((" open(): pthread_create success"));
            }
            else
            {
               ETG_TRACE_ERR((" open(): pthread_create failed: ERROR=%d (%s)", result, strerror(result)));
               goOn = false;
            }
         }

         if(false == goOn)
         {
            close(_fd);
            _fd = _defaultFd;
         }
      }
   }

   _lock.unlock();
}

void SppTestApp::closeDevice(void)
{
   ETG_TRACE_USR1((" close()"));

   _lock.lock();

   _terminate = true;

   if(_defaultFd != _fd)
   {
      close(_fd);
      _fd = _defaultFd;
   }

   _lock.unlock();
}

void* SppTestApp::deviceHandler(void* arg)
{
   SppTestApp* own = (SppTestApp*)arg;

   ThreadInfo threadInfo;
   ETG_TRACE_USR1((" deviceHandler(): enter --- %s", threadInfo.getInfo()));

   if(own)
   {
      bool goon = true;
      int result;
      int errorCode;
      struct pollfd fds[1];
      unsigned char buffer[1024];

      fds[0].fd = own->_fd;
      fds[0].events = POLLIN | POLLERR | POLLHUP | POLLNVAL;

      while((false == own->_terminate) && (true == goon))
      {
         result = poll(fds, 1, own->_defaultTimeout);

         if(0 > result)
         {
            errorCode = errno;
            ETG_TRACE_ERR((" deviceHandler(): poll failed: ERROR=%d (%s)", errorCode, strerror(errorCode)));
         }
         else if(0 == result)
         {
            // timeout => send data
            bool sendData = false;

            if(0 == own->_maxSendLoopCounter)
            {
               sendData = true;
            }
            else if(own->_sendLoopCounter < own->_maxSendLoopCounter)
            {
               sendData = true;
               ++own->_sendLoopCounter;
            }
            else
            {
               // maximum sending loops reached
               ETG_TRACE_USR1((" deviceHandler(): maximum sending loops reached"));
               goon = false;
            }

            if(true == sendData)
            {
               own->sendLargeData();
            }
         }
         else
         {
            // event on fds occurred
            if(POLLIN & fds[0].revents)
            {
               // data received
               ssize_t nmb = read(own->_fd, buffer, sizeof(buffer));

               if(0 > nmb)
               {
                  errorCode = errno;
                  ETG_TRACE_ERR((" deviceHandler(): read failed: ERROR=%d (%s)", errorCode, strerror(errorCode)));
               }
               else
               {
                  // no data or data
                  own->printData(buffer, (size_t)nmb);
               }
            }

            if(POLLERR & fds[0].revents)
            {
               // error => continue
               ETG_TRACE_ERR((" deviceHandler(): POLLERR received"));
            }

            if(POLLHUP & fds[0].revents)
            {
               // device disconnected
               ETG_TRACE_ERR((" deviceHandler(): POLLHUP received"));
               goon = false;
            }

            if(POLLNVAL & fds[0].revents)
            {
               // invalid file descriptor; observation: received after device was closed
               ETG_TRACE_ERR((" deviceHandler(): POLLNVAL received"));
               goon = false;
            }
         }
      }

      own->_lock.lock();

      if(own->_defaultFd != own->_fd)
      {
         close(own->_fd);
         own->_fd = own->_defaultFd;
      }

      own->_lock.unlock();
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   ETG_TRACE_USR1((" deviceHandler(): exit"));

   return NULL;
}

void SppTestApp::printData(unsigned char* buffer, size_t length)
{
   if(NULL == buffer)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   const size_t max = 16;
   char printBuffer[240];
   size_t rows = (size_t)(length / max);
   size_t remain = (size_t)(length % max);
   size_t c = 0;
   size_t d;
   size_t pc;
   int p;

   ETG_TRACE_USR1((" printData(): read data: size=%u", length));

   for(size_t i = 0; i < rows; i++)
   {
      pc = 0;
      d = c;
      for(size_t j = 0; j < max; j++)
      {
         p = snprintf(&printBuffer[pc], (sizeof(printBuffer) - pc), "%02X ", buffer[d]);
         if(0 <= p)
         {
            pc += p;
            d++;
         }
         else
         {
            ETG_TRACE_ERR((" printData(): snprintf failed"));
         }
      }

      p = snprintf(&printBuffer[pc], (sizeof(printBuffer) - pc), "        ");
      if(0 <= p)
      {
         pc += p;
      }
      else
      {
         ETG_TRACE_ERR((" printData(): snprintf failed"));
      }

      d = c;
      for(size_t j = 0; j < max; j++)
      {
         char character = (char)buffer[d];

         if((' ' <= character) && (character <= '~'))
         {
            // printable character
         }
         else
         {
            // control character
            character = '.';
         }

         p = snprintf(&printBuffer[pc], (sizeof(printBuffer) - pc), "%c", character);
         if(0 <= p)
         {
            pc += p;
            d++;
         }
         else
         {
            ETG_TRACE_ERR((" printData(): snprintf failed"));
         }
      }

      c += max;

      ETG_TRACE_USR1((" printData(): %s", printBuffer));
   }

   if(0 < remain)
   {
      pc = 0;
      d = c;
      for(size_t j = 0; j < remain; j++)
      {
         p = snprintf(&printBuffer[pc], (sizeof(printBuffer) - pc), "%02X ", buffer[d]);
         if(0 <= p)
         {
            pc += p;
            d++;
         }
         else
         {
            ETG_TRACE_ERR((" printData(): snprintf failed"));
         }
      }

      for(size_t j = remain; j < max; j++)
      {
         p = snprintf(&printBuffer[pc], (sizeof(printBuffer) - pc), "   ");
         if(0 <= p)
         {
            pc += p;
         }
         else
         {
            ETG_TRACE_ERR((" printData(): snprintf failed"));
         }
      }

      p = snprintf(&printBuffer[pc], (sizeof(printBuffer) - pc), "        ");
      if(0 <= p)
      {
         pc += p;
      }
      else
      {
         ETG_TRACE_ERR((" printData(): snprintf failed"));
      }

      d = c;
      for(size_t j = 0; j < remain; j++)
      {
         char character = (char)buffer[d];

         if((' ' <= character) && (character <= '~'))
         {
            // printable character
         }
         else
         {
            // control character
            character = '.';
         }

         p = snprintf(&printBuffer[pc], (sizeof(printBuffer) - pc), "%c", character);
         if(0 <= p)
         {
            pc += p;
            d++;
         }
         else
         {
            ETG_TRACE_ERR((" printData(): snprintf failed"));
         }
      }

      for(size_t j = remain; j < max; j++)
      {
         p = snprintf(&printBuffer[pc], (sizeof(printBuffer) - pc), " ");
         if(0 <= p)
         {
            pc += p;
         }
         else
         {
            ETG_TRACE_ERR((" printData(): snprintf failed"));
         }
      }

      ETG_TRACE_USR1((" printData(): %s", printBuffer));
   }

   ETG_TRACE_USR1((" printData(): read data: ***end***"));
}

void SppTestApp::sendData(void)
{
   if(false == _send)
   {
      // sending is disabled
      return;
   }

   // The names "John Doe" or "John Roe" for men, "Jane Doe" or "Jane Roe" for women, or "Johnnie Doe" and "Janie Doe" for children, or just "Doe" non-gender-specifically are used as placeholder names for a party whose true identity is unknown or must be withheld in a legal action, case, or discussion.
   const char* dataText = "The names \"John Doe\" or \"John Roe\" for men, \"Jane Doe\" or \"Jane Roe\" for women, or \"Johnnie Doe\" and \"Janie Doe\" for children, or just \"Doe\" non-gender-specifically are used as placeholder names for a party whose true identity is unknown or must be withheld in a legal action, case, or discussion.";
   const size_t lengthText = strlen(dataText);

   const size_t lengthBinary = 256;
   unsigned char dataBinary[lengthBinary];
   for(size_t i = 0; i < lengthBinary; i++)
   {
      dataBinary[i] = (unsigned char)i;
   }

   const unsigned char* data = (const unsigned char*)dataText;
   size_t length = lengthText;

   if(true == _toggle)
   {
      if(true == _dataMode)
      {
         data = (const unsigned char*)dataText;
         length = lengthText;
         _dataMode = false;
      }
      else
      {
         data = dataBinary;
         length = lengthBinary;
         _dataMode = true;
      }
   }

   const size_t max = 40;
   size_t rows = (size_t)(length / max);
   size_t remain = (size_t)(length % max);
   size_t c = 0;
   ssize_t nmb;
   int errorCode;

   ETG_TRACE_USR1((" sendData(): send data: size=%u", length));

   for(size_t i = 0; i < rows; i++)
   {
      nmb = write(_fd, (const void *)(&data[c]), max);

      if(0 > nmb)
      {
         errorCode = errno;
         ETG_TRACE_ERR((" sendData(): write failed: ERROR=%d (%s)", errorCode, strerror(errorCode)));
      }
      else
      {
         // no data or data
         ETG_TRACE_USR1((" sendData(): %d bytes sent", nmb));
      }

      c += max;
   }

   // if(0 < remain)
   {
      nmb = write(_fd, (const void *)(&data[c]), remain);

      if(0 > nmb)
      {
         errorCode = errno;
         ETG_TRACE_ERR((" sendData(): write failed: ERROR=%d (%s)", errorCode, strerror(errorCode)));
      }
      else
      {
         // no data or data
         ETG_TRACE_USR1((" sendData(): %d bytes sent", nmb));
      }
   }

   ETG_TRACE_USR1((" sendData(): send data: ***end***"));
}

void SppTestApp::sendLargeData(void)
{
   if(false == _send)
   {
      // sending is disabled
      return;
   }

   const size_t lengthText(20480);
   unsigned char dataBinary[lengthText + 1];
   const unsigned char startCharCounter(32); // space
   const unsigned char endCharCounter(126); // tilde
   unsigned char charCounter(startCharCounter);

   for(size_t i = 0; i < lengthText; i++)
   {
      dataBinary[i] = charCounter;
      ++charCounter;
      if(endCharCounter <= charCounter)
      {
         charCounter = startCharCounter;
      }
   }

   dataBinary[lengthText] = 0;

   sendMessage(_fd, lengthText, dataBinary);
}

void SppTestApp::sendMessage(int deviceFd, const size_t msgLenTotal, const unsigned char* msg)
{
   ETG_TRACE_USR1(("sendMessage(): write request received with %u", msgLenTotal));

   size_t byteCounter(0);
   ssize_t nmb;
   size_t retryCount(0);
   const size_t maxRetryCount(3);

   while(byteCounter < msgLenTotal)
   {
      nmb = write(deviceFd, (msg + byteCounter), (msgLenTotal - byteCounter));

      if(0 <= nmb)
      {
         ETG_TRACE_USR1(("sendMessage(): sent data %u of %u", (size_t)nmb, msgLenTotal));
         byteCounter += (size_t)nmb;
      }
      else
      {
         const int errorCode(errno);
         ETG_TRACE_USR1(("sendMessage(): errno: %d %s", errorCode, strerror(errorCode)));

         if((EAGAIN == errorCode) && (retryCount < maxRetryCount))
         {
            ETG_TRACE_USR1(("sendMessage(): retryCount %u", retryCount));
            retryCount++;
            usleep(10000); // retry after 10 msec
         }
         else
         {
            ETG_TRACE_USR1(("sendMessage(): ERROR: write request but device is not longer connected"));
            break;
         }
      }
   }

   ETG_TRACE_USR1(("sendMessage(): write request returned"));
}

} //btstackif
