/**
 * @file ExtendedTimerEntry.cpp
 *
 * @par SW-Component
 * Timer
 *
 * @brief Extended timer entry.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Extended timer entry.
 */

#include "ExtendedTimerEntry.h"
#include "ITimerPool.h"
#include "IExtendedTimer.h"
#include "FwAssert.h"

namespace btstackif {

ExtendedTimerEntry::ExtendedTimerEntry() :
_timerPool(0),
_timer(0),
_inUse(false)
{
}

ExtendedTimerEntry::~ExtendedTimerEntry()
{
   _timerPool = 0;
   _timer = 0;
}

ExtendedTimerEntry::ExtendedTimerEntry(const ExtendedTimerEntry& ref)
{
   _timerPool = 0;
   setTimerPool(ref._timerPool);
   _timer = 0;
   setTimer(ref._timer);
   _inUse = ref._inUse;
}

ExtendedTimerEntry& ExtendedTimerEntry::operator=(const ExtendedTimerEntry& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   _timerPool = 0;
   setTimerPool(ref._timerPool);
   _timer = 0;
   setTimer(ref._timer);
   _inUse = ref._inUse;

   return *this;
}

bool ExtendedTimerEntry::operator==(const ExtendedTimerEntry& ref) const
{
   bool result = true;

   result = (true == result) && (_timerPool == ref._timerPool);
   result = (true == result) && (_timer == ref._timer);
   result = (true == result) && (_inUse == ref._inUse);

   return result;
}

bool ExtendedTimerEntry::operator!=(const ExtendedTimerEntry& ref) const
{
   return !(operator==(ref));
}

void ExtendedTimerEntry::start(IN IBasicControl* control, IN IExtendedTimeoutHandler* handler, IN const BTSTimeValue timeout)
{
   FW_IF_NULL_PTR_RETURN(_timerPool);
   FW_IF_NULL_PTR_RETURN(control);
   FW_IF_NULL_PTR_RETURN(handler);

   // check if timer was already allocated
   if(0 == _timer)
   {
      // get new timer
      _timer = _timerPool->getExtendedTimer();
   }

   FW_IF_NULL_PTR_RETURN(_timer);

   _timer->start(control, handler, timeout);
}

void ExtendedTimerEntry::stop(void)
{
   // check if timer was already allocated
   if(0 == _timer)
   {
      // no timer => nothing to do
      return;
   }

   _timer->stop();

   // keep timer
}

void ExtendedTimerEntry::release(void)
{
   // check if timer was already allocated
   if(0 == _timer)
   {
      // no timer => nothing to do
      return;
   }

   // ensure that timer is stopped
   _timer->stop();

   FW_IF_NULL_PTR_RETURN(_timerPool);

   // release timer
   _timerPool->releaseExtendedTimer(_timer);

   // reset timer
   _timer = 0;
}

bool ExtendedTimerEntry::isActive(void) const
{
   // check if timer was already allocated
   if(0 == _timer)
   {
      // no timer
      return false;
   }

   return _timer->isActive();
}

bool ExtendedTimerEntry::compare(IN const BTSTimerId timerId) const
{
   if(0 == _timer)
   {
      // no timer => compare not possible
      return false;
   }

   if(0 == _timer->getTimerId())
   {
      // timer was never started (no id assigned) => compare not possible
      return false;
   }

   if(timerId == _timer->getTimerId())
   {
      // timer id matches
      return true;
   }
   else
   {
      // timer id mismatch
      return false;
   }
}

} //btstackif
