/**
 * @file EventQueue.cpp
 *
 * @par SW-Component
 * Base
 *
 * @brief Event queue.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation for event queue handling.
 */

#include "EventQueue.h"

#include <cstdlib>
#include <cstring>
#include <cerrno>

#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EventQueue.cpp.trc.h"
#endif
#endif

namespace btstackif {

EventQueue::EventQueue()
{
   _eventMask = 0;
   memset(&_semaphore, 0, sizeof(_semaphore));
   if(-1 == sem_init(&_semaphore, 0, 0))
   {
      // check result for debugging purpose

      /*
       * from http://pubs.opengroup.org/onlinepubs/009695399/functions/sem_init.html:
         The sem_init() function shall fail if:

         [EINVAL]
            The value argument exceeds {SEM_VALUE_MAX}.
            => value is 0 and cannot exceed SEM_VALUE_MAX
         [ENOSPC]
            A resource required to initialize the semaphore has been exhausted, or the limit on semaphores ( {SEM_NSEMS_MAX}) has been reached.
            => should never happen
         [EPERM]
            The process lacks the appropriate privileges to initialize the semaphore.
            => should never happen else there is a general problem
         [ENOSYS]
            pshared is nonzero, but the system does not support process-shared semaphores
            => pshared is 0
       *
       */

      int errNumber = errno;
      ETG_TRACE_ERR((" EventQueue(): sem_init failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
      FW_NORMAL_ASSERT_ALWAYS();
   }
   // _lock needs no explicit initialization
}

EventQueue::~EventQueue()
{
   (void)sem_destroy(&_semaphore); // ignore return value
}

void EventQueue::post(IN const BTSEventMask eventMask)
{
   _lock.lock();

   _eventMask |= eventMask;

   _lock.unlock();

   /* HINT: next part was never tested!!!
   int semCount = 0;
   if(-1 == sem_getvalue(&_semaphore, &semCount))
   {
      int errNumber = errno;
      ETG_TRACE_ERR((" post(): sem_init failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
      FW_NORMAL_ASSERT_ALWAYS();
      semCount = 0;
   }
   else
   {
      ETG_TRACE_USR1((" post(): semCount=%d", semCount));
   }

   if(2 > semCount) */
   {
      if(-1 == sem_post(&_semaphore))
      {
         // check result for debugging purpose

         /*
          * from http://pubs.opengroup.org/onlinepubs/009695399/functions/sem_post.html:
            The sem_post() function may fail if:

            [EINVAL]
               The sem argument does not refer to a valid semaphore.
               => a valid semaphore is provided
            [EOVERFLOW]
               The maximum allowable value for a semaphore would be exceeded.
               => should never happen
          *
          */

         int errNumber = errno;
         ETG_TRACE_ERR((" post(): sem_post failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }
}

BTSErrorCode EventQueue::wait(OUT BTSEventMask& eventMask, IN const BTSTimeValue timeoutInMs)
{
   (void)(timeoutInMs); // currently ignored

   int result = -1;

   // old: while(1)
   for(;;)
   {
      result = sem_wait(&_semaphore);
      if(-1 == result)
      {
         /*
          * from http://pubs.opengroup.org/onlinepubs/009695399/functions/sem_wait.html:
            The sem_trywait() and sem_wait() functions shall fail if:

            [EAGAIN]
               The semaphore was already locked, so it cannot be immediately locked by the sem_trywait() operation ( sem_trywait() only).
               => valid for sem_trywait() only, no need to handle

            The sem_trywait() and sem_wait() functions may fail if:

            [EDEADLK]
               A deadlock condition was detected.
               => handle as error
            [EINTR]
               A signal interrupted this function.
               => handle as normal behavior, do a while loop
            [EINVAL]
               The sem argument does not refer to a valid semaphore.
               => a valid semaphore is provided
            [ETIMEDOUT]
               The call timed out before the semaphore could be locked.
               => valid for sem_timedwait() only, no need to handle
          *
          */

         int errNumber = errno;

         if(EINTR != errNumber)
         {
            // error
            ETG_TRACE_ERR((" wait(): sem_wait failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
            FW_NORMAL_ASSERT_ALWAYS();
            break;
         }
      }
      else
      {
         // success
         break;
      }
   }

   if(-1 == result)
   {
      // do not touch eventMask
   }
   else
   {
      _lock.lock();

      eventMask |= _eventMask;
      _eventMask = 0;

      _lock.unlock();

      return BTS_OK;
   }

   return BTS_ERROR;
}

void EventQueue::clear(void)
{
   _lock.lock();

   _eventMask = 0;

   _lock.unlock();
}

} //btstackif
