/**
 * @file BtsUtils.cpp
 *
 * @par SW-Component
 * Utilities
 *
 * @brief Utilities.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Utilities.
 */

#include "BtsUtils.h"

#include <cstring>
#include <cstdlib>
#include <cstdio>
#include <sys/types.h>
#include <dirent.h>
#include <unistd.h>

#include "FwStringUtils.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BtsUtils.cpp.trc.h"
#endif
#endif

namespace btstackif {

static const BTSBDAddress bdAddressExample = "aabbccddeeff";
// static const ::std::string bdAddressFormat = "%02X%02X%02X%02X%02X%02X";
static const BTSLinkKey linkKeyExample = "aabbccddeeff00112233445566778899";
static const BTSDLinkKey dLinkKeyExample = "aabbccddeeff00112233445566778899aabbccddeeff00112233445566778899";
static const BTSPinCode pinCodeExample = "0123456789012345";
static const BTSNumericValue numericValueExample = "012345";
static const ::std::string numericValueFormat = "%06u";
static const BTSUuid uuid128 = "0000000000001000800000805F9B34FB"; // UUID 128 bit, max size
static const BTSUuid uuid32  = "00000000"; // UUID 32 bit
static const BTSUuid uuid16  = "0000"; // UUID 16 bit
static const BTSUserId userIdExample = "01234567890123456789";

void convertBinary2HexStringValue(OUT ::std::string& stringValue, IN const ::std::vector< unsigned char >& binaryValue, IN const ::std::string& example, IN const bool allowEmpty /*= false*/)
{
   if((binaryValue.size() << 1) == example.size())
   {
      char buffer[1 + example.size()];
      memset(buffer, 0, sizeof(buffer));
      char hexString[3];
      for(size_t i = 0; i < binaryValue.size(); i++)
      {
         convertBinary2HexString(hexString, binaryValue[i]);
         buffer[2 * i    ] = hexString[0];
         buffer[2 * i + 1] = hexString[1];
      }
      stringValue = buffer;
   }
   else
   {
      // wrong size
      if((0 == binaryValue.size()) && (true == allowEmpty))
      {
         // empty value allowed
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }
}

void convertHexString2BinaryValue(OUT ::std::vector< unsigned char >& binaryValue, IN const ::std::string& stringValue, IN const ::std::string& example, IN const bool allowEmpty /*= false*/)
{
   if(example.size() == stringValue.size())
   {
      size_t binaryCounter = 0;
      size_t stringCounter = 0;
      char tmpBuffer[3];

      tmpBuffer[2] = '\0';

      binaryValue.reserve((example.size() >> 1));

      while(((stringCounter + 1) < example.size()) && (binaryCounter < binaryValue.capacity()))
      {
         tmpBuffer[0] = stringValue[stringCounter];
         stringCounter++;
         tmpBuffer[1] = stringValue[stringCounter];
         stringCounter++;
         binaryValue.push_back(convertHexString2Binary(tmpBuffer));
         binaryCounter++;
      }
   }
   else
   {
      // wrong size
      if((0 == stringValue.size()) && (true == allowEmpty))
      {
         // empty value allowed
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }
}

void convertBdAddressString2Binary(OUT BTSBinaryBDAddress& binaryDeviceAddress, IN const BTSBDAddress& stringDeviceAddress, IN const bool allowEmpty /*= false*/)
{
   if(bdAddressExample.size() == stringDeviceAddress.size())
   {
      convertHexString2BinaryValue(binaryDeviceAddress.getData(), stringDeviceAddress, bdAddressExample);

      // old implementation => keep it
#if 0
      size_t binaryCounter = 0;
      size_t stringCounter = 0;
      char tmpBuffer[3];

      tmpBuffer[2] = '\0';

      while(((stringCounter + 1) < bdAddressExample.size()) && (binaryCounter < binaryDeviceAddress.capacity()))
      {
         tmpBuffer[0] = stringDeviceAddress[stringCounter];
         stringCounter++;
         tmpBuffer[1] = stringDeviceAddress[stringCounter];
         stringCounter++;
         binaryDeviceAddress.push_back(convertHexString2Binary(tmpBuffer));
         binaryCounter++;
      }
#endif
   }
   else
   {
      // wrong size
      if(false == allowEmpty)
      {
         FW_NORMAL_ASSERT_ALWAYS();
      }

      // fill dummy data
      for(size_t i = 0; i < binaryDeviceAddress.getMinSize(); i++)
      {
         binaryDeviceAddress.push_back(0);
      }
   }
}

void convertBdAddressBinary2String(OUT BTSBDAddress& stringDeviceAddress, IN const BTSBinaryBDAddress& binaryDeviceAddress)
{
   convertBdAddressBinary2String(stringDeviceAddress, binaryDeviceAddress.getData());
   ::fw::convertString2LowerCase(stringDeviceAddress);
}

void convertBdAddressBinary2String(OUT BTSBDAddress& stringDeviceAddress, IN const ::std::vector< unsigned char >& binaryDeviceAddress)
{
   if((binaryDeviceAddress.size() << 1) == bdAddressExample.size())
   {
      convertBinary2HexStringValue(stringDeviceAddress, binaryDeviceAddress, bdAddressExample);
      ::fw::convertString2LowerCase(stringDeviceAddress);
   }
   else
   {
      // wrong size
      FW_NORMAL_ASSERT_ALWAYS();

      // set dummy data
      stringDeviceAddress = BTS_DUMMY_BDADDRESS;
   }
}

unsigned char convertHexString2Binary(IN const char* hexString)
{
   unsigned char binaryVal = 0;

   if((hexString) && (2 == strlen(hexString)))
   {
      unsigned char lowNibble = 0;
      unsigned char highNibble = 0;

      if(('0' <= hexString[0]) && (hexString[0] <= '9'))
      {
         // numeric character
         highNibble = (unsigned char)(((unsigned char)(hexString[0] - '0')) << 4);
      }
      else if(('a' <= hexString[0]) && (hexString[0] <= 'f'))
      {
         // alphabetic character
         highNibble = (unsigned char)(((unsigned char)((hexString[0] - 'a') + 0x0A)) << 4);
      }
      else if(('A' <= hexString[0]) && (hexString[0] <= 'F'))
      {
         // alphabetic character
         highNibble = (unsigned char)(((unsigned char)((hexString[0] - 'A') + 0x0A)) << 4);
      }
      else
      {
         // invalid character
      }

      if(('0' <= hexString[1]) && (hexString[1] <= '9'))
      {
         // numeric character
         lowNibble = (unsigned char)(((unsigned char)(hexString[1] - '0')));
      }
      else if(('a' <= hexString[1]) && (hexString[1] <= 'f'))
      {
         // alphabetic character
         lowNibble = (unsigned char)(((unsigned char)((hexString[1] - 'a') + 0x0A)));
      }
      else if(('A' <= hexString[1]) && (hexString[1] <= 'F'))
      {
         // alphabetic character
         lowNibble = (unsigned char)(((unsigned char)((hexString[1] - 'A') + 0x0A)));
      }
      else
      {
         // invalid character
      }

      binaryVal = highNibble | lowNibble;
   }

   return binaryVal;
}

void convertBinary2HexString(OUT char* hexString, IN const unsigned char binaryVal)
{
   if(NULL == hexString)
   {
      // NULL pointer
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   unsigned char lowNibble = binaryVal & 0x0F;
   unsigned char highNibble = (binaryVal >> 4) & 0x0F;

   switch(highNibble)
   {
      case 0x00:
         hexString[0] = '0';
         break;
      case 0x01:
         hexString[0] = '1';
         break;
      case 0x02:
         hexString[0] = '2';
         break;
      case 0x03:
         hexString[0] = '3';
         break;
      case 0x04:
         hexString[0] = '4';
         break;
      case 0x05:
         hexString[0] = '5';
         break;
      case 0x06:
         hexString[0] = '6';
         break;
      case 0x07:
         hexString[0] = '7';
         break;
      case 0x08:
         hexString[0] = '8';
         break;
      case 0x09:
         hexString[0] = '9';
         break;
      case 0x0A:
         hexString[0] = 'A';
         break;
      case 0x0B:
         hexString[0] = 'B';
         break;
      case 0x0C:
         hexString[0] = 'C';
         break;
      case 0x0D:
         hexString[0] = 'D';
         break;
      case 0x0E:
         hexString[0] = 'E';
         break;
      case 0x0F:
         hexString[0] = 'F';
         break;
      default:
         break;
   }

   switch(lowNibble)
   {
      case 0x00:
         hexString[1] = '0';
         break;
      case 0x01:
         hexString[1] = '1';
         break;
      case 0x02:
         hexString[1] = '2';
         break;
      case 0x03:
         hexString[1] = '3';
         break;
      case 0x04:
         hexString[1] = '4';
         break;
      case 0x05:
         hexString[1] = '5';
         break;
      case 0x06:
         hexString[1] = '6';
         break;
      case 0x07:
         hexString[1] = '7';
         break;
      case 0x08:
         hexString[1] = '8';
         break;
      case 0x09:
         hexString[1] = '9';
         break;
      case 0x0A:
         hexString[1] = 'A';
         break;
      case 0x0B:
         hexString[1] = 'B';
         break;
      case 0x0C:
         hexString[1] = 'C';
         break;
      case 0x0D:
         hexString[1] = 'D';
         break;
      case 0x0E:
         hexString[1] = 'E';
         break;
      case 0x0F:
         hexString[1] = 'F';
         break;
      default:
         break;
   }

   hexString[2] = '\0';
}

bool isValidBdAddress(IN const BTSBDAddress& address, IN const bool allowEmpty /*= true*/)
{
   // from interface point of view an empty BT address is also valid in some cases
   if((true == allowEmpty) && (0 == address.size()))
   {
      return true;
   }

   if(bdAddressExample.size() != address.size())
   {
      return false;
   }

   for(size_t i = 0; i < address.size(); i++)
   {
      if(('0' <= address[i]) && (address[i] <= '9'))
      {
         // numeric character
      }
      else if(('a' <= address[i]) && (address[i] <= 'f'))
      {
         // alphabetic character
      }
      else if(('A' <= address[i]) && (address[i] <= 'F'))
      {
         // alphabetic character
      }
      else
      {
         // invalid character
         return false;
      }
   }

   return true;
}

bool isValidLinkKey(IN const BTSLinkKey& linkKey)
{
   // from interface point of view an empty link key is also valid
   if((0 != linkKey.size()) && (linkKeyExample.size() != linkKey.size()))
   {
      return false;
   }

   for(size_t i = 0; i < linkKey.size(); i++)
   {
      if(('0' <= linkKey[i]) && (linkKey[i] <= '9'))
      {
         // numeric character
      }
      else if(('a' <= linkKey[i]) && (linkKey[i] <= 'f'))
      {
         // alphabetic character
      }
      else if(('A' <= linkKey[i]) && (linkKey[i] <= 'F'))
      {
         // alphabetic character
      }
      else
      {
         // invalid character
         return false;
      }
   }

   return true;
}

bool isValidPinCode(IN const BTSPinCode& pinCode)
{
   if(pinCodeExample.size() < pinCode.size())
   {
      return false;
   }

   return true;
}

bool isValidNumericValue(IN const BTSNumericValue& numericValue)
{
   // from interface point of view an empty numeric value is also valid
   if((0 != numericValue.size()) && (numericValueExample.size() != numericValue.size()))
   {
      return false;
   }

   return true;
}

bool isValidUuid(IN const BTSUuid& uuid)
{
   // from interface point of view an empty UUID is also valid
   if((0 != uuid.size()) && (uuid128.size() != uuid.size()) && (uuid32.size() != uuid.size()) && (uuid16.size() != uuid.size()))
   {
      return false;
   }

   return true;
}

bool isValidDLinkKey(IN const BTSDLinkKey& dLinkKey)
{
   // from interface point of view an empty d link key is also valid
   if((0 != dLinkKey.size()) && (dLinkKeyExample.size() != dLinkKey.size()))
   {
      return false;
   }

   for(size_t i = 0; i < dLinkKey.size(); i++)
   {
      if(('0' <= dLinkKey[i]) && (dLinkKey[i] <= '9'))
      {
         // numeric character
      }
      else if(('a' <= dLinkKey[i]) && (dLinkKey[i] <= 'f'))
      {
         // alphabetic character
      }
      else if(('A' <= dLinkKey[i]) && (dLinkKey[i] <= 'F'))
      {
         // alphabetic character
      }
      else
      {
         // invalid character
         return false;
      }
   }

   return true;
}

bool isValidUserId(IN const BTSUserId& userId)
{
   if(userIdExample.size() < userId.size())
   {
      return false;
   }

   return true;
}

void convertNumericValueBinary2String(OUT BTSNumericValue& stringNumericValue, IN const unsigned int binaryNumericValue)
{
   // numeric value is a 6 digit value => max 999999
   // in case of just works the value is 0xFFFFFFFF => return empty string
   if(0xFFFFFFFF == binaryNumericValue)
   {
      return;
   }

   if(999999 < binaryNumericValue)
   {
      // invalid value
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   char buffer[1 + numericValueExample.size()];
   memset(buffer, 0, sizeof(buffer));
   snprintf(buffer, sizeof(buffer), numericValueFormat.c_str(), binaryNumericValue);
   stringNumericValue = buffer;
}

void convertNumericValueString2Binary(OUT unsigned int& binaryNumericValue, IN const BTSNumericValue& stringNumericValue)
{
   // max 999999
   if(numericValueExample.size() < stringNumericValue.size())
   {
      // invalid value
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   for(size_t i = 0; i < stringNumericValue.size(); i++)
   {
      if(('0' <= stringNumericValue[i]) && (stringNumericValue[i] <= '9'))
      {
         // valid character
      }
      else
      {
         // invalid character
         FW_NORMAL_ASSERT_ALWAYS();
         return;
      }
   }

   binaryNumericValue = (unsigned int)atol(stringNumericValue.c_str());
}

void convertLinkKeyBinary2String(OUT BTSLinkKey& stringLinkKey, IN const ::std::vector< unsigned char >& binaryLinkKey, IN const bool allowEmpty /*= false*/)
{
   convertBinary2HexStringValue(stringLinkKey, binaryLinkKey, linkKeyExample, allowEmpty);
}

void convertDLinkKeyBinary2String(OUT BTSDLinkKey& stringDLinkKey, IN const ::std::vector< unsigned char >& binaryDLinkKey, IN const bool allowEmpty /*= false*/)
{
   convertBinary2HexStringValue(stringDLinkKey, binaryDLinkKey, dLinkKeyExample, allowEmpty);
}

void convertLinkKeyString2Binary(OUT ::std::vector< unsigned char >& binaryLinkKey, IN const BTSLinkKey& stringLinkKey, IN const bool allowEmpty /*= false*/)
{
   convertHexString2BinaryValue(binaryLinkKey, stringLinkKey, linkKeyExample, allowEmpty);
}

void convertDLinkKeyString2Binary(OUT ::std::vector< unsigned char >& binaryDLinkKey, IN const BTSDLinkKey& stringDLinkKey, IN const bool allowEmpty /*= false*/)
{
   convertHexString2BinaryValue(binaryDLinkKey, stringDLinkKey, dLinkKeyExample, allowEmpty);
}

void convertUuidBinary2String(OUT BTSUuid& stringUuid, IN const ::std::vector< unsigned char >& binaryUuid)
{
   // uuid can have different sizes TODO: check for to-upper conversion
   if(((binaryUuid.size() << 1) == uuid128.size()) || ((binaryUuid.size() << 1) == uuid32.size()) || ((binaryUuid.size() << 1) == uuid16.size()))
   {
      char buffer[1 + uuid128.size()];
      memset(buffer, 0, sizeof(buffer));
      char hexString[3];
      for(size_t i = 0; i < binaryUuid.size(); i++)
      {
         convertBinary2HexString(hexString, binaryUuid[i]);
         buffer[2 * i    ] = hexString[0];
         buffer[2 * i + 1] = hexString[1];
      }
      stringUuid = buffer;
   }
   else
   {
      // wrong size
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void convertUuidString2Binary(OUT ::std::vector< unsigned char >& binaryUuid, IN const BTSUuid& stringUuid)
{
   // uuid can have different sizes
   if(stringUuid.size() == uuid128.size())
   {
      convertHexString2BinaryValue(binaryUuid, stringUuid, uuid128, false);
   }
   else if(stringUuid.size() == uuid32.size())
   {
      convertHexString2BinaryValue(binaryUuid, stringUuid, uuid32, false);
   }
   else if(stringUuid.size() == uuid16.size())
   {
      convertHexString2BinaryValue(binaryUuid, stringUuid, uuid16, false);
   }
   else
   {
      // wrong size
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void listCharacterDevices(IN const ::std::string& filter)
{
   DIR* dir;
   struct dirent* ent;
   char buf[512];

   if(NULL == (dir = opendir("/dev")))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   while(NULL != (ent = readdir(dir)))
   {
      if(DT_CHR == ent->d_type)
      {
         bool printDevice = true;

         if(filter.empty())
         {
            // no filter
         }
         else
         {
            // check beginning of the entry
            if(filter.size() <= strlen(ent->d_name))
            {
               strncpy(buf, ent->d_name, filter.size());
               buf[filter.size()] = '\0';

               if(0 != filter.compare(buf))
               {
                  // no match
                  printDevice = false;
               }
            }
            else
            {
               // length of filter is greater => no match possible
               printDevice = false;
            }
         }

         if(printDevice)
         {
            ETG_TRACE_USR1((" listCharacterDevices(): dev=%s", ent->d_name));
         }
      }
   }

   closedir(dir);
}

BTSDisconnectReason convertInternalDisconnectReason2Public(IN const BTSInternalDisconnectReason internalReason)
{
   BTSDisconnectReason reason(BTS_DISCONNECT_REASON_NOT_VALID);

   switch(internalReason)
   {
      case BTS_INT_DISCONNECT_REASON_LOCAL_DEVICE:
         reason = BTS_DISCONNECT_REASON_NORMAL_LOSS_LOCAL;
         break;
      case BTS_INT_DISCONNECT_REASON_REMOTE_DEVICE:
         reason = BTS_DISCONNECT_REASON_NORMAL_LOSS_REMOTE;
         break;
      case BTS_INT_DISCONNECT_REASON_EXCEPTION:
         // 20160815: do not use EXCEPTION as disconnect reason; instead use REMOTE_DEVICE
         // 20171006: handle EXCEPTION as LINK_LOSS
         reason = BTS_DISCONNECT_REASON_ABNORMAL_LOSS;
         break;
      case BTS_INT_DISCONNECT_REASON_LINK_LOSS:
         reason = BTS_DISCONNECT_REASON_ABNORMAL_LOSS;
         break;
      case BTS_INT_DISCONNECT_REASON_LAST:
      default:
         // should never happen
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   return reason;
}

BTSProtocolId convertSupportedService2Protocol(IN const BTSSupportedServicesBit supportedServicesBit)
{
   BTSProtocolId protocol(BTS_PROTO_LAST);

   switch(supportedServicesBit)
   {
      case BTS_SUPP_SRV_HFP:
         protocol = BTS_PROTO_HFP;
         break;
      case BTS_SUPP_SRV_A2DP:
      case BTS_SUPP_SRV_AVRCP:
         protocol = BTS_PROTO_AVP;
         break;
      case BTS_SUPP_SRV_PBAP:
         protocol = BTS_PROTO_PIM;
         break;
      case BTS_SUPP_SRV_MAP:
         protocol = BTS_PROTO_MSG;
         break;
      case BTS_SUPP_SRV_DUN:
         protocol = BTS_PROTO_DUN;
         break;
      case BTS_SUPP_SRV_PAN:
         protocol = BTS_PROTO_PAN;
         break;
      case BTS_SUPP_SRV_SPP:
         protocol = BTS_PROTO_SPP;
         break;
      case BTS_SUPP_SRV_PNP:
      case BTS_SUPP_SRV_DID:
      case BTS_SUPP_SRV_LAST:
      default:
         // should never happen
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   return protocol;
}

uint32_t calcMessageTraceId(IN const uint32_t component, IN const uint32_t stackInterface, IN const uint32_t opCode)
{
   // |31 .. 28|27 .. 24|23 .. 20|19 .. 16||15 .. 12|11 .. 8|7 .. 4|3 .. 0|
   // | <-           -> | <-  -> | <-                                  -> |
   // | component       |        | message opcode                         |
   // |                 | stack interface
#if 0
   const uint32_t traceId(((component & 0x000000FF) << 24) | ((stackInterface & 0x0000000F) << 20) | (opCode & 0x000FFFFF));
   ETG_TRACE_USR1((" calcMessageTraceId(): component=%u stackInterface=%u opCode=%u => traceId=0x%08X", component, stackInterface, opCode, traceId));
   return traceId;
#else
   return (((component & 0x000000FF) << 24) | ((stackInterface & 0x0000000F) << 20) | (opCode & 0x000FFFFF));
#endif
}

BTSFunctionBlock extractSubComponentFromBts2AppOpCode(IN const BTSBts2AppOpcode opCode)
{
   // current opcode definition (example):
   // Bts2AppOC_ConnectionBlockStart = (1 << 11) | (BTS_FB_CONNECTION << 12),
   return (BTSFunctionBlock)(((uint32_t)opCode) >> 12);
}

BTSFunctionBlock extractSubComponentFromApp2BtsOpCode(IN const BTSApp2BtsOpcode opCode)
{
   // current opcode definition (example):
   // App2BtsOC_ConnectionBlockStart = (BTS_FB_CONNECTION << 12),
   return (BTSFunctionBlock)(((uint32_t)opCode) >> 12);
}

} //btstackif
