/**
 * @file BtsTimerMaster.cpp
 *
 * @par SW-Component
 * Timer
 *
 * @brief Timer master.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Timer master handling.
 */

#include "BtsTimerMaster.h"
#include "FwLock.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BtsTimerMaster.cpp.trc.h"
#endif
#endif

namespace btstackif {

static LockForeverAndNonReentrant _lock;

TimerMaster::TimerMaster()
{
}

TimerMaster::~TimerMaster()
{
}

void TimerMaster::requestCurrentTime(void)
{
   if((_currentTimeSecondsList.size() > _context) && (_currentTimeNanoSecondsList.size() > _context))
   {
      getCurrentTime(_currentTimeSecondsList[_context], _currentTimeNanoSecondsList[_context]);

      if(true == _enableTrace4CurrentTime)
      {
         ETG_TRACE_USR1((" requestCurrentTime(): [context=%u]: currentTime=%d.%09d maxTime=%d.%09d", _context, _currentTimeSecondsList[_context], _currentTimeNanoSecondsList[_context], _maxTimeSeconds, _maxTimeNanoSeconds));
         // example output: currentTime=1147322.803665708 maxTime=2147483647.999999999 => as expected
      }
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void TimerMaster::checkElapsedTimer(void)
{
   if((_timerList.size() > _context) && (_currentTimeSecondsList.size() > _context) && (_currentTimeNanoSecondsList.size() > _context))
   {
      checkForTimeout(_timerList[_context], _context, _currentTimeSecondsList[_context], _currentTimeNanoSecondsList[_context]);
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void TimerMaster::setDefaultContext(void)
{
   _lock.lock();

   if(false == _defaultContextSet)
   {
      _defaultContextSet = true;
      ETG_TRACE_USR1((" setDefaultContext()"));
   }
   else
   {
      // only 1 timer master is allowed to use the default context
      FW_NORMAL_ASSERT_ALWAYS();
   }

   _lock.unlock();
}

void TimerMaster::createNewContext(void)
{
   _lock.lock();

   if(_maxNmbContexts > _usedNmbContexts)
   {
      _context = _usedNmbContexts++;
      ETG_TRACE_USR1((" createNewContext(): _context=%u _usedNmbContexts=%u", _context, _usedNmbContexts));
   }
   else
   {
      // check and increase _maxNmbContexts
      FW_NORMAL_ASSERT_ALWAYS();
   }

   _lock.unlock();
}

void TimerMaster::setCurrentTimeSeconds(IN const BTSMonotonicTimeSeconds seconds)
{
   if((0 <= seconds) && (_currentTimeSecondsList.size() > _context))
   {
      _currentTimeSecondsList[_context] = seconds;
   }
}

} //btstackif
