/**
 * @file BtsTimerBase.cpp
 *
 * @par SW-Component
 * Timer
 *
 * @brief Timer base.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Base class for timer handling.
 */

#include "BtsTimerBase.h"
#include "ITimeoutHandler.h"

#include <cstring>
#include <cerrno>

#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BtsTimerBase.cpp.trc.h"
#endif
#endif

namespace btstackif {

TimerItem::TimerItem()
{
   callback = NULL;
   timeoutHandler = NULL;
   id = 0;
   startSeconds = 0;
   startNanoSeconds = 0;
   timeoutSeconds = 0;
   timeoutNanoSeconds = 0;
   setTimeoutValue = 0;
   checkTimeoutValue = 0;
   cyclic = false;
   handler = NULL;
}

TimerItem::TimerItem(IN const TimerItem& ref)
{
   callback = NULL;
   setCallback(ref.callback);
   timeoutHandler = NULL;
   setTimeoutHandler(ref.timeoutHandler);
   id = ref.id;
   startSeconds = ref.startSeconds;
   startNanoSeconds = ref.startNanoSeconds;
   timeoutSeconds = ref.timeoutSeconds;
   timeoutNanoSeconds = ref.timeoutNanoSeconds;
   setTimeoutValue = ref.setTimeoutValue;
   checkTimeoutValue = ref.checkTimeoutValue;
   cyclic = ref.cyclic;
   handler = NULL;
   setTimerActiveHandler(ref.handler);
}

TimerItem& TimerItem::operator=(IN const TimerItem& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   callback = NULL;
   setCallback(ref.callback);
   timeoutHandler = NULL;
   setTimeoutHandler(ref.timeoutHandler);
   id = ref.id;
   startSeconds = ref.startSeconds;
   startNanoSeconds = ref.startNanoSeconds;
   timeoutSeconds = ref.timeoutSeconds;
   timeoutNanoSeconds = ref.timeoutNanoSeconds;
   setTimeoutValue = ref.setTimeoutValue;
   checkTimeoutValue = ref.checkTimeoutValue;
   cyclic = ref.cyclic;
   handler = NULL;
   setTimerActiveHandler(ref.handler);

   return *this;
}

bool TimerItem::operator==(IN const TimerItem& ref) const
{
   bool result = true;

   result = (true == result) && (callback == ref.callback);
   result = (true == result) && (timeoutHandler == ref.timeoutHandler);
   result = (true == result) && (id == ref.id);
   result = (true == result) && (startSeconds == ref.startSeconds);
   result = (true == result) && (startNanoSeconds == ref.startNanoSeconds);
   result = (true == result) && (timeoutSeconds == ref.timeoutSeconds);
   result = (true == result) && (timeoutNanoSeconds == ref.timeoutNanoSeconds);
   result = (true == result) && (setTimeoutValue == ref.setTimeoutValue);
   result = (true == result) && (checkTimeoutValue == ref.checkTimeoutValue);
   result = (true == result) && (cyclic == ref.cyclic);
   result = (true == result) && (handler == ref.handler);

   return result;
}

bool TimerItem::operator!=(IN const TimerItem& ref) const
{
   return !(operator==(ref));
}

TimerItem::~TimerItem()
{
   callback = NULL;
   timeoutHandler = NULL;
   handler = NULL;
}

//------------------------------------------------------------------------------

unsigned int TimerBase::_defaultContext = 0U;
unsigned int TimerBase::_maxNmbContexts = 5U;
unsigned int TimerBase::_usedNmbContexts = 1U; // first one is the default context
bool TimerBase::_initDone = false;
bool TimerBase::_defaultContextSet = false;
BTSMonotonicTimeSeconds TimerBase::_maxTimeSeconds = (BTSMonotonicTimeSeconds)((1UL << ((8 * sizeof(BTSMonotonicTimeSeconds)) - 1)) - 1);
BTSMonotonicTimeNanoSeconds TimerBase::_maxTimeNanoSeconds = 999999999L;
::std::vector< BTSMonotonicTimeSeconds > TimerBase::_currentTimeSecondsList;
::std::vector< BTSMonotonicTimeNanoSeconds > TimerBase::_currentTimeNanoSecondsList;
::std::vector< ::std::list< TimerItem > > TimerBase::_timerList;
bool TimerBase::_enableTrace4CurrentTime = false;
bool TimerBase::_enableTrace4TimerStatistic = false;
bool TimerBase::_enableTrace4TimoutCheck = false;
::std::vector< BTSTimerId > TimerBase::_nextTimerIdList;

TimerBase::TimerBase()
{
   _active = false;
   _context = _defaultContext;
}

TimerBase::~TimerBase()
{
}

void TimerBase::setActiveFlag(IN const bool enable)
{
   _active = enable;
}

void TimerBase::init(void)
{
   if(true == _initDone)
   {
      return;
   }

   _currentTimeSecondsList.reserve(_maxNmbContexts);
   _currentTimeNanoSecondsList.reserve(_maxNmbContexts);
   _timerList.reserve(_maxNmbContexts);
   _nextTimerIdList.reserve(_maxNmbContexts);

   ::std::list< TimerItem > emptyItem;
   for(size_t i = 0; i < _maxNmbContexts; i++)
   {
      _currentTimeSecondsList.push_back(0L);
      _currentTimeNanoSecondsList.push_back(0L);
      _timerList.push_back(emptyItem);
      _nextTimerIdList.push_back(1U);
   }

   _initDone = true;
}

void TimerBase::getCurrentTime(OUT BTSMonotonicTimeSeconds& timeSeconds, OUT BTSMonotonicTimeNanoSeconds& timeNanoSeconds)
{
   struct timespec currentTime;

   // get current time
   if(-1 == clock_gettime(CLOCK_MONOTONIC, &currentTime))
   {
      // check result for debugging purpose

      /*
       * from http://pubs.opengroup.org/onlinepubs/009695399/functions/clock_getres.html:
         The clock_getres(), clock_gettime(), and clock_settime() functions shall fail if:

         [EINVAL]
            The clock_id argument does not specify a known clock.
            => valid clock id is given
       *
       */

      int errNumber = errno;
      ETG_TRACE_ERR((" getCurrentTime(): clock_gettime failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
      FW_NORMAL_ASSERT_ALWAYS();
   }

   timeSeconds = currentTime.tv_sec;
   timeNanoSeconds = currentTime.tv_nsec;


   if(0 > timeNanoSeconds)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      timeNanoSeconds = 0;
   }
   if(0 > timeSeconds)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      timeSeconds = 0;
   }
   if(_maxTimeNanoSeconds < timeNanoSeconds)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      timeNanoSeconds = 0;
      increaseTimeSeconds(timeSeconds, 1);
   }
}

void TimerBase::checkForTimeout(INOUT std::list< TimerItem >& timerList, IN const unsigned int context, IN const BTSMonotonicTimeSeconds timeSeconds, IN const BTSMonotonicTimeNanoSeconds timeNanoSeconds)
{
   ::std::vector< TimerItem > callbackList;
   callbackList.reserve(timerList.size());

   // first step: collect all elapsed entries
   BTSTimeValue cmpDiff;
   ::std::list< TimerItem >::iterator it = timerList.begin();
   while(it != timerList.end())
   {
      cmpDiff = calcDiff(it->startSeconds, it->startNanoSeconds, timeSeconds, timeNanoSeconds, __LINE__);

      if(true == _enableTrace4TimoutCheck)
      {
         ETG_TRACE_USR1((" checkForTimeout(): [context=%u]: ID=%u Diff=%u", context, it->id, cmpDiff));
      }

      if(cmpDiff >= it->checkTimeoutValue)
      {
         // timer is elapsed
         callbackList.push_back(*it);
         it = timerList.erase(it);
      }
      else
      {
         // end now because we have an ordered list
         break;
      }
   }

   // second step: check for cyclic timer
   for(size_t i = 0; i < callbackList.size(); i++)
   {
      if(true == callbackList[i].cyclic)
      {
         (void)addTimer(context, callbackList[i].id, callbackList[i].setTimeoutValue, callbackList[i].callback, callbackList[i].timeoutHandler, callbackList[i].cyclic, false, true, callbackList[i].startSeconds, callbackList[i].startNanoSeconds);
      }
      else
      {
         if(NULL != callbackList[i].handler)
         {
            callbackList[i].handler->setActiveFlag(false);
         }
      }
   }

   // third step: call callback functions
   for(size_t i = 0; i < callbackList.size(); i++)
   {
      if(true == _enableTrace4TimoutCheck)
      {
         ETG_TRACE_USR1((" checkForTimeout(): [context=%u]: callbackList[%u]: callback=%p timeoutHandler=%p", context, i, (void*)callbackList[i].callback, (void*)callbackList[i].timeoutHandler));
      }

      if(NULL != callbackList[i].callback)
      {
         callbackList[i].callback();
      }
      else if(NULL != callbackList[i].timeoutHandler)
      {
         callbackList[i].timeoutHandler->handleTimeout(callbackList[i].id);
      }
   }

   callbackList.clear();
}

BTSTimerId TimerBase::addTimer(IN const unsigned int context, IN const BTSTimerId id, IN const BTSTimeValue timeout, IN BTSTimerCallBack callback, IN ITimeoutHandler* timeoutHandler, IN const bool cyclic, IN const bool active)
{
   return addTimer(context, id, timeout, callback, timeoutHandler, cyclic, active, false, 0, 0);
}

void TimerBase::removeTimer(IN const unsigned int context, IN const BTSTimerId id)
{
   if(_timerList.size() <= context)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   ::std::list< TimerItem >& timerList = _timerList[context];

   ::std::list< TimerItem >::iterator it = timerList.begin();
   while(it != timerList.end())
   {
      if(id == it->id)
      {
         // erase now
         (void)timerList.erase(it);
         break;
      }

      ++it;
   }

   if(true == _enableTrace4TimerStatistic)
   {
      printTimerStatistic(context);
   }
}

void TimerBase::printTimerStatistic(IN const unsigned int context) const
{
   if(_timerList.size() <= context)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   const ::std::list< TimerItem >& timerList = _timerList[context];

   unsigned int c = 0;
   for(::std::list< TimerItem >::const_iterator it = timerList.begin(); it != timerList.end(); ++it)
   {
      ETG_TRACE_USR1((" printTimerStatistic(): [context=%u]: [%02u]: ID=%u SetTimeoutValue=%u CheckTimeoutValue=%u TimeoutTime=%d.%09d", context, c, it->id, it->setTimeoutValue, it->checkTimeoutValue, it->timeoutSeconds, it->timeoutNanoSeconds));
      c++;
   }
   ETG_TRACE_USR1((" printTimerStatistic(): ___"));
}

BTSTimerId TimerBase::addTimer(IN const unsigned int context, IN const BTSTimerId id, IN const BTSTimeValue timeout, IN BTSTimerCallBack callback, IN ITimeoutHandler* timeoutHandler, IN const bool cyclic, IN const bool active, IN const bool restart, IN const BTSMonotonicTimeSeconds oldStartSeconds, IN const BTSMonotonicTimeNanoSeconds oldStartNanoSeconds)
{
   if((_timerList.size() <= context) || (_currentTimeSecondsList.size() <= context) || (_currentTimeNanoSecondsList.size() <= context) || (_nextTimerIdList.size() <= context))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return 0;
   }

   ::std::list< TimerItem >& timerList = _timerList[context];
   const BTSMonotonicTimeSeconds currentTimeSeconds = _currentTimeSecondsList[context];
   const BTSMonotonicTimeNanoSeconds currentTimeNanoSeconds = _currentTimeNanoSecondsList[context];
   BTSTimerId& nextTimerId = _nextTimerIdList[context];

   BTSTimerId returnId;
   TimerItem item;
   BTSTimeValue cmpDiff;
   BTSTimeValue newTimeout = timeout;
   BTSMonotonicTimeSeconds offsetTimeSeconds;
   BTSMonotonicTimeNanoSeconds offsetTimeNanoSeconds;

   // check for restart (cyclic timer)
   if(true == restart)
   {
      offsetTimeSeconds = newTimeout / 1000;
      offsetTimeNanoSeconds = (newTimeout % 1000) * 1000000; // max 999000000, less than max

      // calculate new start time based on previous start time
      item.startSeconds = oldStartSeconds;
      item.startNanoSeconds = oldStartNanoSeconds;
      addTimeOffset(item.startSeconds, item.startNanoSeconds, offsetTimeSeconds, offsetTimeNanoSeconds);

      // calculate elapsed time
      cmpDiff = calcDiff(item.startSeconds, item.startNanoSeconds, currentTimeSeconds, currentTimeNanoSeconds, __LINE__);

      // reduce elapsed time if current check timestamp is very late (this means timer would have been elapsed more than once)
      int c = 0;
      while(cmpDiff > newTimeout)
      {
         cmpDiff -= newTimeout;
         c++;
      }
      // now cmpDiff is in range of 0...newTimeout

      // correct start time
      for(int i = 1; i < c; i++)
      {
         addTimeOffset(item.startSeconds, item.startNanoSeconds, offsetTimeSeconds, offsetTimeNanoSeconds);
      }

      // calculate remaining time and set as new timeout
      newTimeout -= cmpDiff;
   }
   else
   {
      item.startSeconds = currentTimeSeconds;
      item.startNanoSeconds = currentTimeNanoSeconds;
   }

   offsetTimeSeconds = newTimeout / 1000;
   offsetTimeNanoSeconds = (newTimeout % 1000) * 1000000; // max 999000000, less than max
   item.timeoutSeconds = item.startSeconds;
   item.timeoutNanoSeconds = item.startNanoSeconds;
   addTimeOffset(item.timeoutSeconds, item.timeoutNanoSeconds, offsetTimeSeconds, offsetTimeNanoSeconds);

   // get next timer id (reuse timer id if possible)
   if(0 < id)
   {
      returnId = id;
   }
   else
   {
      returnId = nextTimerId++; // in case of timer are created and destroyed dynamically a timer pool for dynamic usage is needed; timer pool is implemented
   }
   item.id = returnId;

   // set callback
   item.callback = callback;
   item.timeoutHandler = timeoutHandler;

   // set timeout
   item.setTimeoutValue = timeout;
   item.checkTimeoutValue = newTimeout;

   // set cyclic flag
   item.cyclic = cyclic;

   // set handler
   item.handler = this;

   // in case of restarting timer we have to remove the existing item
   if(true == active)
   {
      ::std::list< TimerItem >::iterator it = timerList.begin();
      while(it != timerList.end())
      {
         if(id == it->id)
         {
            // erase now
            (void)timerList.erase(it);
            break;
         }

         ++it;
      }
   }

   // go through timer list and insert at right position
   bool inserted = false;

   ::std::list< TimerItem >::iterator it = timerList.begin();
   while(it != timerList.end())
   {
      // get elapsed time
      cmpDiff = calcDiff(it->startSeconds, it->startNanoSeconds, currentTimeSeconds, currentTimeNanoSeconds, __LINE__);

      if(cmpDiff >= it->checkTimeoutValue)
      {
         // timer is already elapsed => continue with next
      }
      else
      {
         // timer is not elapsed => check remaining time
         if(newTimeout < (it->checkTimeoutValue - cmpDiff))
         {
            // insert now
            timerList.insert(it, item);
            inserted = true;
            break;
         }
      }

      ++it;
   }

   if(false == inserted)
   {
      timerList.push_back(item);
   }

   if(true == _enableTrace4TimerStatistic)
   {
      printTimerStatistic(context);
   }

   return returnId;
}

void TimerBase::addTimeOffset(INOUT BTSMonotonicTimeSeconds& timeSeconds, INOUT BTSMonotonicTimeNanoSeconds& timeNanoSeconds, IN const BTSMonotonicTimeSeconds offSeconds, IN const BTSMonotonicTimeNanoSeconds offNanoSeconds)
{
   BTSMonotonicTimeSeconds offS = offSeconds;
   BTSMonotonicTimeNanoSeconds offNS = offNanoSeconds;

   // do some checks (make lint happy)
   if(0 > timeNanoSeconds)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      timeNanoSeconds = 0;
   }
   if(0 > timeSeconds)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      timeSeconds = 0;
   }
   if(0 > offNS)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      offNS = 0;
   }
   if(0 > offS)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      offS = 0;
   }
   if(_maxTimeNanoSeconds < timeNanoSeconds)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      timeNanoSeconds = 0;
      increaseTimeSeconds(timeSeconds, 1);
   }
   if(_maxTimeNanoSeconds < offNS)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      offNS = 0;
      if(_maxTimeSeconds > offS) // do not increase offset greater than max
      {
         offS++;
      }
   }

   // first check nano seconds
   BTSMonotonicTimeNanoSeconds remainingNanoSeconds = _maxTimeNanoSeconds - timeNanoSeconds; // 0 or greater
   if(offNS > remainingNanoSeconds)
   {
      timeNanoSeconds = (offNS - remainingNanoSeconds) - 1; // wrap around
      if(_maxTimeSeconds > offS) // do not increase offset greater than max
      {
         offS++;
      }
   }
   else
   {
      timeNanoSeconds += offNS;
   }

   // increase seconds
   increaseTimeSeconds(timeSeconds, offS);
}

void TimerBase::increaseTimeSeconds(INOUT BTSMonotonicTimeSeconds& timeSeconds, IN const BTSMonotonicTimeSeconds offSeconds)
{
   BTSMonotonicTimeSeconds remainingSeconds = _maxTimeSeconds - timeSeconds; // 0 or greater
   if(offSeconds > remainingSeconds)
   {
      timeSeconds = (offSeconds - remainingSeconds) - 1; // wrap around
   }
   else
   {
      timeSeconds += offSeconds;
   }
}

BTSTimeValue TimerBase::calcDiff(IN const BTSMonotonicTimeSeconds startSeconds, IN const BTSMonotonicTimeNanoSeconds startNanoSeconds, IN const BTSMonotonicTimeSeconds endSeconds, IN const BTSMonotonicTimeNanoSeconds endNanoSeconds, IN const unsigned int line) const
{
   unsigned long int diff;

   if(endSeconds >= startSeconds)
   {
      diff = 1000 * (endSeconds - startSeconds);

      if(endNanoSeconds >= startNanoSeconds)
      {
         diff += (endNanoSeconds - startNanoSeconds) / 1000000;
      }
      else
      {
         diff += (((_maxTimeNanoSeconds - startNanoSeconds) + endNanoSeconds) + 1) / 1000000;
         if(1000 <= diff)
         {
            diff -= 1000;
         }
      }
   }
   else
   {
      diff = 1000 * (((_maxTimeSeconds - startSeconds) + endSeconds) + 1);

      if(endNanoSeconds >= startNanoSeconds)
      {
         diff += (endNanoSeconds - startNanoSeconds) / 1000000;
      }
      else
      {
         diff += (((_maxTimeNanoSeconds - startNanoSeconds) + endNanoSeconds) + 1) / 1000000;
         if(1000 <= diff)
         {
            diff -= 1000;
         }
      }
   }

   // special check for unexpected time difference value (assume 2h = 7.200.000ms as maximum)
   if(diff > 7200000)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      ETG_TRACE_ERR((" calcDiff(): diff=%u start=%d.%09d end=%d.%09d (from line=%u)", diff, startSeconds, startNanoSeconds, endSeconds, endNanoSeconds, line));
   }

#if 0
   if(diff > 1000)
   {
      ETG_TRACE_USR1((" calcDiff(): diff=%u start=%d.%09d end=%d.%09d", diff, startSeconds, startNanoSeconds, endSeconds, endNanoSeconds));
   }
#endif

   return (BTSTimeValue)diff;
}

} //btstackif
